﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/ovln/ovln_NotificationAgent.h>

#include <nn/nn_Common.h>
#include "testOvln_Utility.h"

using namespace nnt::ovln;

// 実装
#include <nn/sf/sf_DefaultAllocationPolicy.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/ovln/ovln_Result.h>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <cstring>

namespace {

class Agent
{
private:

    nn::ovln::NotificationAgent* m_pAgent;

public:

    explicit Agent(nn::MemoryResource* pMemoryResource) NN_NOEXCEPT
        : m_pAgent(nn::ovln::CreateNotificationAgent(pMemoryResource))
    {
    }

    ~Agent() NN_NOEXCEPT
    {
        nn::ovln::DestroyNotificationAgent(m_pAgent);
    }

    nn::sf::SharedPointer<nn::ovln::IReceiverService> GetReceiverService() NN_NOEXCEPT
    {
        return nn::ovln::GetReceiverService(m_pAgent);
    }

    nn::sf::SharedPointer<nn::ovln::ISenderService> GetSenderService() NN_NOEXCEPT
    {
        return nn::ovln::GetSenderService(m_pAgent);
    }

};

typedef nn::sf::SharedPointer<nn::ovln::IReceiver> Receiver;
typedef nn::sf::SharedPointer<nn::ovln::ISender> Sender;

inline nn::ovln::RawMessage MakeTestRawMessage(int seed) NN_NOEXCEPT
{
    nn::ovln::RawMessage ret = {};
    ret.tag = seed;
    ret.dataSize = seed;
    std::memset(&ret.data, seed, ret.dataSize);
    return ret;
}

const nn::ovln::RawMessage RawMessage1 = MakeTestRawMessage(1);
const nn::ovln::RawMessage RawMessage2 = MakeTestRawMessage(2);
const nn::ovln::RawMessage RawMessage3 = MakeTestRawMessage(3);
const nn::ovln::RawMessage RawMessage4 = MakeTestRawMessage(4);
const nn::ovln::RawMessage RawMessage10 = MakeTestRawMessage(10);
const nn::ovln::RawMessage RawMessage100 = MakeTestRawMessage(100);

inline bool Equals(const nn::ovln::RawMessage& lhs, const nn::ovln::RawMessage& rhs) NN_NOEXCEPT
{
    return true
        && lhs.tag == rhs.tag
        && lhs.dataSize == rhs.dataSize
        && std::memcmp(&lhs.data, &rhs.data, lhs.dataSize) == 0;
}

#define NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, event, message) \
    do \
    { \
        ASSERT_TRUE((event).TryWait()); \
        ::nn::ovln::RawMessage receivedMessage; \
        NNT_OVLN_ASSERT_RESULT_SUCCESS((receiver)->Receive(&receivedMessage)); \
        ASSERT_TRUE(Equals((message), receivedMessage)); \
    } while (NN_STATIC_CONDITION(false))

#define NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, event) \
    do \
    { \
        ASSERT_TRUE(!(event).TryWait()); \
        ::nn::ovln::RawMessage receivedMessage; \
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultNoMessage, (receiver)->Receive(&receivedMessage)); \
    } while (NN_STATIC_CONDITION(false))

TEST(ovln_NotificationAgent, Initialize)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, OpenReceiver)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        Receiver receiver1;
        mr.SetAllocationEnabled(false);
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultMemoryAllocationFailed, receiverService->OpenReceiver(&receiver1));
        mr.SetAllocationEnabled(true);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver1));
        Receiver receiver2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver2));
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, ReceiverAddSource)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        mr.SetAllocationEnabled(false);
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultMemoryAllocationFailed, receiver->AddSource(SourceName1));
        mr.SetAllocationEnabled(true);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultSourceIsBusy, receiver->AddSource(SourceName1));
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->RemoveSource(SourceName1));
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, ReceiverMultiAddSource)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultSourceIsBusy, receiver->AddSource(SourceName1));
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->RemoveSource(SourceName1));
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, ReceiverAddSourceMulti)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        Receiver receiver1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver1));
        Receiver receiver2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver2));

        // receiver1 が SourceName1 を追加
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver1->AddSource(SourceName1));
        // receiver2 は SourceName1 を追加できない
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultSourceIsBusy, receiver2->AddSource(SourceName1));
        // receiver1 が SourceName1 を解放
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver1->RemoveSource(SourceName1));
        // receiver2 は SourceName1 を追加できる
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver2->AddSource(SourceName1));
        // receiver2 は SourceName1 を追加できない
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultSourceIsBusy, receiver1->AddSource(SourceName1));
        // receiver2 を解放 → SourceName1 も解放
        receiver2.Reset();
        // receiver1 は SourceName1 を追加できる
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver1->AddSource(SourceName1));
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

void AttachReceiverEvent(nn::os::SystemEvent* pEvent, Receiver receiver) NN_NOEXCEPT
{
    nn::sf::NativeHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(receiver->GetReceiveEventHandle(&handle));
    pEvent->AttachReadableHandle(handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_ManualClear);
    handle.Detach();
}

TEST(ovln_NotificationAgent, RecieverEvent)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, OpenSender)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto senderService = agent.GetSenderService();
        Sender sender1;
        mr.SetAllocationEnabled(false);
        NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultMemoryAllocationFailed, senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(1)));
        mr.SetAllocationEnabled(true);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(1)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName1, MakeQueueAttribute(2)));
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, Send)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto senderService = agent.GetSenderService();
        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(1)));
        NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionBackError));
        // メッセージ未受信で破棄してもリークしないことのチェック
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendAndReceive1)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(1)));

        {
            // メッセージがないため受信不可
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);

            // メッセージを send すると、受信可能
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionBackError));
            ASSERT_TRUE(e.TryWait());
            ASSERT_TRUE(e.TryWait());
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);

            // メッセージがもうないため、受信不可
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);

            // メッセージを残したまま sender を破棄すると、メッセージも破棄されて受信不可
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage2, SendOptionBackError));
            ASSERT_TRUE(e.TryWait());
            sender = nullptr;
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendAndReceive2)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        // NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1)); // ここでは追加しない

        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(1)));

        {
            // メッセージが全くないため受信不可
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);

            // メッセージを send しても、対象ソース未指定のため受信不可
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionBackError));
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);

            // 対象ソースを指定すると、受信可能
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));
            ASSERT_TRUE(e.TryWait());
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);

            // メッセージを send しても、対象ソースを削除すると受信不可
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage2, SendOptionBackError));
            ASSERT_TRUE(e.TryWait());
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->RemoveSource(SourceName1));
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendAndReceive3)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    const int CountMax = 10;
    for (auto count = 0; count <= CountMax; ++count)
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(count)));

        for (auto j = 0; j < 3; ++j) // キューが空になれば再度エンキューできることのため何度か繰り返す
        {
            // 複数を send
            for (auto i = 0; i < count; ++i)
            {
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(MakeTestRawMessage(i), SendOptionBackError));
            }
            // これ以上はキューがいっぱいで send できない
            NNT_OVLN_ASSERT_RESULT_FAILED(nn::ovln::ResultQueueIsFull, sender->Send(MakeTestRawMessage(count), SendOptionBackError));
            // send した順番に受け取れることをチェック
            for (auto i = 0; i < count; ++i)
            {
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, MakeTestRawMessage(i));
            }
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendMultiAndReceive1)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        const auto Count = 5;

        Sender sender1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(Count)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName1, MakeQueueAttribute(Count)));

        {
            // 複数の Sender から一つの Source へのテスト
            for (auto i = 0; i < Count; ++i)
            {
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(MakeTestRawMessage(i * 2 + 0), SendOptionBackError));
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(MakeTestRawMessage(i * 2 + 1), SendOptionBackError));
            }
            for (auto i = 0; i < Count * 2; ++i)
            {
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, MakeTestRawMessage(i));
            };
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendMultiAndReceive2)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));
        // NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName2)); // まだ 2 は追加しない

        const auto Count = 5;

        Sender sender1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(Count)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName2, MakeQueueAttribute(Count)));

        {
            // 複数の Source からのテスト
            for (auto i = 0; i < Count; ++i)
            {
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(MakeTestRawMessage(i * 2 + 0), SendOptionBackError));
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(MakeTestRawMessage(i * 2 + 1), SendOptionBackError));
            }
            for (auto i = 0; i < Count; ++i)
            {
                // この時点では SourceName1 からのもののみを受信
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, MakeTestRawMessage(i * 2 + 0));
            };
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);

            // SourceName2 を追加
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName2));
            for (auto i = 0; i < Count; ++i)
            {
                // 残っていた SourceName2 からのものを受信
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, MakeTestRawMessage(i * 2 + 1));
            };
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
        {
            // SourceName1 をはずす(SourceName2 のみ指定)
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->RemoveSource(SourceName1));

            // 1, 2, 1, 2 で追加
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionBackError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionBackError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionBackError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionBackError));

            // SourceName2 からのみ受信
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage2);
            ASSERT_TRUE(e.TryWait()); // まだ受信できる

            // SourceName1 を追加すると、1 から二つ受信した後に 2 から一つ受信する
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendMultiAndReceiveMulti)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver1));
        nn::os::SystemEvent e1;
        AttachReceiverEvent(&e1, receiver1);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver1->AddSource(SourceName1));

        Receiver receiver2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver2));
        nn::os::SystemEvent e2;
        AttachReceiverEvent(&e2, receiver2);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver2->AddSource(SourceName2));

        const auto Count = 5;

        Sender sender1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(Count)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName2, MakeQueueAttribute(Count)));

        {
            // 複数を send し順番に受け取れることのテスト
            for (auto i = 0; i < Count; ++i)
            {
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(MakeTestRawMessage(i * 2 + 0), SendOptionBackError));
                NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(MakeTestRawMessage(i * 2 + 1), SendOptionBackError));
            }
            for (auto i = 0; i < Count; ++i)
            {
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver1, e1, MakeTestRawMessage(i * 2 + 0));
                NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver2, e2, MakeTestRawMessage(i * 2 + 1));
            }
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver1, e1);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver2, e2);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendAndReceiveMulti)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver1));
        nn::os::SystemEvent e1;
        AttachReceiverEvent(&e1, receiver1);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver1->AddSource(SourceName1));

        Receiver receiver2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver2));
        nn::os::SystemEvent e2;
        AttachReceiverEvent(&e2, receiver2);
        // NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver2->AddSource(SourceName1));

        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(1)));

        {
            // SourceName1 に send
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionBackError));
            ASSERT_TRUE(e1.TryWait());

            // 受け取らずに receiver1 を破棄
            receiver1 = nullptr;

            // receiver2 に SourceName1 を追加して受信
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver2->AddSource(SourceName1));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver2, e2, RawMessage1);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendOption1)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender, SourceName1, MakeQueueAttribute(2)));

        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionFrontError));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage2, SendOptionFrontError));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage2);
        }
        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage1, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender->Send(RawMessage2, SendOptionFrontError));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage2);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendOption2)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        Sender sender1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(2)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName1, MakeQueueAttribute(2)));

        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontError));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage2);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontError));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontError));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage1);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage2);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_NotificationAgent, SendOption3)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    {
        Agent agent(&mr);
        auto receiverService = agent.GetReceiverService();
        auto senderService = agent.GetSenderService();

        Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiverService->OpenReceiver(&receiver));
        nn::os::SystemEvent e;
        AttachReceiverEvent(&e, receiver);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver->AddSource(SourceName1));

        Sender sender1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender1, SourceName1, MakeQueueAttribute(1)));
        Sender sender2;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(senderService->OpenSender(&sender2, SourceName1, MakeQueueAttribute(1)));

        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontRemoveFront));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
        {
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage1, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage2, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender2->Send(RawMessage4, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1->Send(RawMessage3, SendOptionFrontRemoveBack));
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage4);
            NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, e, RawMessage3);
            NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(receiver, e);
        }
    }
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

}
