﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

const size_t    HEAP_SIZE = 32 * 1024 * 1024;

namespace nn { namespace os {
uintptr_t   GetMemoryHeapAddress();
size_t      GetMemoryHeapSize();
}}

/*!
  @brief        メモリブロックの確保、アクセス、解放を行うデモ。

                メモリブロックを確保し、確保したメモリブロックのサイズと開始アドレスを表示します。
                確保したメモリブロックに対して、書き込みと読み込みを行います。

  @return       なし。
*/
void MemoryDemo1()
{
    NN_LOG("-- Memory Demo1 --\n");

    const int SIZE = 16 * 1024 * 1024;

    uintptr_t   address;
    nn::Result  result = nn::os::AllocateMemoryBlock( &address, SIZE );
    if ( !result.IsSuccess() )
    {
        NN_LOG("*** Failed to allocate memory.\n");
        return;
    }

    NN_LOG("* Allocate Size:%08x\n", address );
    NN_LOG("* Allocate Addr:%08x\n", SIZE );

    // メモリに値を書き込む
    int* p = reinterpret_cast<int*>( address );
    NN_LOG("* Write From   :%08x\n", p);
    for(int i = 0; i < 4096; i++)
    {
        *p = i;
        p++;
    }
    NN_LOG("*       To     :%08x\n", p);

    // メモリから値を読み込む
    p = reinterpret_cast<int*>( address );
    NN_LOG("* Read From    :%08x\n", p);
    for(int i = 0; i < 4096; i++)
    {
        NN_ASSERT(*p == i);
        p++;
    }
    NN_LOG("*      To      :%08x\n", p);

    // メモリブロックを解放
    nn::os::FreeMemoryBlock( address, SIZE );
}

/*!
  @brief        メモリブロックに関するデモです。
                メモリブロックのデモを実行します。

  @return       なし。
*/
void MemoryDemo()
{
    NN_LOG("*** Memory Demo starts. ***\n");

    // メモリヒープが小さければ大きくする
    size_t size = nn::os::GetMemoryHeapSize();
    if (size < HEAP_SIZE)
    {
        NN_ASSERT( nn::os::SetMemoryHeapSize( HEAP_SIZE ).IsSuccess(),
            "*** Cannot increase memory heap size." );
        NN_LOG("* Heap Address:%08x\n", nn::os::GetMemoryHeapAddress());
        NN_LOG("* Heap Size   :%08x\n", nn::os::GetMemoryHeapSize());
    }
    MemoryDemo1();

    NN_LOG("*** End of Memory Demo ***\n");
}

