﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "os_DemoUtil.h"

namespace DemoUtil {

namespace {
    // CreateThreads で使用するスレッドスタック
    NN_ALIGNAS(4096)    char s_ThreadStacks[OsMaxThreadNum][OsThreadStackSize];
}

/*!
  @brief        複数のスレッドを作成します。

                func をスレッド関数とするスレッドを num 個作成します。
                スレッドスタックのサイズ、作成できるスレッドの上限は固定です。

                CreateThreads で作成したスレッドが破棄される前に
                再度 CreateThreads を呼ぶと、
                以前のスレッドスタックを上書きしてしまうので注意が必要です。

  @param[in]    threads     スレッドオブジェクトの配列
  @param[in]    func        スレッド関数
  @param[in]    num         作成するスレッドの個数

  @return       なし。
*/
void CreateThreads(nn::os::ThreadType* pThreads, nn::os::ThreadFunction func, u32 num)
{
    NN_ASSERT(num <= OsMaxThreadNum);

    for(u32 i = 0; i < num; ++i)
    {
        auto* arg = reinterpret_cast<char*>(0) + i;
        nn::os::CreateThread( &pThreads[i], func, reinterpret_cast<void*>(arg), s_ThreadStacks[i], OsThreadStackSize, nn::os::DefaultThreadPriority - 1 );
        nn::os::StartThread( &pThreads[i] );
    }
}

/*!
  @brief        複数のスレッドの終了を待機します。

                num 個のスレッドの終了を待ちます。

  @param[in]    threads     スレッドオブジェクトの配列
  @param[in]    num         結合するスレッドの個数

  @return       なし。
*/
void WaitThreads(nn::os::ThreadType* pThreads, u32 num)
{
    NN_ASSERT(num <= OsMaxThreadNum);

    for(u32 i = 0; i < num; ++i)
    {
        nn::os::WaitThread( &pThreads[i] );
    }
}

void DestroyThreads(nn::os::ThreadType* pThreads, u32 num)
{
    NN_ASSERT(num <= OsMaxThreadNum);

    for(u32 i = 0; i < num; ++i)
    {
        nn::os::DestroyThread( &pThreads[i] );
    }
}


#define PRINT_COUNT(obj)                \
    NN_LOG("  %-10s %3d/%3d\n",         \
        #obj,                           \
        nn::os::obj::GetCurrentCount(), \
        nn::os::obj::GetMaxCount() )

void DumpObjectCounts()
{
    NN_LOG("----\n");
    NN_LOG("DumpObjectCounts\n");
#if 0   // 将来のためのコードを残しておく
    PRINT_COUNT(Event);
    PRINT_COUNT(Mutex);
    PRINT_COUNT(Semaphore);
    PRINT_COUNT(Thread);
    PRINT_COUNT(Timer);
#else
    NN_LOG("*** Cannot get object counts on Siglo.\n");
#endif
    NN_LOG("----\n");
}

}
