﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <thread>
#endif

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkThreadLocalStorage.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace tls {

namespace {
    int g_InitialUsedTlsSlot;
}

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 4096 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 4096 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread3[ 4096 ];

enum TestSyncPoint
{
    NNT_POINT_THREAD1_START             = 1,
    NNT_POINT_THREAD2_START,
    NNT_POINT_THREAD3_START,
    NNT_POINT_THREAD1_AFTER_SET_TLS,
    NNT_POINT_THREAD2_AFTER_SET_TLS,
    NNT_POINT_THREAD3_AFTER_SET_TLS,
    NNT_POINT_THREAD1_AFTER_GET_TLS,
    NNT_POINT_THREAD2_AFTER_GET_TLS,
    NNT_POINT_THREAD3_AFTER_GET_TLS,
};

//---------------------------------------------------------------------------
//  ThreadLocalStorage 関連機能のテスト
//---------------------------------------------------------------------------

//-[ アプリの静的コンストラクタから OS-API が呼べるか ]----------------------

class StaticClassForTlsTest
{
public:
    StaticClassForTlsTest();
    ~StaticClassForTlsTest();

    bool            GetPassed()
    {
        return m_passed;
    }
    nn::Result      GetResult()
    {
        return m_result;
    }
    nn::os::TlsSlot GetSlotNum()
    {
        return m_slotNum;
    }

private:
    bool            m_passed;   // コンストラクタ通過フラグ
    nn::Result      m_result;   // コンストラクタ内で発行した OS-API の返値
    nn::os::TlsSlot m_slotNum;  // 獲得した TLS スロット番号
};

// コンストラクタ
StaticClassForTlsTest::StaticClassForTlsTest()
{
    m_passed = true;

    m_result = nn::os::AllocateTlsSlot( &m_slotNum, NULL );

    NNT_OS_LOG(NN_TEXT("StaticClassForTlsTest: TLS を確保しました。\n"));

    // ここではまだ、GoogleTest-API を使えないので、
    // m_result 値を取っておいて後で検査する。
}

// デストラクタ
StaticClassForTlsTest::~StaticClassForTlsTest()
{
    nn::os::FreeTlsSlot( m_slotNum );
    NNT_OS_LOG(NN_TEXT("StaticClassForTlsTest: TLS を開放しました。\n"));
}

StaticClassForTlsTest g_staticClassForTlsTest;


TEST(CallOsapisFromStaticConstructor, test_CallAllocateTls)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    bool    passed = g_staticClassForTlsTest.GetPassed();

    NNT_OS_LOG(NN_TEXT("StaticClassForTlsTest クラスのコンストラクタは通過済みか？ ... "));
    NNT_OS_LOG(passed ? "true\n" : "false\n");
    EXPECT_TRUE( passed );

    NNT_OS_LOG(NN_TEXT("コンストラクタ内での AllocateTlsSlot() は成功か？ ... "));
    nn::Result   result = g_staticClassForTlsTest.GetResult();
    NNT_OS_LOG( result.IsSuccess() ? "Success\n" : "Fail\n");
    EXPECT_TRUE( result.IsSuccess() );

    if ( result.IsSuccess() )
    {
        nn::os::FreeTlsSlot( g_staticClassForTlsTest.GetSlotNum() );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------

nn::os::ThreadType* g_mainThread;
nn::os::ThreadType  g_Thread1;
nn::os::ThreadType  g_Thread2;
nn::os::ThreadType  g_Thread3;

const int   priThread1  = 8;
const int   priThread2  = 16;
const int   priThread3  = 24;

nn::os::TlsSlot g_TlsSlot0;
nn::os::TlsSlot g_TlsSlot1;
nn::os::TlsSlot g_TlsSlot2;
nn::os::TlsSlot g_TlsSlot3;


//---------------------------------------------------------------------------
//  TLS テスト用 TLS デストラクタ関数
//---------------------------------------------------------------------------

static int  g_TlsDestructor1Pass = 0;
static int  g_TlsDestructor3Pass = 0;

void    tlsDestructor1(uintptr_t tlsValue)
{
    NN_UNUSED( tlsValue );

    if (g_TlsDestructor1Pass == 0)
    {
        SEQ_CHECK(891);
        NNT_OS_LOG("%s is passed in Thread1", __FUNCTION__);
        CheckBool( true );
        g_TlsDestructor1Pass++;
    }
    else if (g_TlsDestructor1Pass == 1)
    {
        SEQ_CHECK(894);
        NNT_OS_LOG("%s is passed in Thread2", __FUNCTION__);
        CheckBool(true);

        SEQ_CHECK(895);
        NNT_OS_LOG("Free TLS Slot1: ");
        nn::os::FreeTlsSlot( g_TlsSlot1 );
        CheckBool( true );
        g_TlsDestructor1Pass++;
    }
    else
    {
        SEQ_ERROR();
    }
}


void    tlsDestructor3(uintptr_t tlsValue)
{
    NN_UNUSED( tlsValue );

    if (g_TlsDestructor3Pass == 0)
    {
        SEQ_CHECK(892);
        NNT_OS_LOG("%s is passed in Thread1", __FUNCTION__);
        CheckBool(true);
        g_TlsDestructor3Pass++;

        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_AFTER_GET_TLS );
    }
    else if (g_TlsDestructor3Pass == 1)
    {
        SEQ_CHECK(896);
        NNT_OS_LOG("%s is passed in Thread2", __FUNCTION__);
        CheckBool(true);

        SEQ_CHECK(897);
        NNT_OS_LOG("Free TLS Slot3: ");
        nn::os::FreeTlsSlot( g_TlsSlot3 );
        CheckBool( true );
        g_TlsDestructor3Pass++;

        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_AFTER_GET_TLS );
    }
    else
    {
        SEQ_ERROR();
    }
}


//---------------------------------------------------------------------------
//  TLS テスト用 Thread 関数
//---------------------------------------------------------------------------

void    ThreadFunc1(void* exinf)
{
    NN_UNUSED( exinf );

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    SEQ_CHECK(501);
    NNT_OS_LOG("Thread1:\n");

    // TLS を４つ設定
    {
        SEQ_CHECK(502);
        NNT_OS_LOG("Set TLS in Slot0 of Thread1: 0x00F10011 ");
        nn::os::SetTlsValue(g_TlsSlot0, 0x00F10011);
        CheckBool( true );

        SEQ_CHECK(503);
        NNT_OS_LOG("Set TLS in Slot1 of Thread1: 0x00F10022 ");
        nn::os::SetTlsValue(g_TlsSlot1, 0x00F10022);
        CheckBool( true );

        SEQ_CHECK(504);
        NNT_OS_LOG("Set TLS in Slot2 of Thread1: 0x00F10033 ");
        nn::os::SetTlsValue(g_TlsSlot2, 0x00F10033);
        CheckBool( true );

        SEQ_CHECK(505);
        NNT_OS_LOG("Set TLS in Slot3 of Thread1: 0x00F10044 ");
        nn::os::SetTlsValue(g_TlsSlot3, 0x00F10044);
        CheckBool( true );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_AFTER_SET_TLS );

    // 設定した４つの TLS の値の確認
    {
    uintptr_t   value;
    nn::Result  result;

        SEQ_SET(800);
        NNT_OS_LOG("Verify TLS in Slot0 of Thread1: ");
        value = nn::os::GetTlsValue(g_TlsSlot0);
        CheckBool( true );

        SEQ_CHECK(801);
        CheckParam( value, static_cast<uintptr_t>(0x00F10011) );

        SEQ_CHECK(802);
        NNT_OS_LOG("Verify TLS in Slot1 of Thread1: ");
        value = nn::os::GetTlsValue(g_TlsSlot1);
        CheckBool( true );

        SEQ_CHECK(803);
        CheckParam( value, static_cast<uintptr_t>(0x00F10022) );

        SEQ_CHECK(804);
        NNT_OS_LOG("Verify TLS in Slot2 of Thread1: ");
        value = nn::os::GetTlsValue(g_TlsSlot2);
        CheckBool( true );

        SEQ_CHECK(805);
        CheckParam( value, static_cast<uintptr_t>(0x00F10033) );

        SEQ_CHECK(806);
        NNT_OS_LOG("Verify TLS in Slot3 of Thread1: ");
        value = nn::os::GetTlsValue(g_TlsSlot3);
        CheckBool( true );

        SEQ_CHECK(807);
        CheckParam( value, static_cast<uintptr_t>(0x00F10044) );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_AFTER_GET_TLS );

    // スレッドの終了
    SEQ_SET(890);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);
}

void    ThreadFunc2(void* exinf)
{
    NN_UNUSED( exinf );

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    SEQ_CHECK(601);
    NNT_OS_LOG("Thread2:\n");

    // TLS を４つ設定
    {
        SEQ_CHECK(602);
        NNT_OS_LOG("Set TLS in Slot0 of Thread2: 0x00F20011 ");
        nn::os::SetTlsValue(g_TlsSlot0, 0x00F20011);
        CheckBool( true );

        SEQ_CHECK(603);
        NNT_OS_LOG("Set TLS in Slot1 of Thread2: 0x00F20022 ");
        nn::os::SetTlsValue(g_TlsSlot1, 0x00F20022);
        CheckBool( true );

        SEQ_CHECK(604);
        NNT_OS_LOG("Set TLS in Slot2 of Thread2: 0x00F20033 ");
        nn::os::SetTlsValue(g_TlsSlot2, 0x00F20033);
        CheckBool( true );

        SEQ_CHECK(605);
        NNT_OS_LOG("Set TLS in Slot3 of Thread2: 0x00F20044 ");
        nn::os::SetTlsValue(g_TlsSlot3, 0x00F20044);
        CheckBool( true );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_START );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_AFTER_SET_TLS );

    // 設定した４つの TLS の値の確認
    {
    uintptr_t   value;
    nn::Result  result;

        SEQ_SET(810);
        NNT_OS_LOG("Verify TLS in Slot0 of Thread2: ");
        value = nn::os::GetTlsValue(g_TlsSlot0);
        CheckBool( true );

        SEQ_CHECK(811);
        CheckParam( value, static_cast<uintptr_t>(0x00F20011) );

        SEQ_CHECK(812);
        NNT_OS_LOG("Verify TLS in Slot1 of Thread2: ");
        value = nn::os::GetTlsValue(g_TlsSlot1);
        CheckBool( true );

        SEQ_CHECK(813);
        CheckParam( value, static_cast<uintptr_t>(0x00F20022) );

        SEQ_CHECK(814);
        NNT_OS_LOG("Verify TLS in Slot2 of Thread2: ");
        value = nn::os::GetTlsValue(g_TlsSlot2);
        CheckBool( true );

        SEQ_CHECK(815);
        CheckParam( value, static_cast<uintptr_t>(0x00F20033) );

        SEQ_CHECK(816);
        NNT_OS_LOG("Verify TLS in Slot3 of Thread2: ");
        value = nn::os::GetTlsValue(g_TlsSlot3);
        CheckBool( true );

        SEQ_CHECK(817);
        CheckParam( value, static_cast<uintptr_t>(0x00F20044) );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD3_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_AFTER_GET_TLS );

    // スレッドの終了
    SEQ_CHECK(893);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);
}

void    ThreadFunc3(void* exinf)
{
    NN_UNUSED( exinf );

    // タイミング合わせ
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_START );

    SEQ_CHECK(701);
    NNT_OS_LOG("Thread3:\n");

    // TLS を４つ設定
    {
        SEQ_CHECK(702);
        NNT_OS_LOG("Set TLS in Slot0 of Thread3: 0x00F30011 ");
        nn::os::SetTlsValue(g_TlsSlot0, 0x00F30011);
        CheckBool( true );

        SEQ_CHECK(703);
        NNT_OS_LOG("Set TLS in Slot1 of Thread3: 0x00F30022 ");
        nn::os::SetTlsValue(g_TlsSlot1, 0x00F30022);
        CheckBool( true );

        SEQ_CHECK(704);
        NNT_OS_LOG("Set TLS in Slot2 of Thread3: 0x00F30033 ");
        nn::os::SetTlsValue(g_TlsSlot2, 0x00F30033);
        CheckBool( true );

        SEQ_CHECK(705);
        NNT_OS_LOG("Set TLS in Slot3 of Thread3: 0x00F30044 ");
        nn::os::SetTlsValue(g_TlsSlot3, 0x00F30044);
        CheckBool( true );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_AFTER_SET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_AFTER_SET_TLS );

    // 設定した４つの TLS の値の確認
    {
    uintptr_t   value;
    nn::Result  result;

        SEQ_SET(820);
        NNT_OS_LOG("Verify TLS in Slot0 of Thread3: ");
        value = nn::os::GetTlsValue(g_TlsSlot0);
        CheckBool( true );

        SEQ_CHECK(821);
        CheckParam( value, static_cast<uintptr_t>(0x00F30011) );

        SEQ_CHECK(822);
        NNT_OS_LOG("Verify TLS in Slot1 of Thread3: ");
        value = nn::os::GetTlsValue(g_TlsSlot1);
        CheckBool( true );

        SEQ_CHECK(823);
        CheckParam( value, static_cast<uintptr_t>(0x00F30022) );

        SEQ_CHECK(824);
        NNT_OS_LOG("Verify TLS in Slot2 of Thread3: ");
        value = nn::os::GetTlsValue(g_TlsSlot2);
        CheckBool( true );

        SEQ_CHECK(825);
        CheckParam( value, static_cast<uintptr_t>(0x00F30033) );

        SEQ_CHECK(826);
        NNT_OS_LOG("Verify TLS in Slot3 of Thread3: ");
        value = nn::os::GetTlsValue(g_TlsSlot3);
        CheckBool( true );

        SEQ_CHECK(827);
        CheckParam( value, static_cast<uintptr_t>(0x00F30044) );
    }

    // タイミング合わせ
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_AFTER_GET_TLS );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD3_AFTER_GET_TLS );

    // スレッドの終了
    SEQ_CHECK(898);
    NNT_OS_LOG("Just ExitThread in %s\n", __FUNCTION__);
}

//---------------------------------------------------------------------------
// テスト前の準備
TEST(TestPreparation1, test_Preparation)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // メインスレッドオブジェクトの検査
    nn::os::ThreadType* pThread = nn::os::GetCurrentThread();
    g_mainThread = pThread;

    SEQ_SET(100);
    NNT_OS_LOG("MainThreadObject is 0x%p", pThread);
    CheckBool( pThread != NULL );

    SEQ_CHECK(101);
    NNT_OS_LOG("MainThreadObject is not 0x%p", &g_Thread1);
    CheckBool( pThread != &g_Thread1 );

    SEQ_CHECK(102);
    NNT_OS_LOG("MainThreadObject is not 0x%p", &g_Thread2);
    CheckBool( pThread != &g_Thread2 );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// 全ての TLS スロットが未割当状態か確認
TEST(AllocateTlsSlot, test_BeforeAllocateTlsSlot)
{
    nn::Result  result;
    int         maxTlsSlot;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    SEQ_SET(200);
    maxTlsSlot = nn::os::TlsSlotCountMax;
    NNT_OS_LOG("GetMaxTlsSlot: %d", maxTlsSlot);
    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// TLS スロットの確保
TEST(AllocateTlsSlot, test_AllocateTlsSlot)
{
    nn::Result  result;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    SEQ_SET(300);
    g_InitialUsedTlsSlot = nn::os::GetUsedTlsSlotCount();
    NNT_OS_LOG("GetUsedTlsSlot: %d\n", g_InitialUsedTlsSlot);

    SEQ_CHECK(301);
    NNT_OS_LOG("Allocate: ");
    result = nn::os::AllocateTlsSlot(&g_TlsSlot0, NULL);
    NNT_OS_LOG("g_TlsSlot0=0x%p ", g_TlsSlot0);
    CheckBool( result.IsSuccess() );

    SEQ_CHECK(302);
    NNT_OS_LOG("Allocate: ");
    result = nn::os::AllocateTlsSlot(&g_TlsSlot1, tlsDestructor1);
    NNT_OS_LOG("g_TlsSlot1=0x%p ", g_TlsSlot1);
    CheckBool( result.IsSuccess() );

    SEQ_CHECK(303);
    NNT_OS_LOG("Allocate: ");
    result = nn::os::AllocateTlsSlot(&g_TlsSlot2, NULL);
    NNT_OS_LOG("g_TlsSlot2=0x%p ", g_TlsSlot2);
    CheckBool( result.IsSuccess() );

    SEQ_CHECK(304);
    NNT_OS_LOG("Allocate: ");
    result = nn::os::AllocateTlsSlot(&g_TlsSlot3, tlsDestructor3);
    NNT_OS_LOG("g_TlsSlot3=0x%p ", g_TlsSlot3);
    CheckBool( result.IsSuccess() );

    SEQ_CHECK(305);
    NNT_OS_LOG("GetUsedTlsSlot: ");
    auto usedTlsSlot = nn::os::GetUsedTlsSlotCount();
    CheckParam( usedTlsSlot, g_InitialUsedTlsSlot + 4 );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// スレッドの作成（ここで作成したスレッドを次の TEST で使う）
TEST(TestPreparation2, test_CreateThread)
{
    nn::Result  result;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // スレッド１の生成
    SEQ_SET(400);
    NNT_OS_LOG("Create Thread1:");
    result = nn::os::CreateThread(
                     &g_Thread1,                        // Thread オブジェクト
                     &ThreadFunc1,                      // エントリ関数
                     reinterpret_cast<void*>(0x5555),   // 引数
                     g_StackOfThread1,                  // スタック
                     sizeof(g_StackOfThread1),          // スタックサイズ
                     priThread1);                       // 優先度

    CheckBool( result.IsSuccess() );


    // スレッド２の生成
    SEQ_CHECK(401);
    NNT_OS_LOG("Create Thread2:");
    result = nn::os::CreateThread(
                     &g_Thread2,                        // Thread オブジェクト
                     &ThreadFunc2,                      // エントリ関数
                     reinterpret_cast<void*>(0xAAAA),   // 引数
                     g_StackOfThread2,                  // スタック
                     sizeof(g_StackOfThread2),          // スタックサイズ
                     priThread2);                       // 優先度

    CheckBool( result.IsSuccess() );


    // スレッド３の生成
    SEQ_CHECK(402);
    NNT_OS_LOG("Create Thread3:");
    result = nn::os::CreateThread(
                     &g_Thread3,                        // Thread オブジェクト
                     &ThreadFunc3,                      // エントリ関数
                     reinterpret_cast<void*>(0xBBBB),   // 引数
                     g_StackOfThread3,                  // スタック
                     sizeof(g_StackOfThread3),          // スタックサイズ
                     priThread3);                       // 優先度

    CheckBool( result.IsSuccess() );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// 各スレッドの動作開始
TEST(TestUsingTls, test_StartThread)
{
    nn::Result  result;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // シーケンス制御の初期化
    SEQ_SYNC_INIT();

    // スレッド１の動作開始 ---------------
    SEQ_SET(500);
    NNT_OS_LOG("Start Thread1:");
    nn::os::StartThread( &g_Thread1 );
    CheckBool( true );

    // スレッド２の動作開始 ---------------
    NNT_OS_LOG("Start Thread2:");
    nn::os::StartThread( &g_Thread2 );
    CheckBool( true );

    // スレッド３の動作開始 ---------------
    NNT_OS_LOG("Start Thread3:");
    nn::os::StartThread( &g_Thread3 );
    CheckBool( true );

    // テスト開始
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );

    // スレッドの終了待ち
    nn::os::WaitThread( &g_Thread1 );
    nn::os::WaitThread( &g_Thread2 );
    nn::os::WaitThread( &g_Thread3 );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// TLS 関連のエラーハンドリング
TEST(TestTlsErrorHandling, test_TlsErrorHandling)
{
    nn::Result  result;
    int         maxTlsSlot;
    int         usedTlsSlot;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    SEQ_SET(900);
    maxTlsSlot = nn::os::TlsSlotCountMax;
    NNT_OS_LOG("GetMaxTlsSlot: %d\n", maxTlsSlot);

    SEQ_CHECK(901);
    NNT_OS_LOG("GetUsedTlsSlot: ");
    usedTlsSlot = nn::os::GetUsedTlsSlotCount();
    CheckParam( usedTlsSlot, g_InitialUsedTlsSlot + 2 );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// スレッドの破棄
TEST(TestPostProcess, test_DestroyThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // スレッド１の破棄
    SEQ_SET(1000);
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &g_Thread1 );
    CheckBool( true );

    // スレッド２の破棄
    SEQ_CHECK(1001);
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &g_Thread2 );
    CheckBool( true );

    // スレッド３の破棄
    SEQ_CHECK(1002);
    NNT_OS_LOG("Destroy Thread3:");
    nn::os::DestroyThread( &g_Thread3 );
    CheckBool( true );

    // エージングのため初期状態に戻しておく（TLS の返却、Pass フラグの初期化）
    nn::os::FreeTlsSlot( g_TlsSlot0 );
    nn::os::FreeTlsSlot( g_TlsSlot2 );
    g_TlsDestructor1Pass = 0;
    g_TlsDestructor3Pass = 0;

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  TLS 確保後の TLS 値が正しくゼロ初期化されているかのテスト
//  一旦 TLS を確保して SetTlsValue() した後、一度 TLS を解放した後、
//  再度同じスロット番号の TLS を確保した時に、その TLS 値が 0 に初期化
//  されているかをテストする。

TEST(TestTlsInitializingAfterAllocated, test_IsTlsValueZeroAfterAllocated)
{
    nn::os::TlsSlot tlsSlot;
    nn::os::TlsSlot tlsSlotArray[ nn::os::TlsSlotCountMax ];

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // まず最初に TLS を１つ確保する
    SEQ_SET(1100);
    NNT_OS_LOG("Allocate TLS slot: ");
    nn::Result result = nn::os::AllocateTlsSlot(&tlsSlot, NULL);
    NNT_OS_LOG("tlsSlot=0x%p ", tlsSlot);
    CheckBool( result.IsSuccess() );

    // TLS に値を設定
    SEQ_NONE();
    nn::os::SetTlsValue( tlsSlot, 0x76543210 );
    uintptr_t value = nn::os::GetTlsValue( tlsSlot );
    NNT_OS_LOG("Set and Verify TLS value=0x%x ", value);
    CheckBool( value == 0x76543210 );

    // TLS を解放
    SEQ_NONE();
    NNT_OS_LOG("Free TLS slot: ");
    nn::os::FreeTlsSlot( tlsSlot );
    CheckBool( true );

    // 再度同じスロット番号の TLS が取れるまでループ
    SEQ_SET(1101);
    NNT_OS_LOG(NN_TEXT("Allocate TLS slot: 同じ TLS スロット番号を確保するまで繰返す\n"));
    int tlsCount = 0;
    for (;;)
    {
        nn::os::TlsSlot tmpTlsSlot;

        result = nn::os::AllocateTlsSlot(&tmpTlsSlot, NULL);
        ASSERT_TRUE( result.IsSuccess() );

        // 先程取得した TLS スロット番号が一致するならループを抜ける
        if (tlsSlot._innerValue == tmpTlsSlot._innerValue)
        {
            break;
        }

        tlsSlotArray[ tlsCount ] = tmpTlsSlot;
        ++tlsCount;
    }
    SEQ_NONE();
    NNT_OS_LOG("tlsSlot=0x%p\n", tlsSlot);

    // 獲得した TLS の TLS 値が 0 に初期化されているか確認
    SEQ_SET(1102);
    value = nn::os::GetTlsValue( tlsSlot );
    NNT_OS_LOG("Get TLS value=0x%x ", value);
    CheckBool( value == 0x0 );
    ASSERT_TRUE( 0x0 == value );

    // 全ての TLS を解放
    SEQ_NONE();
    NNT_OS_LOG("Free All TLS slots");
    for (int i=0; i<tlsCount; ++i)
    {
        nn::os::FreeTlsSlot( tlsSlotArray[ i ] );
    }
    CheckBool( true );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
// std::thread で使用するテスト用スレッド関数
//
void StdThreadTestFunction(nn::os::EventType* event)
{
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが起動しました。\n"));

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッド内で既存の TLS を操作\n"));
    nn::os::TlsSlot staticTls = g_staticClassForTlsTest.GetSlotNum();

    NNT_OS_LOG(NN_TEXT("sub:  TLS 値の初期値は 0 か"));
    uintptr_t value = nn::os::GetTlsValue( staticTls );
    CheckBool( value == 0 );

    NNT_OS_LOG(NN_TEXT("sub:  TLS に適当な値をセットする"));
    uintptr_t tickValue = static_cast<uintptr_t>( nn::os::GetSystemTick().GetInt64Value() );
    nn::os::SetTlsValue( staticTls, tickValue );
    CheckBool( true );

    NNT_OS_LOG(NN_TEXT("sub:  TLS にセットした値を取得できるか"));
    value = nn::os::GetTlsValue( staticTls );
    CheckBool( value == tickValue );

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが終了します。\n"));

    if (event != NULL)
    {
        nn::os::SignalEvent(event);
    }
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::join() する。
//
TEST(StdThreadTest, test_StdThreadAndJoin)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        std::thread stdThread( StdThreadTestFunction, nullptr );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドに join します。\n"));

        stdThread.join();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  以下の std::thread で生成したスレッドを detach するテストは、
//  TLS のテストでは行なわないこととする（Thread ではテスト実施済みである）。
//
//  本テスト以降に TLS デストラクタを登録するテストを行なうケースにおいて、
//  detach したスレッドに対しても TLS デストラクタが起動してしまうため、
//  テストのシーケンスを崩す要因になっている。
//
//  本質的に detach したスレッドの完了を待つ方法は存在せず、また、
//  そのようなスレッドに対して WaitThread() も利用してはならないため、
//  AllInOne なども考慮すると、後続にどんなテストが来るか分からない状況で
//  他のテストに副作用を与えないことを保証することが事実上不可能である。
//
#if 0
//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::detach() する。
//
TEST(StdThreadTest, test_StdThreadAndDetach)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        nn::os::EventType event;
        nn::os::InitializeEvent(&event, false, nn::os::EventClearMode_ManualClear);
        std::thread stdThread( StdThreadTestFunction, &event );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドを detach します。\n"));

        stdThread.detach();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));

        // 子スレッドが終了するまで待機するための処理
        nn::os::WaitEvent( &event );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}
#endif
#endif /* defined(NN_BUILD_CONFIG_OS_WIN32) */


//---------------------------------------------------------------------------
// ThreadLocalStorage クラスの動作確認
//
const uintptr_t g_TlsSetValue2OfSubThread = 0x89abcdef;

void ThreadLocalStorageClassTestDestructor(uintptr_t value)
{
    SEQ_CHECK(1404);
    NNT_OS_LOG(NN_TEXT("sub : TLS デストラクタの引数値（0x%08x）が正しいか"), g_TlsSetValue2OfSubThread);
    CheckBool( value == g_TlsSetValue2OfSubThread );
}

void ThreadLocalStorageClassTestSubThread(void* arg)
{
    auto& tls2 = *static_cast<nn::os::ThreadLocalStorage*>(arg);

    SEQ_CHECK(1402);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの起動\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 の初期値は 0 か"));
    CheckBool( tls2.GetValue() == 0);

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 に値をセット（0x%08x）\n"), g_TlsSetValue2OfSubThread);
    tls2.SetValue( g_TlsSetValue2OfSubThread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 の値は 0x%08x に更新されたか"), g_TlsSetValue2OfSubThread);
    CheckBool( tls2.GetValue() == g_TlsSetValue2OfSubThread );

    SEQ_CHECK(1403);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの終了\n"));
}

TEST(ThreadLocalStorageClass, test_ThreadLocalStorageClassTest)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(1400);
        NNT_OS_LOG(NN_TEXT("main: ThreadLocalStorage インスタンスの生成\n"));
        nn::os::ThreadLocalStorage tls1;
        nn::os::ThreadLocalStorage tls2(&ThreadLocalStorageClassTestDestructor);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 の初期値は 0 か"));
        CheckBool( tls1.GetValue() == 0);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 の初期値は 0 か"));
        CheckBool( tls2.GetValue() == 0);

        const uintptr_t tlsSetValue1 = 0x87654321;
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 に値をセット（0x%08x）\n"), tlsSetValue1);
        tls1.SetValue( tlsSetValue1 );

        const uintptr_t tlsSetValue2 = 0xFEDCBA98;
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 に値をセット（0x%08x）\n"), tlsSetValue2);
        tls2.SetValue( tlsSetValue2 );

        // 子スレッドの生成
        SEQ_CHECK(1401);
        NNT_OS_LOG(NN_TEXT("main: 子スレッドを生成"));

        nn::os::ThreadType  thread;
        auto result = nn::os::CreateThread(
                                        &thread,
                                        &ThreadLocalStorageClassTestSubThread,
                                        &tls2,
                                        g_StackOfThread2,
                                        sizeof(g_StackOfThread2),
                                        nn::os::DefaultThreadPriority);
        CheckBool( result.IsSuccess() );

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // 子スレッドの完了待ち
        nn::os::WaitThread( &thread );

        SEQ_CHECK(1405);
        NNT_OS_LOG(NN_TEXT("main: tls1 の値は 0x%08x のまま維持されているか"), tlsSetValue1);
        CheckBool( tls1.GetValue() == tlsSetValue1 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 の値は 0x%08x のまま維持されているか"), tlsSetValue2);
        CheckBool( tls2.GetValue() == tlsSetValue2 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 および tls2 のインスタンスの破棄\n"));

        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// SdkThreadLocalStorage クラスの動作確認
//
const uintptr_t g_SdkTlsSetValue2OfSubThread = 0x89abcdef;

void SdkThreadLocalStorageClassTestDestructor(uintptr_t value)
{
    SEQ_CHECK(1454);
    NNT_OS_LOG(NN_TEXT("sub : TLS デストラクタの引数値（0x%08x）が正しいか"), g_SdkTlsSetValue2OfSubThread);
    CheckBool( value == g_SdkTlsSetValue2OfSubThread );
}

void SdkThreadLocalStorageClassTestSubThread(void* arg)
{
    auto& tls2 = *static_cast<nn::os::SdkThreadLocalStorage*>(arg);

    SEQ_CHECK(1452);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの起動\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 の初期値は 0 か"));
    CheckBool( tls2.GetValue() == 0);

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 に値をセット（0x%08x）\n"), g_SdkTlsSetValue2OfSubThread);
    tls2.SetValue( g_SdkTlsSetValue2OfSubThread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : tls2 の値は 0x%08x に更新されたか"), g_SdkTlsSetValue2OfSubThread);
    CheckBool( tls2.GetValue() == g_SdkTlsSetValue2OfSubThread );

    SEQ_CHECK(1453);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの終了\n"));
}

TEST(SdkThreadLocalStorageClass, test_ThreadLocalStorageClassTest)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(1450);
        NNT_OS_LOG(NN_TEXT("main: SdkThreadLocalStorage インスタンスの生成\n"));
        nn::os::SdkThreadLocalStorage tls1;
        nn::os::SdkThreadLocalStorage tls2(&SdkThreadLocalStorageClassTestDestructor);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 の初期値は 0 か"));
        CheckBool( tls1.GetValue() == 0);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 の初期値は 0 か"));
        CheckBool( tls2.GetValue() == 0);

        const uintptr_t tlsSetValue1 = 0x87654321;
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 に値をセット（0x%08x）\n"), tlsSetValue1);
        tls1.SetValue( tlsSetValue1 );

        const uintptr_t tlsSetValue2 = 0xFEDCBA98;
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 に値をセット（0x%08x）\n"), tlsSetValue2);
        tls2.SetValue( tlsSetValue2 );

        // 子スレッドの生成
        SEQ_CHECK(1451);
        NNT_OS_LOG(NN_TEXT("main: 子スレッドを生成"));

        nn::os::ThreadType  thread;
        auto result = nn::os::CreateThread(
                                        &thread,
                                        &SdkThreadLocalStorageClassTestSubThread,
                                        &tls2,
                                        g_StackOfThread2,
                                        sizeof(g_StackOfThread2),
                                        nn::os::DefaultThreadPriority);
        CheckBool( result.IsSuccess() );

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // 子スレッドの完了待ち
        nn::os::WaitThread( &thread );

        SEQ_CHECK(1455);
        NNT_OS_LOG(NN_TEXT("main: tls1 の値は 0x%08x のまま維持されているか"), tlsSetValue1);
        CheckBool( tls1.GetValue() == tlsSetValue1 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls2 の値は 0x%08x のまま維持されているか"), tlsSetValue2);
        CheckBool( tls2.GetValue() == tlsSetValue2 );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: tls1 および tls2 のインスタンスの破棄\n"));

        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
//  std::thread のスレッド終了時に
//  nn::os::ThreadLocalStorage クラスのデストラクタ関数が呼ばれるか
//
bool g_ThreadLocalStorageClassTestDestructorForStdThreadIsCalled;

void ThreadLocalStorageClassTestDestructorForStdThread(uintptr_t value)
{
    SEQ_CHECK(1502);
    NNT_OS_LOG(NN_TEXT("sub : デストラクタ関数が呼ばれました。\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : デストラクタ関数の引数が正しいか"));

    CheckParam( value, static_cast<uintptr_t>(0xdeadbeaf) );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : デストラクタ関数を抜けます\n"));
    g_ThreadLocalStorageClassTestDestructorForStdThreadIsCalled = true;
}

void ThreadLocalStorageClassTestThreadForStdThread(nn::os::ThreadLocalStorage* tls)
{
    SEQ_CHECK(1501);
    NNT_OS_LOG(NN_TEXT("sub : TLS の初期値が 0 か"));
    CheckBool( tls->GetValue() == 0 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : TLS に 0xdeadbeaf を設定\n"));
    tls->SetValue( 0xdeadbeaf );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : スレッドを終了\n"));
}


TEST(ThreadLocalStorageClass, test_ThreadLocalStorageClassTestForStdThread)
{
    CLEAR_GOOGLE_TEST();

    g_ThreadLocalStorageClassTestDestructorForStdThreadIsCalled = false;

    {
        SEQ_SET(1500);
        NNT_OS_LOG(NN_TEXT("main: ThreadLocalStorage インスタンスの生成\n"));
        nn::os::ThreadLocalStorage  tls(&ThreadLocalStorageClassTestDestructorForStdThread);

        // スレッドを生成して終了時にデストラクタ関数が呼ばれるかをチェック
        std::thread thread(ThreadLocalStorageClassTestThreadForStdThread, &tls);
        thread.join();

        SEQ_CHECK(1503);
        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッド終了時に TLS デストラクタが呼ばれたか"));
        CheckBool(g_ThreadLocalStorageClassTestDestructorForStdThreadIsCalled);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: ThreadLocalStorage インスタンスの破棄\n"));
    }

    JUDGE_GOOGLE_TEST();
}
#endif // defined(NN_BUILD_CONFIG_OS_WIN32)

//---------------------------------------------------------------------------
//  引数でない TLS スロットの値を変更してしまうデストラクタが呼ばれた回数
//  nn::os::ThreadLocalStorage クラスのデストラクタ関数が呼ばれるか
//
int g_TlsDestructorToModifyTlsValueCalledCount;
nn::os::TlsSlot g_TlsSlotForDestructorTest0;
nn::os::TlsSlot g_TlsSlotForDestructorTest1;

const uintptr_t TlsTestValueA = 0xdeadbeef;
const uintptr_t TlsTestValueB = 0xbaddcafe;

void    tlsDestructorToModifyTlsValue(uintptr_t tlsValue)
{
    g_TlsDestructorToModifyTlsValueCalledCount++;

    // 引数の値から TLS スロットを特定し、もう一方の TLS スロットに値を入れ直す
    if (tlsValue == TlsTestValueB)
    {
        nn::os::SetTlsValue( g_TlsSlotForDestructorTest1, TlsTestValueA );
    }
    else if (tlsValue == TlsTestValueA)
    {
        nn::os::SetTlsValue( g_TlsSlotForDestructorTest0, TlsTestValueB );
    }
}

void TlsDestructorWithValueTestSubThread(void* arg)
{
    NN_UNUSED( arg );

    SEQ_CHECK(1602);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの起動\n"));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : TLS Slot0 に TlsTestValueB を設定\n"));
    nn::os::SetTlsValue( g_TlsSlotForDestructorTest0, TlsTestValueB );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("sub : TLS Slot1 に TlsTestValueA を設定\n"));
    nn::os::SetTlsValue( g_TlsSlotForDestructorTest1, TlsTestValueA );

    SEQ_CHECK(1603);
    NNT_OS_LOG(NN_TEXT("sub : 子スレッドの終了\n"));
}

TEST(TestTlsDestructorWithValue, test_TlsDestructorWithValue)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::Result result;

    g_TlsDestructorToModifyTlsValueCalledCount = 0;

    SEQ_SET(1600);
    NNT_OS_LOG(NN_TEXT("main: TLS スロットの確保\n"));

    result = nn::os::AllocateTlsSlot( &g_TlsSlotForDestructorTest0, tlsDestructorToModifyTlsValue );
    CheckBool( result.IsSuccess() );

    result = nn::os::AllocateTlsSlot( &g_TlsSlotForDestructorTest1, tlsDestructorToModifyTlsValue );
    CheckBool( result.IsSuccess() );

    // 子スレッドの生成
    SEQ_CHECK(1601);
    NNT_OS_LOG(NN_TEXT("main: 子スレッドを生成"));

    nn::os::ThreadType  thread;
    result = nn::os::CreateThread(
                                &thread,
                                &TlsDestructorWithValueTestSubThread,
                                NULL,
                                g_StackOfThread1,
                                sizeof(g_StackOfThread1),
                                nn::os::DefaultThreadPriority);
    CheckBool( result.IsSuccess() );

    // 小スレッドを起動
    nn::os::StartThread( &thread );

    // 小スレッドの完了待ち
    nn::os::WaitThread( &thread );

    // 小スレッドの削除
    nn::os::DestroyThread( &thread );

    // TLS デストラクタが 4回以上呼ばれていれば OK
    // (初回のループで2回呼ばれ、後のループでは2回以上呼ばれる想定)
    SEQ_CHECK(1604);
    NNT_OS_LOG(NN_TEXT("main: TLS スロットの値を変えてしまうデストラクタが複数回呼ばれたか\n"));
    CheckBool( g_TlsDestructorToModifyTlsValueCalledCount >= 4 );

    // TLS スロットを初期状態に戻す
    nn::os::FreeTlsSlot( g_TlsSlotForDestructorTest0 );
    nn::os::FreeTlsSlot( g_TlsSlotForDestructorTest1 );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

// GetTlsValue の速度を計測
#define NNT_OS_GTV(v,s)    (v = nn::os::GetTlsValue(s))
TEST(GetTlsValueSpeedTest, testGetTlsValueSpeed)
{
    const int loopCount = 10000;

    uintptr_t v;
    double sum = 0;
    double max = 0;
    double min = 10000;
    double leg;
    int64_t start;
    nn::os::TlsSlot slot;
    nn::Result result = nn::os::AllocateTlsSlot(&slot, NULL);

    CheckBool( result.IsSuccess() );

    // 適当な値をセット
    // 0xdeadbeef や 0xbaddcafe は他のテストで使用している意味のある値なので、
    // ここでは使用しない。
    nn::os::SetTlsValue( slot, 1 );

    // 時間計測
    for (int l = 0; l < loopCount; l++)
    {
        start = nn::os::GetSystemTick().GetInt64Value();

        auto& s = slot;
        NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s);
        NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s);
        NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s);
        NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s); NNT_OS_GTV(v,s);

        leg = (nn::os::GetSystemTick().GetInt64Value() - start) / 20.0;
        sum += leg;

        // 最悪値と最善値を記録
        if (leg > max)
        {
            max = leg;
        }
        if (leg < min)
        {
            min = leg;
        }
    }

    // 値を補正し、1000回あたりの回数にする
    sum -= max + min;
    sum = sum * 1000 / (loopCount - 2);

    // 結果の表示
    nn::os::Tick tick(static_cast<int64_t>(sum));
    NNT_OS_LOG("GetTlsValue : time=%lldus (tick=%lld) per %d calls.\n",
        tick.ToTimeSpan().GetMicroSeconds(), tick.GetInt64Value(), 1000);

    // TLS スロットを初期状態に戻す
    nn::os::FreeTlsSlot( slot );
}

TEST(SdkInternalTls, DoNotChangeUserTlsCount)
{
    nn::os::TlsSlot slot;

    int initialSlotCount = nn::os::GetUsedTlsSlotCount();
    NNT_OS_LOG("GetUsedTlsSlot: %d\n", initialSlotCount);

    // SDK 用の TLS を確保
    auto result = nn::os::SdkAllocateTlsSlot(&slot, NULL);
    ASSERT_TRUE(result.IsSuccess());

    // ユーザー用の TLS スロットが減っていないことを確認
    int slotCount = nn::os::GetUsedTlsSlotCount();
    NNT_OS_LOG("GetUsedTlsSlot: %d (After SdkAllocateTlsSlot)\n", slotCount);
    EXPECT_EQ(initialSlotCount, slotCount);

    // SDK 用の TLS を開放
    nn::os::FreeTlsSlot(slot);

    // ユーザー用の TLS スロットが増えていないことを確認
    slotCount = nn::os::GetUsedTlsSlotCount();
    NNT_OS_LOG("GetUsedTlsSlot: %d (After FreeTlsSlot)\n", slotCount);
    EXPECT_EQ(initialSlotCount, slotCount);
}

TEST(SdkInternalTls, SetGetValue)
{
    nn::os::TlsSlot slot;

    // SDK 用の TLS を確保
    auto result = nn::os::SdkAllocateTlsSlot(&slot, NULL);
    ASSERT_TRUE(result.IsSuccess());

    // TLS 値をセット
    nn::os::SetTlsValue(slot, 0xa5a5a5a5);

    // TLS 値をゲット
    uintptr_t value = nn::os::GetTlsValue(slot);
    EXPECT_EQ(0xa5a5a5a5, value);

    // SDK 用の TLS を開放
    nn::os::FreeTlsSlot(slot);
}

}}} // namespace nnt::os::tls

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of ThreadLocalStorage APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of ThreadLocalStorage APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
