﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include "../Common/test_Helper.h"
#include "../Common/test_NamedPipe.h"
#include "test_SystemEventAsInterProcessEventServer.h"


namespace nnt { namespace os { namespace systemEventAsInterProcessEvent {

// ポート名。NUL 文字含めて 12 文字まで
char    g_PortName[12];

NN_ALIGNAS(4096) char   g_ServerThreadStack[8192];
nn::os::ThreadType      g_ServerThread;
nn::os::SystemEventType g_ServerSync;

//-----------------------------------------------------------------------------
//  サーバ用スレッドの実装
//  本当は別プロセスで実行されるべきだが、API のテストとしては、
//  とりあえず、同一プロセス内で実行する。

void SystemEventAsInterProcessEventServerThread(void* arg)
{
    NN_UNUSED(arg);
    nnt::os::detail::NamedPipeServer serverPipe( g_PortName );

    NNT_OS_LOG("*** Start Server\n");

    nn::os::SignalSystemEvent(&g_ServerSync);

    // Pipe に接続
    serverPipe.Connect();

    for (;;)
    {
        // Pipe から受信
        auto value = serverPipe.RecvHandle();
        if (value == 0)
        {
            // 終了
            break;
        }

        // value を ReadableHandle に見立てて SystemEvent を初期化し、
        // 50msec 後に SystemEvent に対して signal する。
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(50) );
        {
            nn::os::SystemEventType   event;

            nn::os::AttachWritableHandleToSystemEvent(&event, nn::os::NativeHandle(value), false, nn::os::EventClearMode_ManualClear);

            nn::os::SignalSystemEvent(&event);

            nn::os::DestroySystemEvent(&event);
        }

        serverPipe.DestroyHandle(value);
    }

    NNT_OS_LOG("*** End Server\n");
}

//-----------------------------------------------------------------------------
//  サーバ用スレッドを起動する。
//  実装として、プロセス生成も可能であれば別プロセスとして生成する。
//
void StartTestServer()
{
    nn::os::CreateSystemEvent(&g_ServerSync, nn::os::EventClearMode_AutoClear, false);

    // テストで使用するポート名を自動生成
    nnt::os::detail::CreateUniquePortName( g_PortName , sizeof(g_PortName) );

    // サーバスレッドを生成
    nn::os::CreateThread(&g_ServerThread, SystemEventAsInterProcessEventServerThread, const_cast<char*>(g_PortName), g_ServerThreadStack, sizeof(g_ServerThreadStack), -8);

    // サーバを起動
    nn::os::StartThread(&g_ServerThread);

    nn::os::WaitSystemEvent(&g_ServerSync);
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(300) );
}

//-----------------------------------------------------------------------------
//  サーバ用スレッドを削除する。
//  実装として、プロセス生成も可能であれば別プロセスとして生成する。
//
void StopTestServer(nnt::os::detail::NamedPipeClient& clientPipe)
{
    // 終了通知
    clientPipe.SendHandle( detail::RequireEndOfTest, 0 );

    // サーバが終了するのを待って削除
    nn::os::WaitThread(&g_ServerThread);
    nn::os::DestroyThread(&g_ServerThread);

    nn::os::DestroySystemEvent(&g_ServerSync);
}

}}} // namespace nnt::os::systemEventAsInterProcessEvent

