﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_MemoryAccess.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <malloc.h>
#endif

namespace nn { namespace os {

uintptr_t   GetMemoryHeapAddress();
size_t      GetMemoryHeapSize();

}}  // namespace nn::os

namespace nnt { namespace os { namespace memoryPermission {

//---------------------------------------------------------------------------
//  メモリ関連機能のテスト
//---------------------------------------------------------------------------

inline size_t RoundUpToMemoryPageSize(size_t size)
{
    return (size + nn::os::MemoryPageSize - 1) & ~(nn::os::MemoryPageSize - 1);
}

#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
const uintptr_t g_writeData = 0x9abcdef0;            // メモリへの書込みデータ
#endif

namespace {

struct  AllocatedMemory
{
    uintptr_t   address;
    size_t      size;
    bool        allocated;
};

nn::Result  TestAllocateMemory( AllocatedMemory* mem, size_t size )
NN_NOEXCEPT
{
    uintptr_t   address;

    nn::Result  result = nn::os::AllocateMemoryBlock( &address, size );

    if ( result.IsSuccess() )
    {
        mem->address    = address;
        mem->size       = size;
    }
    mem->allocated  = result.IsSuccess() ? true : false;

    return result;
}

inline uintptr_t   TestGetMemoryAddress( AllocatedMemory* mem )
NN_NOEXCEPT
{
    return mem->address;
}

inline size_t   TestGetMemorySize( AllocatedMemory* mem )
NN_NOEXCEPT
{
    return mem->size;
}

// 指定された位置（access）へのメモリ読み書きを実施
void TestAccessMemoryPermission(uintptr_t access, bool readable, bool writable)
NN_NOEXCEPT
{
    if (readable && writable)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
        uint32_t saved_data = *reinterpret_cast<volatile uint32_t*>(access);
        NNT_MEMORY_BARRIER();
        // ここまで来たら Read に成功している
        *reinterpret_cast<volatile uint32_t*>(access) = g_writeData;
        NNT_MEMORY_BARRIER();
        // ここまで来たら Write に成功している
        *reinterpret_cast<volatile uint32_t*>(access) = saved_data;
#else
        if (!detail::MemoryAccessTest(access, detail::MemAttr_Normal_ReadWrite))
        {
            FAIL();
        }
#endif
    }
    else if (readable && !writable)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
        uint32_t saved_data = *reinterpret_cast<volatile uint32_t*>(access);
        NNT_MEMORY_BARRIER();
        NN_UNUSED(saved_data);
        // ここまで来たら Read に成功している
#else
        if (!detail::MemoryAccessTest(access, detail::MemAttr_Normal_ReadOnly))
        {
            FAIL();
        }
#endif


    }
    else if (!readable && !writable)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
        // Do nothing
#else
        if (!detail::MemoryAccessTest(access, detail::MemAttr_Normal_None))
        {
            FAIL();
        }
#endif
    }
    else
    {
        // WriteOnly は未サポート
        NN_ABORT("Not supported\n");
    }
}

}   // namespace


//---------------------------------------------------------------------------
// メモリのアクセスパーミションの変更テスト
//---------------------------------------------------------------------------
// svc::SetMemoryPermission() では svc::QueryMemory() で受け取ることができる
// カーネル管理のメモリ領域の単位でのみメモリアクセス権の変更が可能なため、
// OS ライブラリは svc::QueryMemory() と svc::SetMemoryPermission() を
// 繰り返して大きな領域のメモリアクセス権の変更を行なっている。
//
// このテストでは、R/O や R/W のメモリ領域をあえて分散させ、そのような
// メモリ領域に対して大きめの nn::os::SetMemoryPermission() を発行し、
// OS ライブラリの実装が正しいかをテストする。
//
//
//
//
//
//---------------------------------------------------------------------------

void TestMemoryAccessPermission(int seq, uintptr_t address, size_t size, bool readable, bool writable)
{
    SEQ_CHECK(seq);
    if (readable && writable)
    {
        NNT_OS_LOG("permission: ReadWrite ");
        nn::os::SetMemoryPermission( address, size, nn::os::MemoryPermission_ReadWrite );
    }
    else if (readable && !writable)
    {
        NNT_OS_LOG("permission: ReadOnly");
        nn::os::SetMemoryPermission( address, size, nn::os::MemoryPermission_ReadOnly );
    }
    else if (!readable && !writable)
    {
        NNT_OS_LOG("permission: None");
        nn::os::SetMemoryPermission( address, size, nn::os::MemoryPermission_None );
    }
    else
    {
        return;
    }

    NNT_OS_LOG("(adrs=0x%08x, size=0x%08x)\n", address, size);

    // 実際のメモリ読み書きをページ単位で実施
    uintptr_t access0 = address;
    uintptr_t access1 = (address + (size / 2));
    uintptr_t access2 = address + size - 4;

    TestAccessMemoryPermission(access0, readable, writable);
    TestAccessMemoryPermission(access1, readable, writable);
    TestAccessMemoryPermission(access2, readable, writable);
}


TEST(MemoryHeapAccessPermission, test_MemoryHeapAccessPermission)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    const int size = 32 * 1024 * 1024;

    AllocatedMemory     memBlock;
    nn::Result          result;
    uintptr_t           address;
    size_t              gotSize;

    int seq = 300;
    SEQ_SET( seq );
    NNT_OS_LOG("MemoryAccessPermission test\n");

    // 一度ヒープサイズを 0 に初期化
    // こうしておかないと、既存のメモリヒープの拡張に失敗することがある。
    // （後方のメモリが使用済みなどの場合）
    result = nn::os::SetMemoryHeapSize( 0 );
    ASSERT_TRUE( result.IsSuccess() );

    // その後に 32MB のヒープメモリを確保
    result = nn::os::SetMemoryHeapSize( size );
    SEQ_NONE();
    NNT_OS_LOG("SetMemoryHeapSize(32MB)");
    CheckBool( result.IsSuccess() );

    // GetMemoryHeapAddress() を発行
    address = nn::os::GetMemoryHeapAddress();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapAddress(): address= 0x%08x\n", address);

    // GetMemoryHeapSize() を発行
    gotSize = nn::os::GetMemoryHeapSize();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapSize():    size   = 0x%08x\n", gotSize);

    // AllocateMemoryBlock(32MB) を発行
    SEQ_NONE();
    NNT_OS_LOG("AllocateMemoryBlock(32MB)");
    result = TestAllocateMemory( &memBlock, size );
    CheckBool( result.IsSuccess() );
    if ( !result.IsSuccess() )
    {
        FAIL();
        return;
    }

    // 獲得したメモリブロックのアドレスとサイズの確認
    uintptr_t blockAddress = TestGetMemoryAddress( &memBlock );
    size_t    blockSize    = TestGetMemorySize( &memBlock );
    uintptr_t blockEnd     = blockAddress + blockSize;
    SEQ_CHECK( ++seq );
    NNT_OS_LOG("AllocatedMemoryBlock: Address=0x%p Size=0x%p\n", blockAddress, blockSize);
    if (gotSize != blockSize)
    {
        FAIL();
        return;
    }

    //-----------------------------------------------------------------------
    // 32MB のメモリヒープ全体に対して、
    // 適度な粒度でパーミション変更や Free/Allocate を繰り返す。
    // 具体的には、32MB のブロック全体を splitNum の数に等分割し、
    // それぞれに R/W と R/O を交互に設定する。
    // 分割した時のサイズは分割数によって変化する。
    bool writable = false;
    int  prevSplitNum = 0;
    int  splitNum     = 1;
    while (splitNum <= 610)
    {
        size_t strideSize = RoundUpToMemoryPageSize(blockSize / splitNum);

        for (auto adrs = blockAddress; adrs < blockEnd; adrs += strideSize)
        {
            uintptr_t endAdrs = adrs + strideSize;
            size_t    setSize = (endAdrs <= blockEnd) ? strideSize
                                                      : (blockEnd - adrs);

            TestMemoryAccessPermission(seq, adrs, setSize, false, false);
            TestMemoryAccessPermission(seq, adrs, setSize, true, writable);
            writable = writable ? false : true;
            ++seq;
        }

        // フィボナッチ数列を算出
        int nextSplitNum = prevSplitNum + splitNum;

        prevSplitNum = splitNum;
        splitNum     = nextSplitNum;
    }

    // メモリブロックを解放
    nn::os::FreeMemoryBlock( blockAddress, blockSize );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::memoryPermission

//---------------------------------------------------------------------------
//  Test Main 関数

//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of MemoryHeap APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of MemoryHeap APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
