﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_MemoryAccess.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <malloc.h>
#endif

namespace nn { namespace os {

uintptr_t   GetMemoryHeapAddress();
size_t      GetMemoryHeapSize();

}}  // namespace nn::os

namespace nnt { namespace os { namespace memoryHeap {

//---------------------------------------------------------------------------
//  確保可能な最大サイズのヒープを確保するテスト
//---------------------------------------------------------------------------

#if !defined(NN_BUILD_CONFIG_OS_WIN) && !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
// 以下の理由から、システムプロセスではこのテストは行なわない。
//
// - totalAvailableMemorySize で返る値がアプリプロセスとは意味が異なる
// - システムプロセスでは、そもそもヒープを最大限確保してはならない
//
TEST(TestSetMemoryHeapSize, test_AllocateMaxHeapSize)
{
    // 確保可能なヒープのサイズを取得
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);

    size_t size = static_cast<size_t>(memInfo.totalAvailableMemorySize) - memInfo.totalUsedMemorySize;
    size = nn::util::align_down(size , nn::os::MemoryBlockUnitSize);

    NNT_OS_LOG("totalAvailableMemorySize( %d MB )\n", memInfo.totalAvailableMemorySize / 1024 / 1024);
    NNT_OS_LOG("totalUsedMemorySize( %d MB )\n\n", memInfo.totalUsedMemorySize / 1024 / 1024);
    NNT_OS_LOG("SetMemoryHeapSize( %d MB )\n", size / 1024 / 1024);

    // メモリヒープの確保
    auto result = nn::os::SetMemoryHeapSize(size);
    ASSERT_TRUE(result.IsSuccess());

    NNT_OS_LOG("AllocateMemoryBlock( %d MB )\n", size / 1024 / 1024);

    // メモリブロックを確保
    uintptr_t p;
    result = nn::os::AllocateMemoryBlock(&p, size);
    ASSERT_TRUE(result.IsSuccess());

    // メモリブロックを解放
    nn::os::FreeMemoryBlock(p, size);

    // ヒープサイズを元に戻す
    result = nn::os::SetMemoryHeapSize(0);
    ASSERT_TRUE(result.IsSuccess());
}
#endif

//---------------------------------------------------------------------------
//  ヒープサイズを単純に増減させるテスト
//---------------------------------------------------------------------------

bool TestOfSetMemoryHeapSize(size_t size)
{
    auto result = nn::os::SetMemoryHeapSize( size );
    NNT_OS_LOG("SetMemoryHeapSize( %7d KB ): ", size / 1024);
    if (result.IsSuccess())
    {
        NNT_OS_LOG("OK\n");
        return true;
    }
    else
    {
        NNT_OS_LOG("NG\n");
        return false;
    }
}

TEST(TestSetMemoryHeapSize, test_TestSetMemoryHeapSize)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    const size_t minSize = 32 * 1024 * 1024;
    const size_t maxSize = 3ull * 1024 * 1024 * 1024 - 1;

    // ヒープサイズを MemoryHeapUnitSize から 4GB まで 32MB ずつ増やしていく。
    // 途中で確保できなくなれば、そこでサイズ減少テストへ移行。
    // ※ 4GB は Win64 のための上限。
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    void*   mallocAddress[256];
    int     mallocIndex = 0;
#endif

    size_t size = minSize;
    while (size <= maxSize)
    {
        bool result = TestOfSetMemoryHeapSize(size);
        if (!result)
        {
            break;
        }
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
        // ヒープサイズ変更後に malloc() でメモリを獲得する。
        // Win32 環境において、ヒープ領域の直後の空間が使用されることで、
        // 次ループ以降のヒープサイズ拡大を妨げる可能性をあえて作る。
        void* address = malloc( nn::os::MemoryHeapUnitSize );
        if (address != NULL)
        {
            mallocAddress[ mallocIndex++ ] = address;

            uintptr_t heapTop    = nn::os::GetMemoryHeapAddress();
            uintptr_t heapBottom = heapTop + size;
            NNT_OS_LOG("   heap  = 0x%p - 0x%p\n", heapTop, heapBottom);
            NNT_OS_LOG("   malloc= 0x%p\n", address);

            ASSERT_TRUE( (reinterpret_cast<uintptr_t>(address) < heapTop) ||
                         (reinterpret_cast<uintptr_t>(address) >= heapBottom) );
        }
#endif
        size += 32 * 1024 * 1024;
    }
    // 確保に成功した最大サイズ
    size -= 32 * 1024 * 1024;
    EXPECT_TRUE( size > 0 );

    // 各環境でヒープとして確保できるサイズは別のテストで検証する
#if 0
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // Win32 だけは 1GB 確保できたかをチェックする
    EXPECT_TRUE( size >= 1u * 1024 * 1024 * 1024 );
#elif (defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2))
    // Jetson-tk2 は 1GB - 32MB 確保できたかをチェックする
    // コードと初期スレッドスタックサイズ分減る
    EXPECT_TRUE( size >= 1u * 1024 * 1024 * 1024 - 32u * 1024 * 1024 );
#elif (defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_HARDWARE_NX))
    // NX でも 1GB - 32MB 確保できたかをチェックする
    // コードと初期スレッドスタックサイズ分減る
    EXPECT_TRUE( size >= 1u * 1024 * 1024 * 1024 - 32u * 1024 * 1024 );
#endif
#endif

    // ヒープサイズを最大獲得サイズから MemoryHeapUnitSize まで減らしていく。
    for (; size > minSize; size -= 32 * 1024 * 1024)
    {
        bool result = TestOfSetMemoryHeapSize(size);
        if (!result)
        {
            ADD_FAILURE();
        }
    }

    // ヒープサイズを 0 にしておく
    bool result = TestOfSetMemoryHeapSize(0);
    if (!result)
    {
        ADD_FAILURE();
    }

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    // malloc しておいたメモリを返却
    for (int i=0; i<mallocIndex; ++i)
    {
        free( mallocAddress[i] );
    }
#endif

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  メモリ関連機能のテスト
//---------------------------------------------------------------------------

inline size_t RoundUpToMemoryPageSize(size_t size)
{
    return (size + nn::os::MemoryPageSize - 1) & ~(nn::os::MemoryPageSize - 1);
}

const size_t    g_heapSize = 32 * 1024 * 1024;  // ヒープサイズ
const size_t    g_vSize    =  2 * 1024 * 1024;  // Allocate するサイズ

const uintptr_t g_data2 = 0x9abcdef0;           // メモリへの書込みデータ

namespace {

class AllocatedMemory
{
private:
    uintptr_t   m_Address;
    size_t      m_Size;
    bool        m_Allocated;

public:
    AllocatedMemory() NN_NOEXCEPT : m_Address(0), m_Size(0), m_Allocated(false)
    {
    }

    ~AllocatedMemory() NN_NOEXCEPT
    {
        Free();
    }

    nn::Result  Allocate(size_t size) NN_NOEXCEPT;
    void        Free() NN_NOEXCEPT;

    uintptr_t   GetAddress() const NN_NOEXCEPT
    {
        return m_Allocated ? m_Address : 0;
    }

    size_t      GetSize() const NN_NOEXCEPT
    {
        return m_Allocated ? m_Size : 0;
    }

    bool        IsAllocated() const NN_NOEXCEPT
    {
        return m_Allocated;
    }
};

AllocatedMemory g_memBlock[ 1024 ];

nn::Result AllocatedMemory::Allocate(size_t size ) NN_NOEXCEPT
{
    uintptr_t   address;
    nn::Result  result = nn::os::AllocateMemoryBlock( &address, size );

    if ( result.IsSuccess() )
    {
        m_Address   = address;
        m_Size      = size;
        m_Allocated = true;
    }

    return result;
}

void AllocatedMemory::Free() NN_NOEXCEPT
{
    if (m_Allocated)
    {
        nn::os::FreeMemoryBlock( m_Address, m_Size );
        m_Allocated = false;
    }
}


// 指定された位置（access）へのメモリ読み書きを実施
void TestAccessMemory(int seq, volatile uint32_t* access, bool writable, bool realWrite)
NN_NOEXCEPT
{
    SEQ_SET( seq + 0 );
    NNT_OS_LOG("Memory Read Write Test at 0x%p\n", access);

    // Read 確認
    NNT_OS_LOG("(-----): Read Access ");
#if defined(NN_BUILD_CONFIG_OS_WIN32) || !defined(NNT_OS_SYSTEM_PROGRAM_TEST)
    uint32_t saved_data = *access;
    NN_UNUSED(saved_data);
    NNT_MEMORY_BARRIER();
    // ここまで来たら Read に成功している
    CheckBool( true );
#else
    bool result = detail::MemoryAccessTest(
                                reinterpret_cast<uintptr_t>(access),
                                (writable ? detail::MemAttr_Normal_ReadWrite
                                          : detail::MemAttr_Normal_ReadOnly) );
    CheckBool( result == true );
#endif

    // Write 確認
    NNT_OS_LOG("(-----): Write Access");
    if ((writable == true) || (realWrite == true))
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        *access = g_data2;
        NNT_MEMORY_BARRIER();
        // ここまで来たら Write に成功している
        CheckBool( true );
        *access = saved_data;
#else
        NN_UNUSED(g_data2);
        if (!writable)
        {
            NNT_OS_LOG(" is not permitted");
        }
#if defined(NNT_OS_SYSTEM_PROGRAM_TEST)
        result = detail::MemoryAccessTest(
                                reinterpret_cast<uintptr_t>(access),
                                (writable ? detail::MemAttr_Normal_ReadWrite
                                          : detail::MemAttr_Normal_ReadOnly) );
        CheckBool( result == true );
#endif // NNT_OS_SYSTEM_PROGRAM_TEST
#endif
    }
    else
    {
        NNT_OS_LOG(" ... Skiped\n");
    }
}

}   // namespace


//---------------------------------------------------------------------------
// Allocate 系テスト (Read-Write)
//---------------------------------------------------------------------------
// nn::os::AllocateMemoryBlock() で獲得したメモリ領域に対して、
// 領域の先頭と末尾の２か所に対してメモリリード／ライトを行ない、
// 実際のメモリアクセスの可否を確認する。
//
// writable=false の場合は、メモリブロックを ReadOnly に変えてアクセスする。
//
//                        ↓address                           ↓address+size
//   Memory Space: -------+===================================+--------
//                         ^       <commited region>         ^
//     Test Point:         1                                 2
//---------------------------------------------------------------------------
void   doTestAllocate(int seq, size_t size, int offset, bool readable, bool writable)
{
    AllocatedMemory     memBlock;
    nn::Result          result;
    uintptr_t           address;
    size_t              gotSize;
    volatile uint32_t*  access;

    NN_UNUSED( readable );

    // 一度ヒープサイズを 0 に初期化
    // こうしておかないと、既存のメモリヒープの拡張に失敗することがある。
    // （後方のメモリが使用済みなどの場合）
    result = nn::os::SetMemoryHeapSize( 0 );
    ASSERT_TRUE( result.IsSuccess() );

    // SetMemoryHeapSize() を発行
    result = nn::os::SetMemoryHeapSize( g_heapSize );
    SEQ_NONE();
    NNT_OS_LOG("SetMemoryHeapSize(%d KB): result=", g_heapSize / 1024);
    NNT_OS_LOG( result.IsSuccess() ? "true\n" : "false\n" );
    ASSERT_TRUE( result.IsSuccess() );

    // GetMemoryHeapAddress() を発行
    address = nn::os::GetMemoryHeapAddress();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapAddress(): address=0x%08x\n", address);

    // GetMemoryHeapSize() を発行
    gotSize = nn::os::GetMemoryHeapSize();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapSize():    size   =0x%08x\n", gotSize);

    // AllocateMemoryBlock() を発行
    SEQ_SET( seq );
    NNT_OS_LOG("AllocateMemoryBlock(%d KB): result=", size / 1024);
    result = memBlock.Allocate( size );
    NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
    CheckBool( result.IsSuccess() );
    if ( !result.IsSuccess() )
    {
        FAIL();
        return;
    }

    // アドレスとサイズの取得
    SEQ_SET( seq + 1 );
    address = memBlock.GetAddress();
    gotSize = memBlock.GetSize();
    NNT_OS_LOG("TestGetMemoryInfo() Address=0x%p Size=0x%p\n", address, gotSize);
    NNT_OS_LOG("(-----): Size:");
    CheckParam( size, gotSize );


    // writable==false なら ReadOnly に変更してアクセス確認
    if (writable == false)
    {
        SEQ_SET( seq + 2 );
        NNT_OS_LOG("SetMemoryPermission( ReadOnly ): ");
        nn::os::SetMemoryPermission( address, gotSize, nn::os::MemoryPermission_ReadOnly );
        NNT_OS_LOG("OK\n");
    }

    // テストとしてアクセスすべきアドレス値を計算
    access = reinterpret_cast<volatile uint32_t*>( address + offset );

    // 指定された位置（address + size + offset）へのメモリ読み書きを実施
    TestAccessMemory(seq + 3, access, writable, true);

    // ここで一旦 SetMemoryPermission() でメモリアクセス権を変更する。
    // 元が writable == true なら false へ、flase なら true な属性へ変更。
    {
        SEQ_SET( seq + 4 );
        NNT_OS_LOG("SetMemoryPermission( %s ): ",
                (writable == true) ? "ReadOnly" : "ReadWrite" );
        nn::os::SetMemoryPermission( address, gotSize,
                (writable == true) ? nn::os::MemoryPermission_ReadOnly :
                                     nn::os::MemoryPermission_ReadWrite );
        NNT_OS_LOG("OK\n");

        // メモリアクセステスト
        TestAccessMemory(seq + 5, access, !writable, false);
    }


    // 再度 SetMemoryPermission() でメモリアクセス権を元に戻す
    {
        SEQ_SET( seq + 6 );
        NNT_OS_LOG("SetMemoryPermission( %s ): ",
                (writable == true) ? "ReadWrite" : "ReadOnly" );
        nn::os::SetMemoryPermission( address, gotSize,
                (writable == true) ? nn::os::MemoryPermission_ReadWrite :
                                     nn::os::MemoryPermission_ReadOnly );
        NNT_OS_LOG("OK\n");

        // メモリアクセステスト
        TestAccessMemory(seq + 7, access, writable, true);
    }

    // FreeMemoryBlock() を発行
    memBlock.Free();

    // ヒープサイズを一旦 0 に戻す
    SEQ_NONE();
    NNT_OS_LOG("SetMemoryHeapSize(0 KB): result=");
    result = nn::os::SetMemoryHeapSize( 0 );
    NNT_OS_LOG( result.IsSuccess() ? "true\n" : "false\n" );
    ASSERT_TRUE( result.IsSuccess() );

    // うまく読み書き出来たらエラーなし終了
    SUCCEED();
}

void   doTestAllocateWrapper(int seq, size_t size, int offset, bool readable, bool writable)
{
    if ((readable == false) || (writable == false))
    {
        // テストがメモリアクセス違反を起こして異常終了することが期待
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        EXPECT_DEATH_IF_SUPPORTED(doTestAllocate(seq, size, offset, readable, writable), "");
#else
        doTestAllocate(seq, size, offset, readable, writable);
#endif
        NNT_OS_LOG("Detect Memory Access Violatation\n");
    }
    else
    {
        // テストはメモリアクセス違反を起こさずに正常終了することが期待
        doTestAllocate(seq, size, offset, readable, writable);
    }
}

//---------------------------------------------------------------------------

TEST(AllocateMemoryBlockDeathTest, test_AllocateMemoryBlock1)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 個々のメモリアクセスを DEATH テストとして実施する（ReadWrite）
    doTestAllocateWrapper(110, g_vSize, 0,         true, true);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

TEST(AllocateMemoryBlockDeathTest, test_AllocateMemoryBlock2)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 個々のメモリアクセスを DEATH テストとして実施する（ReadWrite）
    doTestAllocateWrapper(120, g_vSize, g_vSize - 4, true, true);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

TEST(AllocateMemoryBlockDeathTest, test_AllocateMemoryBlock3)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 個々のメモリアクセスを DEATH テストとして実施する（ReadOnly）
    doTestAllocateWrapper(130, g_vSize, 0,         true, false);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

TEST(AllocateMemoryBlockDeathTest, test_AllocateMemoryBlock4)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 個々のメモリアクセスを DEATH テストとして実施する（ReadOnly）
    doTestAllocateWrapper(140, g_vSize, g_vSize - 4, true, false);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// Free 系テスト
//---------------------------------------------------------------------------
// 一度連続領域として Allocate で獲得したメモリ領域に対して、
// その全領域を FreeMemoryBlock() で物理メモリを返却する。
// 領域の先頭と末尾の２か所に対してメモリリードを行ない、
// 実際のメモリアクセスの可否を確認する。
//
//                        ↓address                            ↓address+size
//   Memory Space: -------+====================================+--------
//                         ^ <- Allocated and Freed region -> ^
//     Test Point:         1                                  2
//---------------------------------------------------------------------------
void   doTestFree(int seq, size_t size)
{
    AllocatedMemory     memBlock;
    nn::Result          result;
    uintptr_t           address;
    size_t              gotSize;
    volatile uint32_t   *access;

    // 一度ヒープサイズを 0 に初期化
    // こうしておかないと、既存のメモリヒープの拡張に失敗することがある。
    // （後方のメモリが使用済みなどの場合）
    nn::os::SetMemoryHeapSize( 0 );

    // SetMemoryHeapSize() を発行
    result = nn::os::SetMemoryHeapSize( g_heapSize );
    SEQ_NONE();
    NNT_OS_LOG("SetMemoryHeapSize(): result=");
    NNT_OS_LOG( result.IsSuccess() ? "true\n" : "false\n" );
    ASSERT_TRUE( result.IsSuccess() );

    // GetMemoryHeapAddress() を発行
    address = nn::os::GetMemoryHeapAddress();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapAddress(): address=0x%08x\n", address);

    // GetMemoryHeapSize() を発行
    gotSize = nn::os::GetMemoryHeapSize();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapSize():    size   =0x%08x\n", gotSize);

    // AllocateMemoryBlock() を発行
    SEQ_SET( seq );
    NNT_OS_LOG("AllocateMemoryBlock(): result=");
    result = memBlock.Allocate( size );
    NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
    CheckBool( result.IsSuccess() );
    if ( !result.IsSuccess() )
    {
        FAIL();
        return;
    }

    // アドレスとサイズの取得
    SEQ_SET( seq + 1 );
    address = memBlock.GetAddress();
    gotSize = memBlock.GetSize();
    NNT_OS_LOG("TestGetMemoryInfo() Address=0x%p Size=0x%p\n", address, gotSize);
    NNT_OS_LOG("(-----): Size:");
    CheckParam( size, gotSize );

    // テストとしてアクセスすべきアドレス値を計算
    access = reinterpret_cast<volatile uint32_t*>( address );

    // 獲得したばかりのメモリブロックの R/W を検査する
    TestAccessMemory(seq + 2, access, true, true);

    // 獲得したメモリブロック領域を R/O に変更する
    SEQ_SET( seq + 3 );
    NNT_OS_LOG("SetMemoryPermission( ReadOnly ): ");
    nn::os::SetMemoryPermission(address, size, nn::os::MemoryPermission_ReadOnly);
    NNT_OS_LOG("OK\n");

    // R/O に変更したメモリブロックのアドレスの R/W を検査する
    TestAccessMemory(seq + 4, access, false, false);

    // R/O のまま FreeMemoryBlock() を発行
    SEQ_SET( seq + 5 );
    NNT_OS_LOG("FreeMemoryBlock():");
    memBlock.Free();
    CheckBool( true );

    // メモリブロックを返却し終わったアドレスの R/W を検査する
    TestAccessMemory(seq + 6, access, true, true);

    // SetMemoryHeapSize() を発行
    SEQ_NONE();
    NNT_OS_LOG("SetMemoryHeapSize(0): result=");
    result = nn::os::SetMemoryHeapSize( 0 );
    NNT_OS_LOG( result.IsSuccess() ? "true\n" : "false\n" );
    ASSERT_TRUE( result.IsSuccess() );
}

TEST(FreeMemoryBlockDeathTest, test_FreeMemoryBlock)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // アロケータしたメモリブロックを R/O のまま返却して正しく動くか
    doTestFree(150, g_vSize);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// メモリヒープサイズの増減テスト
//---------------------------------------------------------------------------
// メモリヒープのサイズを変化させながら、Allocate と Free をヒープ限界まで
// 行なうテスト。MemoryManager 内部のリスト管理等が正しく動いていることを
// 確認する意図がある。
//
// Allocate でエラーが返ってきた時点での、トータルメモリ量がヒープサイズと
// 一致していることもテストする。
//---------------------------------------------------------------------------
void   doTestMemoryHeap(int seq, size_t heapSize, size_t allocateSize, int count, size_t leftSize)
{
    AllocatedMemory memBlockFailed;
    AllocatedMemory memBlockDummy;
    uintptr_t       gotAddress;
    size_t          gotSize;
    size_t          totalSize;
    nn::Result      result;

    // ヒープの設定
    SEQ_SET(seq);
    NNT_OS_LOG("Start MemoryHeap increase/decrease test.\n");

    // ヒープを現状より小さくする場合、そのサイズ分だけ確保済みにしてから試す
    if ((heapSize < nn::os::GetMemoryHeapSize()) &&
        (heapSize > 0))
    {
        // 変更後のヒープサイズ分だけメモリを確保してしまう
        SEQ_NONE();
        NNT_OS_LOG("AllocateMemoryBlock( %d KB ): ", heapSize / 1024);
        result = memBlockDummy.Allocate( heapSize );
        NNT_OS_LOG( result.IsSuccess() ? "true " : "false " );
        CheckBool( result.IsSuccess() );
        ASSERT_TRUE( result.IsSuccess() );

        SEQ_NONE();
        gotSize = memBlockDummy.GetSize();
        CheckParam( heapSize, gotSize );

        // ヒープサイズを変更する
        result = nn::os::SetMemoryHeapSize( heapSize );
        SEQ_NONE();
        NNT_OS_LOG("SetMemoryHeapSize( %d MB ): result=", heapSize / 1024 / 1024 );
        NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
        CheckBool( result.IsSuccess() );
        ASSERT_TRUE( result.IsSuccess() );

        // さらにヒープサイズを小さくしようとして ResultBusy() が正しく返るか
        result = nn::os::SetMemoryHeapSize( heapSize - (2 * 1024 * 1024) );
        SEQ_NONE();
        NNT_OS_LOG("SetMemoryHeapSize( %d MB ): result=", (heapSize / 1024 / 1024) - 2 );
        NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
        CheckBool( result <= nn::os::ResultBusy() );
        ASSERT_TRUE( result <= nn::os::ResultBusy() );

        // 確保していたメモリを解放する
        SEQ_NONE();
        NNT_OS_LOG("FreeMemoryBlock( %d KB ): ", heapSize / 1024);
        memBlockDummy.Free();
        CheckBool(true);
    }
    else
    {
        // ヒープが現状より大きくなる場合は普通に確保する
        result = nn::os::SetMemoryHeapSize( heapSize );
        SEQ_NONE();
        NNT_OS_LOG("SetMemoryHeapSize( %d MB ): result=", heapSize / 1024 / 1024 );
        NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
        CheckBool( result.IsSuccess() );
    }

    // GetMemoryHeapAddress() を発行
    gotAddress = nn::os::GetMemoryHeapAddress();
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapAddress(): address=0x%p\n", gotAddress);

    // GetMemoryHeapSize() を発行
    SEQ_NONE();
    NNT_OS_LOG("GetMemoryHeapSize():   ");
    gotSize = nn::os::GetMemoryHeapSize();
    CheckParam( heapSize, gotSize );

    // heapSize == 0 ならリターン
    if (nn::os::GetMemoryHeapSize() == 0)
    {
        NNT_OS_LOG("\n");
        return;
    }

    // MemoryAllocation (成功する)
    SEQ_NONE();
    NNT_OS_LOG("AllocateSize= %d KB\n", allocateSize / 1024);
    totalSize = 0;
    for (int i=0; i<count; i++)
    {
        result = g_memBlock[i].Allocate( allocateSize );
        SEQ_NONE();
        NNT_OS_LOG("[%3d] ", i);
        NNT_OS_LOG( result.IsSuccess() ? "true " : "false " );
        if ( !result.IsSuccess() )
        {
            CheckBool( result.IsSuccess() );
            FAIL();
            return;
        }

        gotAddress = g_memBlock[i].GetAddress();
        NNT_OS_LOG("adrs=0x%p ", gotAddress);

        gotSize = g_memBlock[i].GetSize();
        CheckParam( allocateSize, gotSize );
        totalSize += gotSize;
    }
    SEQ_NONE();
    NNT_OS_LOG("totalSize = %d (KB)\n", totalSize / 1024);

    // メモリ残量の確認その１ (最後の１回はメモリ不足で失敗する)
    result = memBlockFailed.Allocate( allocateSize );
    SEQ_NONE();
    NNT_OS_LOG("[Ex1] Size=%d(KB) ", allocateSize / 1024);
    NNT_OS_LOG( result.IsSuccess() ? "true " : "false " );
    CheckBool( !result.IsSuccess() );
    if ( result.IsSuccess() )
    {
        FAIL();
        return;
    }

    if (leftSize > 0)
    {
        // メモリ残量の確認その２（leftSize + MemoryBlockUnitSize で確認 → 失敗）
        result = memBlockFailed.Allocate( leftSize + nn::os::MemoryBlockUnitSize );
        SEQ_NONE();
        NNT_OS_LOG("[Ex2] Size=%d(KB) ", (leftSize + nn::os::MemoryBlockUnitSize) / 1024);
        NNT_OS_LOG( result.IsSuccess() ? "true " : "false " );
        CheckBool( !result.IsSuccess() );
        if ( result.IsSuccess() )
        {
            FAIL();
            return;
        }

        // メモリ残量の確認その３（leftSize で確認 → 成功）
        result = memBlockDummy.Allocate( leftSize );
        SEQ_NONE();
        NNT_OS_LOG("[Ex3] Size=%d(KB) ", leftSize / 1024);
        NNT_OS_LOG( result.IsSuccess() ? "true " : "false " );
        if ( !result.IsSuccess() )
        {
            CheckBool( result.IsSuccess() );
            FAIL();
            return;
        }

        gotSize = memBlockDummy.GetSize();
        CheckParam( leftSize, gotSize );
        totalSize += gotSize;

        SEQ_NONE();
        NNT_OS_LOG("totalSize = %d (KB)", totalSize / 1024);
        CheckParam( heapSize, totalSize );
    }

    // 全てのメモリブロックを順不同で返却
    SEQ_NONE();
    NNT_OS_LOG("Free all MemoryHeap");
    int index = 1;
    for (int i=0; i<count; i++)
    {
        while ( !(g_memBlock[index].IsAllocated() == true) )
        {
            index = (index < count - 1) ? index + 1 : 0;
        }

        g_memBlock[index].Free();
        index += 23;
        index = (index >= count) ? (index - count) : index;
    }
    if (leftSize > 0)
    {
        memBlockDummy.Free();
    }
    NNT_OS_LOG("... OK\n");

    // ヒープサイズと同じ容量のメモリ獲得して解放
    if (heapSize > 0)
    {
        SEQ_NONE();
        NNT_OS_LOG("AllocateMemoryBlock( %d KB ): ", heapSize / 1024);
        result = memBlockDummy.Allocate( heapSize );
        NNT_OS_LOG( result.IsSuccess() ? "true" : "false" );
        if ( !result.IsSuccess() )
        {
            CheckBool( result.IsSuccess() );
            FAIL();
            return;
        }
        NNT_OS_LOG("\n");

        SEQ_NONE();
        gotSize = memBlockDummy.GetSize();
        CheckParam( heapSize, gotSize );
        memBlockDummy.Free();
    }

    NNT_OS_LOG("\n");
}   // NOLINT(readability/fn_size)

TEST(MemoryHeapScaling, test_MemoryHeapScaling)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 一度ヒープサイズを 0 に初期化
    // こうしておかないと、既存のメモリヒープの拡張に失敗することがある。
    // （後方のメモリが使用済みなどの場合）
    auto result = nn::os::SetMemoryHeapSize( 0 );
    ASSERT_TRUE( result.IsSuccess() );

    // 最初の取得以降は、ヒープサイズを徐々に小さくしていく
    // ２回目以降、ヒープサイズを大きくできるかどうかは環境依存のため
    // heapSize=0 も可能なため、検査しておく。

    switch (nn::os::MemoryBlockUnitSize)
    {
    case 4 * 1024:
        // ------------- seq      HeapSize  AllocationSize Count    LeftSize
        doTestMemoryHeap(200, 32 * 1048576, 64 * 1024 * 11,   46, 384 * 1024 );
        doTestMemoryHeap(210,            0,              0,    0,          0 );
        doTestMemoryHeap(220, 16 * 1048576, 64 * 1024 *  9,   28, 256 * 1024 );
        doTestMemoryHeap(230,  8 * 1048576, 64 * 1024 *  7,   18, 128 * 1024 );
        doTestMemoryHeap(240,  4 * 1048576, 64 * 1024 *  5,   12, 256 * 1024 );
        doTestMemoryHeap(250,  2 * 1048576, 64 * 1024 *  3,   10, 128 * 1024 );
        doTestMemoryHeap(260,  1 * 1048576, 64 * 1024 *  1,   16,          0 );
        break;

    default:
        // ------------- seq      HeapSize  AllocationSize Count LeftSize
        doTestMemoryHeap(200, 32 * 1048576, 6 * 1048576,   5, 2 * 1048576 );
        doTestMemoryHeap(210, 32 * 1048576, 2 * 1048576,  16,           0 );
        doTestMemoryHeap(220,            0,           0,   0,           0 );
        doTestMemoryHeap(230, 30 * 1048576, 4 * 1048576,   7, 2 * 1048576 );
        doTestMemoryHeap(240, 28 * 1048576, 6 * 1048576,   4, 4 * 1048576 );
        doTestMemoryHeap(250, 26 * 1048576, 4 * 1048576,   6, 2 * 1048576 );
        doTestMemoryHeap(260,  2 * 1048576, 2 * 1048576,   1,           0 );
        break;
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::memoryHeap

//---------------------------------------------------------------------------
//  Test Main 関数

//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of MemoryHeap APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of MemoryHeap APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
