﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"
#include "../Common/test_Helper.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

#include <nnt/nntest.h>


namespace nnt { namespace os { namespace fiber {

//-----------------------------------------------------------------------------
//  ファイバ関連機能のテスト
//-----------------------------------------------------------------------------

namespace {

    const int FiberNum  = 8;

    const size_t StackSize = 8192;
    NN_OS_ALIGNAS_FIBER_STACK   uint8_t FiberStackWithNoGuard[StackSize];
    NN_OS_ALIGNAS_THREAD_STACK  uint8_t ThreadStack[StackSize];
    NN_OS_ALIGNAS_THREAD_STACK  uint8_t FiberStack[FiberNum][StackSize];

    // テスト実装は NINTENDOSDK-3609 を参照
    struct NN_ALIGNAS(16) Vec3
    {
        int8_t  a;
        int8_t  b;
        int8_t  c;
    };

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    #if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
    #pragma clang optimize off
    #endif
    // ポインタ値のアライメント判定を静的に判定されてしまうのを回避するため、
    // 自動変数のアドレスをこの関数を通して得るようにする。
    uintptr_t GetPointerValue(void* p) NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(p);
    }
    #if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
    #pragma clang optimize on
    #endif
#endif

    void CheckAlignment()
    {
        int8_t  aa;
        Vec3    v1;
        int8_t  bb;
        Vec3    v2;
        int8_t  cc;
        NN_ALIGNAS(16) int8_t dd;
        int8_t  ee;

        NN_LOG("aa:%p bb:%p cc:%p dd:%p ee:%p\n", &aa, &bb, &cc, &dd, &ee);
        NN_LOG("v1:%p v2:%p\n", &v1, &v2);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // sp レジスタの値もチェックしておく
        uintptr_t sp;
        asm volatile (" mov %0, sp" : "=r"(sp) );
        NN_LOG("sp:%p\n", sp);

    #if defined(NN_OS_CPU_ARM_AARCH32)
        EXPECT_TRUE( (GetPointerValue(&v1) & 0x7) == 0 );
        EXPECT_TRUE( (GetPointerValue(&v2) & 0x7) == 0 );
        EXPECT_TRUE( (sp & 0x7) == 0 );
    #elif defined(NN_OS_CPU_ARM_AARCH64)
        EXPECT_TRUE( (GetPointerValue(&v1) & 0xf) == 0 );
        EXPECT_TRUE( (GetPointerValue(&v2) & 0xf) == 0 );
        EXPECT_TRUE( (sp & 0xf) == 0 );
    #endif
#endif
    }

}   // namespace

//-----------------------------------------------------------------------------
// ファイバ関数
//-----------------------------------------------------------------------------

uintptr_t g_SequenceNum;
nn::os::FiberType   g_Fiber[FiberNum];
nn::os::ThreadType  g_Thread;


nn::os::FiberType* FiberFunction(void* argument)
{
    auto currIndex = reinterpret_cast<uintptr_t>( argument );
    nn::os::FiberType* fiber = &g_Fiber[currIndex];

    {   // １回目

        EXPECT_TRUE( g_SequenceNum == currIndex );
        ++g_SequenceNum;

        NNT_OS_LOG("g_Fiber[%d] is running.\n", currIndex);

        // ファイバの状態確認
        EXPECT_TRUE( &g_Thread  == nn::os::GetCurrentThread() );
        EXPECT_TRUE( fiber      == nn::os::GetCurrentFiber() );

        EXPECT_TRUE( fiber->_state == fiber->State_Running );

        auto   nextIndex = currIndex + 1;
        auto*  nextFiber = (nextIndex < FiberNum) ? &g_Fiber[nextIndex] : NULL;
        nn::os::SwitchToFiber( nextFiber );
    }

    {   // ２回目

        EXPECT_TRUE( g_SequenceNum == currIndex + FiberNum );
        ++g_SequenceNum;

        NNT_OS_LOG("g_Fiber[%d] is running.\n", currIndex);

        // ファイバの状態確認
        EXPECT_TRUE( &g_Thread  == nn::os::GetCurrentThread() );
        EXPECT_TRUE( fiber      == nn::os::GetCurrentFiber() );

        EXPECT_TRUE( fiber->_state == fiber->State_Running );

        auto   nextIndex = currIndex + 1;
        auto*  nextFiber = (nextIndex < FiberNum) ? &g_Fiber[nextIndex] : NULL;
        return nextFiber;
    }
}

//-----------------------------------------------------------------------------
// テスト用スレッド関数
// 以下の順でファイバへの変換および切換えを行ないます。２周します。
//
//  g_Thread -> g_Fiber[0] -> ... -> g_Fiber[7] -> g_Thread ->
//              g_Fiber[0] -> ... -> g_Fiber[7] -> g_Thread -> 終了
//-----------------------------------------------------------------------------

void TestThread(void* argument)
{
    // カレントスレッドでの状態確認
    EXPECT_TRUE( argument  == reinterpret_cast<void*>(0x12345678) );
    EXPECT_TRUE( &g_Thread == nn::os::GetCurrentThread() );
    EXPECT_TRUE( NULL      == nn::os::GetCurrentFiber() );

    // ファイバを作成する
    for (int i=0; i<FiberNum; ++i)
    {
        nn::os::FiberType* fiber = &g_Fiber[i];

        // 引数にファイバのインデックスを渡しておく
        auto* arg = reinterpret_cast<char*>(0) + i;
        nn::os::InitializeFiber( fiber,
                                 FiberFunction,
                                 reinterpret_cast<void*>(arg),
                                 FiberStack[i],
                                 StackSize,
                                 0 );

        EXPECT_TRUE( fiber->_state == fiber->State_Ready );
    }

    // ファイバの切換えテスト
    g_SequenceNum = 0;

    nn::os::SwitchToFiber( &g_Fiber[0] );
    EXPECT_TRUE( g_SequenceNum == FiberNum );

    // ファイバの状態チェック
    for (int i=0; i<FiberNum; ++i)
    {
        nn::os::FiberType* fiber = &g_Fiber[i];
        EXPECT_TRUE( fiber->_state == fiber->State_Ready );
    }

    nn::os::SwitchToFiber( &g_Fiber[0] );
    EXPECT_TRUE( g_SequenceNum == FiberNum * 2 );

    // ファイバを破棄
    for (int i=0; i<FiberNum; ++i)
    {
        nn::os::FiberType* fiber = &g_Fiber[i];

        EXPECT_TRUE( fiber->_state == fiber->State_Completed );
        nn::os::FinalizeFiber( fiber );
        EXPECT_TRUE( fiber->_state == fiber->State_NotInitialized );
    }

    // 最後のカレントスレッドの状態確認
    EXPECT_TRUE( &g_Thread == nn::os::GetCurrentThread() );
    EXPECT_TRUE( NULL      == nn::os::GetCurrentFiber() );
}

//-----------------------------------------------------------------------------
// シングルスレッド基本テスト
// 8 個のファイバを作成し、1 つのスレッド上で順番に回すテスト
//-----------------------------------------------------------------------------

TEST(Fiber, test_FiberOnSingleThread)
{
    auto result = nn::os::CreateThread( &g_Thread, TestThread, reinterpret_cast<void*>(0x12345678), ThreadStack, StackSize, nn::os::DefaultThreadPriority );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &g_Thread );
    nn::os::WaitThread( &g_Thread );
    nn::os::DestroyThread( &g_Thread );
}

//-----------------------------------------------------------------------------
// 1 個のファイバを作成し、ファイバ関数を終了させた後で Switch する。
//-----------------------------------------------------------------------------

nn::os::FiberType* FiberFunctionForCompletedAbort(void* argument)
{
    NN_UNUSED(argument);

    CheckAlignment();

    // 元スレッドに戻る
    return NULL;
}

TEST(FiberDeathTest, test_SwitchToCompletedFiber)
{
    nn::os::FiberType fiber;

    // 引数にファイバのインデックスを渡しておく
    nn::os::InitializeFiber( &fiber,
                             FiberFunctionForCompletedAbort,
                             NULL,
                             FiberStackWithNoGuard,
                             StackSize,
                             nn::os::FiberFlag_NoStackGuard );

    // ここの Switch は成功する
    nn::os::SwitchToFiber( &fiber );

    // もう一度 Switch してアボートするのを確認
    EXPECT_DEATH_IF_SUPPORTED( nn::os::SwitchToFiber(&fiber), "" );

    // ファイバを破棄
    nn::os::FinalizeFiber( &fiber );
}

//-----------------------------------------------------------------------------
// 1 個のファイバを作成し、実行中のファイバ関数（自分自身）に Switch する。
//-----------------------------------------------------------------------------

nn::os::FiberType* FiberFunctionForRunningAbort(void* argument)
{
    NN_UNUSED(argument);

    // もう一度 Switch してアボートするのを確認
    auto* pFiber = nn::os::GetCurrentFiber();
    EXPECT_DEATH_IF_SUPPORTED( nn::os::SwitchToFiber(pFiber), "" );

    // 元スレッドに戻る
    return NULL;
}

TEST(FiberDeathTest, test_SwitchToRunningFiber)
{
    nn::os::FiberType fiber;

    // 引数にファイバのインデックスを渡しておく
    nn::os::InitializeFiber( &fiber,
                             FiberFunctionForRunningAbort,
                             NULL,
                             FiberStackWithNoGuard,
                             StackSize,
                             nn::os::FiberFlag_NoStackGuard );

    // ここの Switch は成功する
    nn::os::SwitchToFiber( &fiber );

    // ファイバを破棄
    nn::os::FinalizeFiber( &fiber );
}

//-----------------------------------------------------------------------------

}}} // namespace nnt::os::fiber

