﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"
#include "../Common/test_Helper.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>

#include <nnt/nntest.h>

namespace nnt { namespace os { namespace debug {

const size_t    MemoryHeapSize      = 0x08000000;
const size_t    TransferMemorySize  = 0x00400000;
const size_t    SharedMemorySize    = 0x00800000;

NN_OS_ALIGNAS_THREAD_STACK  uint8_t ThreadStackArray[0x4000];

//---------------------------------------------------------------------------

extern "C" void nninitStartup()
{
}

//---------------------------------------------------------------------------
//  メモリ情報の取得とダンプ
//---------------------------------------------------------------------------

void DisplayMemoryInfo(nn::os::MemoryInfo* p)
{
    nn::os::QueryMemoryInfo( p );

    NN_LOG("Total Available Memory    : 0x%p\n", p->totalAvailableMemorySize);
    NN_LOG("Total Used Memory         : 0x%p\n", p->totalUsedMemorySize);
    NN_LOG("Total MemoryHeap size     : 0x%p\n", p->totalMemoryHeapSize);
    NN_LOG("Program Size              : 0x%p\n", p->programSize);
    NN_LOG("Allocated MemoryHeap size : 0x%p\n", p->allocatedMemoryHeapSize);
    NN_LOG("Total Threads Stack size  : 0x%p\n", p->totalThreadStackSize);
    NN_LOG("Current thread count      : %d\n",   p->threadCount);
}

//---------------------------------------------------------------------------
//  メモリ情報取得テスト用のサブスレッド
//---------------------------------------------------------------------------

void MemoryInfoThreadFunction(void* arg)
{
    nn::os::Event& event = *reinterpret_cast<nn::os::Event*>(arg);

    event.Wait();
}

//---------------------------------------------------------------------------
//  メモリ情報取得テスト
//---------------------------------------------------------------------------

TEST(QueryMemoryInfo, test_QueryMemoryInfoBasic)
{
    nn::os::MemoryInfo  info;

    nn::os::Event event(nn::os::EventClearMode_AutoClear);

    // メモリ情報取得
    NN_LOG(NN_TEXT("★ OS オブジェクト生成前 ★\n"));
    DisplayMemoryInfo( &info );

    size_t currentTotalStackSize = info.totalThreadStackSize;
    EXPECT_TRUE( info.totalMemoryHeapSize      == 0 );
    EXPECT_TRUE( info.allocatedMemoryHeapSize  == 0 );
    EXPECT_TRUE( info.threadCount == 1 );

    // メモリヒープを設定
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );
    EXPECT_TRUE( result.IsSuccess() );

    // スレッドを 1 つ生成
    nn::os::ThreadType  thread;
    result = nn::os::CreateThread( &thread,
                                   MemoryInfoThreadFunction,
                                   &event,
                                   ThreadStackArray,
                                   sizeof(ThreadStackArray),
                                   nn::os::DefaultThreadPriority );
    EXPECT_TRUE( result.IsSuccess() );
    nn::os::StartThread( &thread );

    // TransferMemory を 1 つ用意してマップ
    uintptr_t transferAddress;
    result = nn::os::AllocateMemoryBlock(&transferAddress, TransferMemorySize);
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::TransferMemory  trans(reinterpret_cast<void*>(transferAddress),
                                  TransferMemorySize,
                                  nn::os::MemoryPermission_None);
    void* transMapAddress;
    result = trans.Map(&transMapAddress, nn::os::MemoryPermission_None);
    EXPECT_TRUE( result.IsSuccess() );

    // SharedMemory を 1 つ用意してマップ
    nn::os::SharedMemory    shmem(SharedMemorySize,
                                  nn::os::MemoryPermission_ReadWrite,
                                  nn::os::MemoryPermission_ReadWrite);
    auto* shmemAddress = shmem.Map(nn::os::MemoryPermission_ReadWrite);
    EXPECT_TRUE( shmemAddress != NULL );


    // メモリ情報を再取得
    NN_LOG(NN_TEXT("★ OS オブジェクト生成後 ★\n"));
    DisplayMemoryInfo( &info );

    EXPECT_TRUE( info.totalMemoryHeapSize      == MemoryHeapSize     );
    EXPECT_TRUE( info.allocatedMemoryHeapSize  == TransferMemorySize );
    EXPECT_TRUE( info.totalThreadStackSize     == currentTotalStackSize + sizeof(ThreadStackArray) );
    EXPECT_TRUE( info.threadCount == 2 );


    // 後片付け
    trans.Unmap();
    trans.~TransferMemory();
    nn::os::FreeMemoryBlock(transferAddress, TransferMemorySize);

    shmem.Unmap();
    shmem.~SharedMemory();

    event.Signal();
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // メモリヒープを設定
    result = nn::os::SetMemoryHeapSize( 0 );
    EXPECT_TRUE( result.IsSuccess() );


    // 再度にもう一度、メモリ情報取得
    NN_LOG(NN_TEXT("★ OS オブジェクト破棄後 ★\n"));
    DisplayMemoryInfo( &info );

    EXPECT_TRUE( info.totalMemoryHeapSize      == 0 );
    EXPECT_TRUE( info.allocatedMemoryHeapSize  == 0 );
    EXPECT_TRUE( info.totalThreadStackSize     == currentTotalStackSize );
    EXPECT_TRUE( info.threadCount == 1 );
}

//---------------------------------------------------------------------------

}}} // namespace nnt::os::debug

