﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SdkMeasureStackUsage.h>

#include <nnt/nntest.h>
#include "../Common/test_SetjmpWithFpuContext.h"
#include "../Common/test_Helper.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

namespace nnt { namespace os { namespace debug {

namespace {

nntosJmpbufWithFpuContext   g_JumpBuf;

NN_ALIGNAS(16) uint8_t      g_UserExceptionHandlerStack[0x4000];
nn::os::UserExceptionInfo   g_UserExceptionInfo;

NN_OS_ALIGNAS_GUARDED_STACK uint8_t g_TestStack[0x4000];

}

//---------------------------------------------------------------------------
//  スレッド内でのスタック情報取得テスト
//---------------------------------------------------------------------------

void TestThreadSubFunctionForStackUsage()
{
    size_t size4;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
    nn::os::YieldThread();
    NN_OS_SDK_GET_STACK_USAGE(size4);
    NN_LOG("[StackUsage] size4 = %zu (byte)\n", size4);
}

void TestThreadFunctionForStackUsage(void* arg)
{
    NN_UNUSED(arg);

    // 何も挟まない場合はサイズ 0 を返す
    {
        size_t size0;
        NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
        NN_OS_SDK_GET_STACK_USAGE(size0);
        NN_LOG("[StackUsage] size0 = %zu (byte)\n", size0);
        NN_ABORT_UNLESS_EQUAL(0, size0);
    }

    // 明示的にスタックを汚した場合
    {
        size_t size;
        NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
        volatile uint64_t* sp;
        asm volatile ("mov %0, sp" : "=&r"(sp) :: "memory");
        // 適当なサイズ分だけスタックを汚す
        const int length = 17;
        for (int i=0; i<length; ++i)
        {
            *(sp - 1 - i) = 0x5a5a5a5a5a5a5a5aull;
        }
        NN_OS_SDK_GET_STACK_USAGE(size);
        NN_LOG("[StackUsage] Make stack region dirty: %zu (byte)\n", size);
        NN_ABORT_UNLESS_EQUAL(length * sizeof(*sp), size);
    }

    // 基本的な計測
    size_t size1;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
    nn::os::GetCurrentThread();
    NN_OS_SDK_GET_STACK_USAGE(size1);
    NN_LOG("[StackUsage] nn::os::GetCurrentThread(): %zu (byte)\n", size1);
    NN_ABORT_UNLESS_GREATER_EQUAL(size1, 0);
    NN_ABORT_UNLESS_LESS(size1, sizeof(g_TestStack));

    // 多重ネスト
    size_t size2;
    size_t size3;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
    {
        NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
        TestThreadSubFunctionForStackUsage();
        NN_OS_SDK_GET_STACK_USAGE(size3);
        NN_LOG("[StackUsage] size3 = %zu (byte)\n", size3);

        NN_ABORT_UNLESS_GREATER_EQUAL(size3, 0);
        NN_ABORT_UNLESS_LESS(size3, sizeof(g_TestStack));
    }
    NN_OS_SDK_GET_STACK_USAGE(size2);
    NN_LOG("[StackUsage] size2 = %zu (byte)\n", size2);
    NN_ABORT_UNLESS_GREATER_EQUAL(size2, 0);
    NN_ABORT_UNLESS_LESS(size2, sizeof(g_TestStack));
    NN_ABORT_UNLESS_GREATER_EQUAL(size2, size3);

    SUCCEED();
}

TEST(MeasureStackUsageTest, test_InThread)
{
    nn::os::ThreadType thread;

    size_t size;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &thread, TestThreadFunctionForStackUsage, nullptr, g_TestStack, sizeof(g_TestStack), nn::os::DefaultThreadPriority) );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    NN_OS_SDK_GET_STACK_USAGE(size);
    NN_LOG("[StackUsage] test_InThread: %zu (byte)\n", size);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 0);
    NN_ABORT_UNLESS_LESS(size, sizeof(g_TestStack));

    SUCCEED();
}

//---------------------------------------------------------------------------
//  ファイバ内でのスタック情報取得テスト
//---------------------------------------------------------------------------

nn::os::FiberType* TestFiberFunctionForStackUsage(void* arg)
{
    auto* nextFiber = reinterpret_cast<nn::os::FiberType*>(arg);

    size_t size;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();
    nn::os::GetCurrentFiber();
    NN_OS_SDK_GET_STACK_USAGE(size);
    NN_LOG("[StackUsage] nn::os::GetCurrentFiber(): %zu (byte)\n", size);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 0);
    NN_ABORT_UNLESS_LESS(size, sizeof(g_TestStack));

    return nextFiber;
}

TEST(MeasureStackUsageTest, test_InFiber)
{
    nn::os::FiberType fiber;

    size_t size;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE();

    nn::os::InitializeFiber( &fiber, TestFiberFunctionForStackUsage, nn::os::GetCurrentFiber(), g_TestStack, sizeof(g_TestStack), 0 );
    nn::os::SwitchToFiber( &fiber );
    nn::os::FinalizeFiber( &fiber );

    NN_OS_SDK_GET_STACK_USAGE(size);
    NN_LOG("[StackUsage] test_InFiber: %zu (byte)\n", size);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 0);
    NN_ABORT_UNLESS_LESS(size, sizeof(g_TestStack));

    SUCCEED();
}

//---------------------------------------------------------------------------
//  ユーザ例外ハンドラ内でのスタック情報取得テスト
//---------------------------------------------------------------------------

void UserExceptionHandlerFunctionForStackUsage(nn::os::UserExceptionInfo* info)
{
    NN_UNUSED( info );

    size_t size;
    NN_OS_SDK_PREPARE_TO_MEASURE_STACK_USAGE_IMPL(g_UserExceptionHandlerStack);
    NN_LOG("[StackUsage] UserExceptionHandlerFunctionForStackUsage\n");
    NN_OS_SDK_GET_STACK_USAGE_IMPL(size, g_UserExceptionHandlerStack);
    NN_LOG("[StackUsage] UserExceptionHandler: %zu (byte)\n", size);
    NN_ABORT_UNLESS_GREATER_EQUAL(size, 0);
    NN_ABORT_UNLESS_LESS(size, sizeof(g_TestStack));

    // 問題なければ、メモリアクセス違反を起こす直前へ戻る
    nntosLongjmpWithFpuContext( &g_JumpBuf );
}

TEST(MeasureStackUsageTest, test_InUserExceptionHandler)
{
    // ユーザ例外ハンドラの登録
    SetUserExceptionHandler(UserExceptionHandlerFunctionForStackUsage, g_UserExceptionHandlerStack, sizeof(g_UserExceptionHandlerStack), &g_UserExceptionInfo);

    // ユーザ例外ハンドラからの復帰ポイント
    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        // メモリアクセス違反を起こしてユーザ例外ハンドラを起動する
        *reinterpret_cast<volatile int*>(NULL) = 0;

        // メモリアクセス違反が起こらなければテスト失敗
        ADD_FAILURE();
    }

    // longjmp() で復帰できればテスト成功
    SUCCEED();

    RestoreUserExceptionHandler();
}

//---------------------------------------------------------------------------

}}} // namespace nnt::os::debug

#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) */
