﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/os/os_UserExceptionHandler.h>

#include <nnt/nntest.h>
#include "../Common/test_SetjmpWithFpuContext.h"
#include "../Common/test_Helper.h"


namespace nnt { namespace os { namespace debug {

nntosJmpbufWithFpuContext   g_JumpBuf;

NN_ALIGNAS(16) uint8_t      g_UserExceptionHandlerStack[16384];
nn::os::UserExceptionInfo   g_UserExceptionInfo;

NN_OS_ALIGNAS_GUARDED_STACK uint8_t g_TestStack[8192];

//---------------------------------------------------------------------------
//  共通テスト関数
//---------------------------------------------------------------------------

void CheckStackRegion(void* stack, size_t stackSize)
{
    uintptr_t   stackTop = reinterpret_cast<uintptr_t>(stack);
    uintptr_t   gotStackTop;
    uintptr_t   gotStackTopForCheck;
    size_t      gotStackSize;
    size_t      gotStackSizeForCheck;

    nn::os::GetCurrentStackInfo( &gotStackTopForCheck, NULL );
    nn::os::GetCurrentStackInfo( NULL, &gotStackSizeForCheck );
    nn::os::GetCurrentStackInfo( &gotStackTop, &gotStackSize );
    EXPECT_TRUE( gotStackTop  == gotStackTopForCheck  );
    EXPECT_TRUE( gotStackSize == gotStackSizeForCheck );

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NNT_OS_LOG("Expect    stack region: 0x%p - 0x%p\n", stackTop,    stackTop + stackSize );
#else
    NN_UNUSED(stackTop);
    NN_UNUSED(stackSize);
#endif
    NNT_OS_LOG("Got       stack region: 0x%p - 0x%p\n", gotStackTop, gotStackTop + gotStackSize );
    NNT_OS_LOG("Real vairable on stack: 0x%p\n", &gotStackSize);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_TRUE( stackTop == gotStackTop );
    EXPECT_TRUE( gotStackTop + gotStackSize == stackTop + stackSize );
#endif

    EXPECT_TRUE( gotStackTop <= reinterpret_cast<uintptr_t>(&gotStackTop) );
    EXPECT_TRUE( reinterpret_cast<uintptr_t>(&gotStackTop) <= gotStackTop + gotStackSize );
}

//---------------------------------------------------------------------------
//  スレッド内でのスタック情報取得テスト
//---------------------------------------------------------------------------

void TestThreadFunctionForStackInfo(void* arg)
{
    NN_UNUSED(arg);

    auto* thread = nn::os::GetCurrentThread();

    NNT_OS_LOG(NN_TEXT("スレッド内でのテスト\n"));
    CheckStackRegion( thread->_stack, thread->_stackSize );
}

TEST(GetCurrentStackInfoTest, test_InThread)
{
    nn::os::ThreadType  thread;

    auto result = nn::os::CreateThread( &thread,
                                        TestThreadFunctionForStackInfo,
                                        NULL,
                                        g_TestStack,
                                        sizeof(g_TestStack),
                                        nn::os::DefaultThreadPriority );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    SUCCEED();
}

//---------------------------------------------------------------------------
//  ファイバ内でのスタック情報取得テスト
//---------------------------------------------------------------------------

nn::os::FiberType* TestFiberFunction(void* arg)
{
    auto* nextFiber = reinterpret_cast<nn::os::FiberType*>(arg);

    auto* fiber = nn::os::GetCurrentFiber();

    NNT_OS_LOG(NN_TEXT("ファイバ内でのテスト\n"));
    CheckStackRegion( fiber->_stack, fiber->_stackSize );

    return nextFiber;
}

TEST(GetCurrentStackInfoTest, test_InFiber)
{
    nn::os::FiberType   fiber;

    nn::os::InitializeFiber( &fiber,
                             TestFiberFunction,
                             nn::os::GetCurrentFiber(),
                             g_TestStack,
                             sizeof(g_TestStack),
                             0 );

    nn::os::SwitchToFiber( &fiber );

    nn::os::FinalizeFiber( &fiber );

    SUCCEED();
}

//---------------------------------------------------------------------------
//  ユーザ例外ハンドラ内でのスタック情報取得テスト
//---------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
void UserExceptionHandlerFunction(nn::os::UserExceptionInfo* info)
{
    NN_UNUSED( info );

    NNT_OS_LOG(NN_TEXT("ユーザ例外ハンドラ内でのテスト\n"));
    CheckStackRegion( g_UserExceptionHandlerStack,
                      sizeof(g_UserExceptionHandlerStack) );

    // 問題なければ、メモリアクセス違反を起こす直前へ戻る
    nntosLongjmpWithFpuContext( &g_JumpBuf );
}

TEST(GetCurrentStackInfoTest, test_InUserExceptionHandler)
{
    // ユーザ例外ハンドラの登録
    SetUserExceptionHandler(UserExceptionHandlerFunction, g_UserExceptionHandlerStack, sizeof(g_UserExceptionHandlerStack), &g_UserExceptionInfo);

    // ユーザ例外ハンドラからの復帰ポイント
    if (nntosSetjmpWithFpuContext( &g_JumpBuf ) == 0)
    {
        // メモリアクセス違反を起こしてユーザ例外ハンドラを起動する
        *reinterpret_cast<volatile int*>(NULL) = 0;

        // メモリアクセス違反が起こらなければテスト失敗
        ADD_FAILURE();
    }

    // longjmp() で復帰できればテスト成功
    SUCCEED();

    RestoreUserExceptionHandler();
}
#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) */


//---------------------------------------------------------------------------
//  CPU アイドル時間の取得テスト
//---------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(GetIdleTickCountTest, test_GetIdleTickCountTest)
{
    auto interval = nn::TimeSpan::FromMilliSeconds(500);
    nn::os::TimerEvent  timer(nn::os::EventClearMode_AutoClear);

    // TimerEvent を使い、CPU 無駄遣いと Signal 待ちの両方を行ない、
    // GetIdleTickCount() でその区間における CPU アイドル時間を取得する。

    auto startTick      = nn::os::GetSystemTick();
    auto startIdleTick  = nn::os::GetIdleTickCount();

    timer.StartPeriodic( interval, interval );

    timer.Wait();               // アイドル区間を作る
    while (!timer.TryWait())    // CPU の無駄使いをポーリングで行なう
    {
        // Do nothing
    }

    auto endIdleTick    = nn::os::GetIdleTickCount();
    auto endTick        = nn::os::GetSystemTick();
    timer.Stop();

    // 計測結果
    auto totalTime  = (endTick     - startTick).ToTimeSpan();
    auto idleTime   = (endIdleTick - startIdleTick).ToTimeSpan();

    NN_LOG("Total time span: %lld (usec)\n", totalTime.GetMicroSeconds());
    NN_LOG("Idle time span:  %lld (usec)\n", idleTime.GetMicroSeconds());

    // 他のスレッドが動作したり、CPU 負荷が変動したとしても、
    // totalTick - idleTick は interval 以上必ずあるはず。
    EXPECT_TRUE( totalTime - idleTime >= interval );

    // アイドル時間はほぼ interval に近い値となるはずだが、
    // ここでは最低限 0 より大きいことだけチェックしておく。
    EXPECT_TRUE( idleTime > 0 );
}
#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) */

//---------------------------------------------------------------------------

}}} // namespace nnt::os::debug

