﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

#include <nnt/nntest.h>
#include <cstring>

namespace nnt { namespace os { namespace compilerThreadLocal {

NN_OS_ALIGNAS_THREAD_STACK char g_Stack1[ 8192 ];
NN_OS_ALIGNAS_THREAD_STACK char g_Stack2[ 8192 ];
NN_OS_ALIGNAS_THREAD_STACK char g_Stack3[ 8192 ];

//---------------------------------------------------------------------------

uintptr_t GetTpidrRo()
{
    uintptr_t value_ro;
#if defined NN_BUILD_CONFIG_CPU_ARM64
        asm volatile("mrs  %0, tpidrro_el0": "=&r"(value_ro) :: "memory");
#else
        asm volatile("mrc  p15, 0, %0, c13, c0, 3": "=&r"(value_ro) :: "memory");
#endif
    //NN_LOG("tpidr_ro = 0x%p\n", value_ro);
    return value_ro;
}

uintptr_t GetTpidrRw()
{
    uintptr_t value_rw;
#if defined NN_BUILD_CONFIG_CPU_ARM64
        asm volatile("mrs  %0, tpidr_el0": "=&r"(value_rw) :: "memory");
#else
        asm volatile("mrc  p15, 0, %0, c13, c0, 2": "=&r"(value_rw) :: "memory");
#endif
    //NN_LOG("tpidr_rw = 0x%p\n", value_rw);
    return value_rw;
}

//---------------------------------------------------------------------------
// Rynda 0.14 以降になったら有効にする（0.13 では non-trivial 非対応）
#if 0
class TestClass
{
private:
    uintptr_t    m_Value;

public:
    TestClass()
    {
        m_Value = 0;
        NN_LOG("TestClass: Constructor. this=0x%p\n", this);
    }

    ~TestClass()
    {
        NN_LOG("TestClass: Destructor.\n");
    }

    void SetValue(uintptr_t value)
    {
        m_Value = value;
    }

    uintptr_t GetValue()
    {
        return m_Value;
    }
};

thread_local TestClass g_Instance;


void ThreadFunc1(void *arg)
{
    NN_UNUSED(arg);

    NN_LOG("ThreadFunc1: arg              = %d\n", reinterpret_cast<uintptr_t>(arg));
    NN_LOG("ThreadFunc1: &g_Instance      = 0x%p\n", &g_Instance);
    NN_LOG("ThreadFunc1: g_Instance.value = %d\n", g_Instance.GetValue());
    g_Instance.SetValue( reinterpret_cast<uintptr_t>(arg) );
    NN_LOG("ThreadFunc1: g_Instance.value = %d\n", g_Instance.GetValue());
}

TEST(CompilerThreadLocal, test_UseThreadLocalInstanceInTwoThreads)
{
    // メインスレッドから呼ぶ
    ThreadFunc1( reinterpret_cast<void*>( 1 ) );

    // スレッドを作成して呼ぶ
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, ThreadFunc1, reinterpret_cast<void*>(2), g_Stack1, sizeof(g_Stack1), nn::os::DefaultThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );
}
#endif

//---------------------------------------------------------------------------

thread_local    int g_TlsValue;

void ThreadFunc2(void *arg)
{
    uintptr_t argument = reinterpret_cast<uintptr_t>(arg);

    // TPIDR レジスタ値の確認
    // スレッド起動直後（TLS 変数アクセス前）の TPIDR レジスタの期待値の確認
    auto tpidr_ro = GetTpidrRo();
    auto tpidr_rw = GetTpidrRw();

    EXPECT_TRUE( tpidr_ro != 0 );
    if (argument == 2)
    {
        // サブスレッドとして起動された場合のみ tpidr_rw の初期値をチェックする
        EXPECT_TRUE( tpidr_rw == 0 );
    }

    // TLS 変数へのアクセス
    // これにより tpidr_rw が書き換えられる
    int* p = &g_TlsValue;
    NN_LOG("&g_TlsValue = 0x%p\n", p);

#if defined(NN_BUILD_CONFIG_ABI_ILP32)
    // ↑ここの defined() は AArch64 向けの thread_local の対応が入れば、
    // 外すことができます。

    // TPIDR レジスタ値の確認
    // TLS 変数へのアクセス後の TPIDR レジスタの期待値の確認
    tpidr_rw = GetTpidrRw();
    EXPECT_TRUE( tpidr_rw != 0 );
#endif
}

TEST(CompilerThreadLocal, test_UseThreadLocalVariableInTwoThreads)
{
    // メインスレッドから呼ぶ
    ThreadFunc2( reinterpret_cast<void*>( 1 ) );

    // TPIDR レジスタ値の保存
    auto saved_tpidr_ro = GetTpidrRo();
    auto saved_tpidr_rw = GetTpidrRw();
    NN_LOG("main: saved_tpidr_ro= 0x%p\n", saved_tpidr_ro);
    NN_LOG("main: saved_tpidr_rw= 0x%p\n", saved_tpidr_rw);

    // スレッドを作成して呼ぶ
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread(&thread, ThreadFunc2, reinterpret_cast<void*>(2), g_Stack1, sizeof(g_Stack1), nn::os::DefaultThreadPriority);
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::StartThread( &thread );
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // TPIDR レジスタ値の確認
    // ThreadFunc2 へ遷移して戻ってきた段階で、保存していた値と一致するかを確認
    auto restored_tpidr_ro = GetTpidrRo();
    auto restored_tpidr_rw = GetTpidrRw();
    NN_LOG("main: restored_tpidr_ro= 0x%p\n", restored_tpidr_ro);
    NN_LOG("main: restored_tpidr_rw= 0x%p\n", restored_tpidr_rw);

    EXPECT_TRUE( saved_tpidr_ro == restored_tpidr_ro );
    EXPECT_TRUE( saved_tpidr_rw == restored_tpidr_rw );
}

//---------------------------------------------------------------------------

}}} // namespace nnt::os::compilerThreadLocal

//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_LOG("nnMain()\n");

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    RUN_ALL_TESTS();
}

