﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

using namespace nn;

namespace {
    const int64_t   SaveDataSize = 0x00200000;   // 2MB
    const int64_t   JournalSize = 0x00200000;   // 2MB
    const int       SaveDataFlag = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    class OlscMountManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
        }

        static void TearDownTestCase()
        {
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
            }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    class Stopwatch
    {
    public:
        Stopwatch(bool enable, const char* fmt, ...) : m_Enable(enable)
        {
            std::va_list vlist;
            va_start(vlist, fmt);
            util::VSNPrintf(m_Title, sizeof(m_Title), fmt, vlist);
            va_end(vlist);

            m_Begin = os::GetSystemTick();
        }
        ~Stopwatch()
        {
            if (m_Enable)
            {
                auto end = os::GetSystemTick();
                NN_LOG("%s: %lld ms\n", m_Title, (end - m_Begin).ToTimeSpan().GetMilliSeconds());
            }
        }
    private:
        char m_Title[128];
        bool m_Enable;
        os::Tick m_Begin;
    };

    Result GetUid(account::Uid* out, int index) NN_NOEXCEPT
    {
        int listCount;
        account::Uid uid[account::UserCountMax];
        NN_RESULT_DO(account::ListAllUsers(&listCount, uid, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uid[index];
        NN_RESULT_SUCCESS;
    }

    const int ThreadStackSize = 16 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
}

TEST_F(OlscMountManagerTest, TryAcquireGlobalSaveData)
{
    using namespace olsc::srv::util;
//    MountManager mm;

    MountInfo mi =
    {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    enum class Tag : Bit8
    {
        SaveType0,
    };

    using MountManagerForTest = MountManager<Tag, 2, 3>;
    MountManagerForTest mm;

    mm.RegisterTag(Tag::SaveType0, mi);

    // 同一のセーブデータに対して、読み・読み・書き
    {
        auto r0 = mm.TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
        EXPECT_TRUE(r0);

        auto r1 = mm.TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
        EXPECT_TRUE(r1);

        auto w0 = mm.TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::ReadAndWrite);
        EXPECT_FALSE(w0);

        mm.Release(*r0, ReferenceMode::Read);
        mm.Release(*r1, ReferenceMode::Read);
    }

    // 同一のセーブデータに対して、書き・読み
    {
        auto w0 = mm.TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::ReadAndWrite);
        EXPECT_TRUE(w0);

        auto r0 = mm.TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
        EXPECT_TRUE(r0);

        os::ThreadType t0;
        NNT_EXPECT_RESULT_SUCCESS(os::CreateThread(&t0, [](void* p) {
            auto m = reinterpret_cast<MountManagerForTest*>(p);
            auto id = m->TryAcquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
            EXPECT_FALSE(id);
        }, &mm, g_ThreadStack, sizeof(g_ThreadStack), os::DefaultThreadPriority));

        os::StartThread(&t0);
        os::DestroyThread(&t0);

        mm.Release(*r0, ReferenceMode::Read);
        mm.Release(*w0, ReferenceMode::ReadAndWrite);
    }
}

TEST_F(OlscMountManagerTest, AcquireUserSaveData)
{
    using namespace olsc::srv::util;
    //    MountManager mm;

    MountInfo mi =
    {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    enum class Tag : Bit8
    {
        SaveType0,
    };

    using MountManagerForTest = MountManager<Tag, 2, 3>;
    MountManagerForTest mm;

    mm.RegisterTag(Tag::SaveType0, mi);

    os::ThreadType t0;
    NNT_EXPECT_RESULT_SUCCESS(os::CreateThread(&t0, [](void* p) {
        auto m = reinterpret_cast<MountManagerForTest*>(p);
        auto id = m->Acquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
        os::SleepThread(TimeSpan::FromSeconds(5));
        m->Release(id, ReferenceMode::Read);
    }, &mm, g_ThreadStack, sizeof(g_ThreadStack), os::DefaultThreadPriority));

    os::StartThread(&t0);
    os::SleepThread(TimeSpan::FromSeconds(1));

    {
        auto id = mm.Acquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::Read);
        mm.Release(id, ReferenceMode::Read);
    }
    {
        auto id = mm.Acquire(Tag::SaveType0, account::InvalidUid, ReferenceMode::ReadAndWrite);
        mm.Release(id, ReferenceMode::ReadAndWrite);
    }

    os::DestroyThread(&t0);
}


TEST_F(OlscMountManagerTest, TryAcquireUserSaveData)
{
    using namespace olsc::srv::util;
    //    MountManager mm;

    MountInfo mi0 =
    {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    MountInfo mi1 =
    {
        SystemSaveDataId + 1,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    enum class Tag : Bit8
    {
        SaveType0,
        SaveType1,
    };

    MountManager<Tag, 2, 3> mm;

    mm.RegisterTag(Tag::SaveType0, mi0);
    mm.RegisterTag(Tag::SaveType1, mi1);

    account::Uid u0;
    NNT_ASSERT_RESULT_SUCCESS(GetUid(&u0, 0));
    account::Uid u1;
    NNT_ASSERT_RESULT_SUCCESS(GetUid(&u1, 1));

    {
        auto r00 = mm.TryAcquire(Tag::SaveType0, u0, ReferenceMode::Read);
        EXPECT_TRUE(r00);

        auto r01 = mm.TryAcquire(Tag::SaveType0, u0, ReferenceMode::Read);
        EXPECT_TRUE(r01);

        auto w00 = mm.TryAcquire(Tag::SaveType0, u0, ReferenceMode::ReadAndWrite);
        EXPECT_FALSE(w00);

        auto r10 = mm.TryAcquire(Tag::SaveType0, u1, ReferenceMode::Read);
        EXPECT_TRUE(r10);

        auto r11 = mm.TryAcquire(Tag::SaveType0, u1, ReferenceMode::Read);
        EXPECT_TRUE(r11);

        auto w10 = mm.TryAcquire(Tag::SaveType0, u1, ReferenceMode::ReadAndWrite);
        EXPECT_FALSE(w10);


        mm.Release(*r00, ReferenceMode::Read);
        mm.Release(*r01, ReferenceMode::Read);
        mm.Release(*r10, ReferenceMode::Read);
        mm.Release(*r11, ReferenceMode::Read);
    }

    {
        auto w00 = mm.TryAcquire(Tag::SaveType0, u0, ReferenceMode::ReadAndWrite);
        EXPECT_TRUE(w00);

        auto r00 = mm.TryAcquire(Tag::SaveType0, u0, ReferenceMode::Read);
        EXPECT_TRUE(r00);

        auto w10 = mm.TryAcquire(Tag::SaveType0, u1, ReferenceMode::ReadAndWrite);
        EXPECT_TRUE(w10);

        auto r10 = mm.TryAcquire(Tag::SaveType0, u1, ReferenceMode::Read);
        EXPECT_TRUE(r10);

        mm.Release(*w00, ReferenceMode::ReadAndWrite);
        mm.Release(*r00, ReferenceMode::Read);
        mm.Release(*w10, ReferenceMode::ReadAndWrite);
        mm.Release(*r10, ReferenceMode::Read);
    }
}

TEST_F(OlscMountManagerTest, MountContext)
{
    using namespace olsc::srv::util;

    MountInfo mi0 =
    {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    enum class Tag : Bit8
    {
        SaveType0,
    };

    using MountManagerForTest = MountManager<Tag, 2, 3>;
    MountManagerForTest mm;

    mm.RegisterTag(Tag::SaveType0, mi0);

    {
        os::ThreadType t0;
        NNT_EXPECT_RESULT_SUCCESS(os::CreateThread(&t0, [](void* p) {
            auto m = reinterpret_cast<MountManagerForTest*>(p);
            auto read01 = m->AcquireForRead(Tag::SaveType0, account::InvalidUid);
            os::SleepThread(TimeSpan::FromSeconds(5));
        }, &mm, g_ThreadStack, sizeof(g_ThreadStack), os::DefaultThreadPriority));

        os::StartThread(&t0);
        os::SleepThread(TimeSpan::FromSeconds(1));

        {
            auto read00 = mm.AcquireForRead(Tag::SaveType0, account::InvalidUid);
        }

        auto write00 = mm.AcquireForWrite(Tag::SaveType0, account::InvalidUid);
        os::DestroyThread(&t0);
    }

}
