﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <olsc_ComponentFileInfoAdaptor.h>
#include <olsc_OutputStream.h>

#include "testOlsc_InputStream.h"

using namespace nn;

namespace component_file_success {

const char ComponentFileJson[] =
"{"
    "\"save_data_archive\": {"
        "\"component_files\" : ["
        "{"
            // 全部最大値
            "\"id\"                      : 9223372036854775807,"
            "\"save_data_archive_id\"    : 9223372036854775807,"
            "\"index\"                   : 65535,"
            "\"datatype\"                : \"save\","
            "\"status\"                  : \"hand_over\","
            "\"raw_size\"                : 4294967295,"
            "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"archive_size\"            : 4294967295,"
            "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
            "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
            "\"created_at_as_unixtime\"  : 9223372036854775807,"
            "\"updated_at_as_unixtime\"  : 9223372036854775807"
        "},"
        "{"
            // 現実的な値、かつ値が混同されていないことが確認できる
            "\"id\"                      : 1111111111111111111,"
            "\"save_data_archive_id\"    : 2222222222222222222,"
            "\"index\"                   : 11111,"
            "\"datatype\"                : \"meta\","
            "\"status\"                  : \"uploading\","
            "\"raw_size\"                : 2222222222,"
            "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"archive_size\"            : 1111111111,"
            "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
            "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
            "\"created_at_as_unixtime\"  : 1514732400,"
            "\"updated_at_as_unixtime\"  : 1514732401"
        "}"
        "]"
    "}"
"}";

struct Cancelable
{
    bool IsCancelled() const { return false; }
};

const Bit8 digest1[] = {
    0x6c, 0x3d, 0x60, 0xca, 0xbe, 0x8a, 0x72, 0x4e,
    0x2d, 0xed, 0x11, 0x31, 0x87, 0xf3, 0x0f, 0x0c,
    0xf7, 0xd5, 0x2b, 0xed, 0xda, 0xa0, 0x26, 0x1b,
    0xec, 0x26, 0xe9, 0xd0, 0x5b, 0x81, 0xad, 0x7f
};

const Bit8 digest2[] = {
    0xdd, 0x5a, 0xbc, 0x08, 0x9e, 0xdc, 0xa3, 0x0c,
    0x35, 0x03, 0x6c, 0xfb, 0x82, 0x8a, 0x14, 0xcd,
};

const char url1[] = "https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91";

TEST(OlscComponentFileJsonAdaptorTest, NormalParse)
{
    const int MetaCount = 3;
    olsc::srv::ComponentFileInfo meta[MetaCount];

    olsc::srv::adaptor::ComponentFileInfoMemoryOutputStream outputStream(meta, MetaCount);
    olsc::srv::adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

    nnt::olsc::MemoryInputStreamForRapidJson inputStream;
    inputStream.Set(ComponentFileJson, sizeof(ComponentFileJson));

    const Cancelable c;
    NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
    NNT_EXPECT_RESULT_SUCCESS(adaptor.GetResult());

    {
        EXPECT_EQ(INT64_MAX, meta[0].id);
        EXPECT_EQ(INT64_MAX, meta[0].sdaId);
        EXPECT_EQ(UINT16_MAX, meta[0].clientArgument.chunkId);
        EXPECT_EQ(olsc::srv::ComponentFileType::Save, meta[0].type);
        EXPECT_EQ(olsc::srv::ComponentFileStatus::HandOver, meta[0].status);
        EXPECT_EQ(UINT32_MAX, meta[0].componentFileSize);
        EXPECT_EQ(0, std::memcmp(digest1, meta[0].componentFileDigest.data, sizeof(digest1)));
        EXPECT_EQ(UINT32_MAX, meta[0].saveDataChunkSize);
        EXPECT_EQ(0, std::memcmp(digest2, meta[0].saveDataChunkDigest.data, sizeof(digest2)));
        EXPECT_EQ(INT64_MAX, meta[0].createdAt.value);
        EXPECT_EQ(INT64_MAX, meta[0].updatedAt.value);
        EXPECT_EQ(nn::util::Strncmp(url1, meta[0].url, nn::olsc::srv::ComponentFileInfo::MaxUrlLength + 1), 0);
    }

    {
        EXPECT_EQ(1111111111111111111, meta[1].id);
        EXPECT_EQ(2222222222222222222, meta[1].sdaId);
        EXPECT_EQ(11111, meta[1].clientArgument.chunkId);
        EXPECT_EQ(olsc::srv::ComponentFileType::Meta, meta[1].type);
        EXPECT_EQ(olsc::srv::ComponentFileStatus::Uploading, meta[1].status);
        EXPECT_EQ(2222222222, meta[1].componentFileSize);
        EXPECT_EQ(0, std::memcmp(digest1, meta[1].componentFileDigest.data, sizeof(digest1)));
        EXPECT_EQ(1111111111, meta[1].saveDataChunkSize);
        EXPECT_EQ(0, std::memcmp(digest2, meta[1].saveDataChunkDigest.data, sizeof(digest2)));
        EXPECT_EQ(1514732400, meta[1].createdAt.value);
        EXPECT_EQ(1514732401, meta[1].updatedAt.value);
        EXPECT_EQ(nn::util::Strncmp(url1, meta[0].url, nn::olsc::srv::ComponentFileInfo::MaxUrlLength + 1), 0);
    }
}

} // namespace component_file

namespace component_file_fail {
    const int JsonLen = 1024;
    const char InvalidDigestLength[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"abD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
            "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"a3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
            "]}}"
    };

    const char InvalidStatus[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploadinga\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}"
    };

    const char InvalidDataType[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"metaa\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}"
    };

    const char InvalidIndex[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 4294967296,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}"
    };

    const char InvalidComponentFileSize[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 4294967296,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 0,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"a3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}"
    };

    const char InvalidSaveDataChunkSize[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 1111111111,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 4294967296,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 1111111111,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 0,"
                "\"encoded_archive_digest\"  : \"a3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
        "]}}"
    };

    const char LackOfNecessaryFieldJson[][JsonLen] =
    {
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\""
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\""
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\""
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"index\"                   : 11111,"
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\""
            "}"
        "]}}",
        "{ \"save_data_archive\": {"
            "\"component_files\": ["
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"get_url\"                 : \"https://scsi-cb-storage-for-test.s3-us-west-2.amazonaws.com/0011223344556677/01FFFFFFFFFFFFFF/1234/12340_1_dedf109d8378158d.bin?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIF3HMDE7QDGFWO5A%2F20170908%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170908T075622Z&X-Amz-Expires=600&X-Amz-SignedHeaders=host&X-Amz-Signature=b065d3a0520b5cd93e75aa8b2156b2fe359d4c619737193255bdeddad6fc3b91\""
            "}"
        "]}}"
    };

    // 1 つ目のパースに失敗すると 2 つ目以降は正常でもスルーされる
    const char IgnoreAfterFailure[] =
    "{"
        "\"save_data_archive\": {"
            "\"component_files\" : ["
            "{"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\""
            "},"
            "{"
                "\"id\"                      : 1111111111111111111,"
                "\"save_data_archive_id\"    : 2222222222222222222,"
                "\"index\"                   : 11111,"
                "\"datatype\"                : \"meta\","
                "\"status\"                  : \"uploading\","
                "\"raw_size\"                : 2222222222,"
                "\"encoded_raw_digest\"      : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
                "\"archive_size\"            : 1111111111,"
                "\"encoded_archive_digest\"  : \"3Vq8CJ7coww1A2z7gooUzQ==\","
                "\"created_at_as_unixtime\"  : 1514732400,"
                "\"updated_at_as_unixtime\"  : 1514732401"
            "}"
            "]"
        "}"
    "}";

    struct Cancelable
    {
        bool IsCancelled() const { return false; }
    };

    template<typename ExpectResultType>
    void TestImpl(const char json[])
    {
        const int MetaCount = 10;
        olsc::srv::ComponentFileInfo meta[MetaCount];

        olsc::srv::adaptor::ComponentFileInfoMemoryOutputStream outputStream(meta, MetaCount);
        olsc::srv::adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

        nnt::olsc::MemoryInputStreamForRapidJson inputStream;
        inputStream.Set(json, JsonLen);

        const Cancelable c;
        NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
        NNT_EXPECT_RESULT_FAILURE(ExpectResultType, adaptor.GetResult());

        EXPECT_TRUE(outputStream.GetCount() == 0);
    }

    template<typename ExpectResultType>
    void RunTest(const char json[][JsonLen], int count)
    {
        for (int i = 0; i < count; ++i)
        {
            TestImpl<ExpectResultType>(json[i]);
        }
    }

    TEST(OlscJsonAdaptorTest, ComponentFileMetaFail)
    {
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidStatus, sizeof(InvalidStatus) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidDataType, sizeof(InvalidDataType) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidIndex, sizeof(InvalidIndex) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidComponentFileSize, sizeof(InvalidComponentFileSize) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidSaveDataChunkSize, sizeof(InvalidSaveDataChunkSize) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(InvalidDigestLength, sizeof(InvalidDigestLength) / JsonLen);
        RunTest<olsc::ResultComponentFileUnacceptableContent>(LackOfNecessaryFieldJson, sizeof(LackOfNecessaryFieldJson) / JsonLen);
        TestImpl<olsc::ResultComponentFileUnacceptableContent>(IgnoreAfterFailure);
    }

} // namespace component_file_fail
