﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

using namespace nn;
using namespace nn::olsc;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    class OlscTransferTaskContextDatabaseTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

    struct ContextParam
    {
        TransferTaskId      id;
        int                 totalComponentCount;
        size_t              totalSize;
    };

    ContextParam    CreateContextParam(int n)
    {
        return ContextParam{
            static_cast<TransferTaskId>(n),
            n + 1,
            static_cast<size_t>((n + 1) * 1024 * 1024),
        };
    }


    using Database = olsc::srv::database::TransferTaskContextDatabase;
    using TransferTaskContext = Database::TransferTaskContext;

    TransferTaskContext   CreateExpectedContext(int n)
    {
        auto p = CreateContextParam(n);

        return TransferTaskContext {
            p.id,
            p.totalSize,
            0,
            0,
            p.totalComponentCount,
            0,
            {},
            nn::ResultSuccess()
        };
    }
}

TEST_F(OlscTransferTaskContextDatabaseTest, Basic)
{
    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    Database db(mountManager);

    for (int i = 0; i < Database::MaxTransferTaskContextCount; ++i)
    {
        // 追加
        auto p = CreateContextParam(i);
        auto appender = [&p, &db]() -> Result {
            return db.CreateTransferTaskContext(p.id, p.totalComponentCount, p.totalSize);
        };
        NNT_EXPECT_RESULT_SUCCESS(appender());
        if (i < Database::MaxTransferTaskContextCount - 1)
        {
            // 重複
            NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskContextDuplicated, appender());
        }

        // 取得
        TransferTaskContext context;
        NNT_ASSERT_RESULT_SUCCESS(db.GetTransferTaskContext(&context, p.id));
        EXPECT_EQ(CreateExpectedContext(i), context);

        // 書き換え
        context.currentComponentIndex++;
        NNT_ASSERT_RESULT_SUCCESS(db.UpdateTransferTaskContext(context));
        TransferTaskContext updatedContext;

        NNT_ASSERT_RESULT_SUCCESS(db.GetTransferTaskContext(&updatedContext, p.id));
        EXPECT_EQ(context, updatedContext);
    }

    for (int i = 0; i < Database::MaxTransferTaskContextCount; ++i)
    {
        // 削除
        auto p = CreateContextParam(i);
        auto remover = [&p, &db]() -> Result {
            return db.RemoveTransferTaskContext(p.id);
        };
        NNT_EXPECT_RESULT_SUCCESS(remover());
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskContextNotFound, remover());

        TransferTaskContext context;
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskContextNotFound, db.GetTransferTaskContext(&context, p.id));
    }
}
