﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/util/olsc_ActiveUserList.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    class OlscUserListTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            nnt::olsc::Finalize();
        }

    protected:
        static Result InitializeSaveData(fs::UserId userId)
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, userId))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }

    };

    class ActiveUserListForTest : public nn::olsc::srv::util::ActiveUserListBase
    {
    public:
        NN_IMPLICIT ActiveUserListForTest(TimeSpan updateInterval) NN_NOEXCEPT
            : nn::olsc::srv::util::ActiveUserListBase(updateInterval)
        {}

        void SetActiveUsers(const account::Uid activeUsers[], int count) NN_NOEXCEPT
        {
            std::memcpy(m_ActiveUsers, activeUsers, sizeof(m_ActiveUsers[0]) * count);
            m_ActiveUserCount = count;
        }

    protected:
        virtual int ListActiveUsers(account::Uid out[], int maxListCount) const NN_NOEXCEPT NN_OVERRIDE
        {
            std::memcpy(out, m_ActiveUsers, sizeof(m_ActiveUsers[0]) * m_ActiveUserCount);
            return m_ActiveUserCount;
        }
    private:
        account::Uid m_ActiveUsers[account::UserCountMax] {};
        int m_ActiveUserCount = 0;
    };
}

// 2 アカウントが存在し NSA が有効なケース
TEST_F(OlscUserListTest, Basic)
{
    nn::olsc::srv::util::ActiveUserList ul(nn::TimeSpan::FromSeconds(1));

    auto validUser0 = nnt::olsc::GetUserId(0);
    auto validUser1 = nnt::olsc::GetUserId(1);
    nn::account::Uid invalidUser = { 1 };

    EXPECT_EQ(0, ul.GetCount());
    EXPECT_FALSE(ul.IsActive(validUser0));

    // 最新情報で更新
    ul.Refresh();
    EXPECT_EQ(2, ul.GetCount());
    EXPECT_TRUE(ul.IsActive(validUser0));
    EXPECT_TRUE(ul.IsActive(validUser1));

    // リフレッシュ直後はすぐに UL チェックすべき
    account::Uid listBuffer[account::UserCountMax];
    auto listedCount = ul.ListUsersToUpdate(listBuffer, std::extent<decltype(listBuffer), 0>::value);
    EXPECT_EQ(2, listedCount);
    EXPECT_FALSE(ul.IsActive(invalidUser));

    // 最終チェック日時の更新
    ul.ScheduleNextUpdate(validUser0);
    ul.ScheduleNextUpdate(validUser1);
    auto interval = ul.GetTimeSpanToNextUpdate();
    EXPECT_TRUE(interval.GetMilliSeconds() > 0 && interval.GetMilliSeconds() <= 1000);

    // 更新後1秒未満はインターバル未経過なため List されない
    listedCount = ul.ListUsersToUpdate(listBuffer, std::extent<decltype(listBuffer), 0>::value);
    EXPECT_EQ(0, listedCount);

    // 更新1秒経過後は List される
    os::SleepThread(TimeSpan::FromSeconds(1));
    listedCount = ul.ListUsersToUpdate(listBuffer, std::extent<decltype(listBuffer), 0>::value);
    EXPECT_EQ(2, listedCount);
    EXPECT_EQ(validUser0, listBuffer[0]);
}

// アクティブから非アクティブに変化するケース
TEST_F(OlscUserListTest, ChangeStateActiveToDeactive)
{
    ActiveUserListForTest ul(nn::TimeSpan::FromSeconds(1));

    const account::Uid ValidUser0 = { 1 };
    const account::Uid ValidUser1 = { 2 };
    const account::Uid ValidUser2 = { 3 };
    const account::Uid ValidUser3 = { 4 };

    const account::Uid ActiveUsers[] = {
        ValidUser0, ValidUser1, ValidUser2, ValidUser3
    };

    ul.SetActiveUsers(ActiveUsers, 4);
    EXPECT_EQ(0, ul.GetCount());

    ul.Refresh();
    EXPECT_EQ(4, ul.GetCount());
    EXPECT_TRUE(ul.IsActive(ValidUser0));
    EXPECT_TRUE(ul.IsActive(ValidUser1));
    EXPECT_TRUE(ul.IsActive(ValidUser2));
    EXPECT_TRUE(ul.IsActive(ValidUser3));

    ul.SetActiveUsers(ActiveUsers, 1);
    ul.Refresh();
    EXPECT_EQ(1, ul.GetCount());
    EXPECT_TRUE(ul.IsActive(ValidUser0));
    EXPECT_FALSE(ul.IsActive(ValidUser1));
    EXPECT_FALSE(ul.IsActive(ValidUser2));
    EXPECT_FALSE(ul.IsActive(ValidUser3));

    ul.SetActiveUsers(ActiveUsers, 0);
    ul.Refresh();
    EXPECT_EQ(0, ul.GetCount());
    EXPECT_FALSE(ul.IsActive(ValidUser0));
    EXPECT_FALSE(ul.IsActive(ValidUser1));
    EXPECT_FALSE(ul.IsActive(ValidUser2));
    EXPECT_FALSE(ul.IsActive(ValidUser3));
}
