﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_Network.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <vector>

#include "testOlsc_Util.h"


using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;

namespace {

    class GetNsaIdTokenTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

    };


    account::Uid GetFirstUserId() NN_NOEXCEPT
    {
        int userCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserCount(&userCount));
        NN_ABORT_UNLESS(userCount > 0);

        account::Uid uid;
        int listCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&listCount, &uid, 1));
        return uid;
    }

    class TimeoutCancelable : public nn::util::Cancelable
    {
    public:
        NN_IMPLICIT TimeoutCancelable(int timeoutSec) NN_NOEXCEPT
        {
            m_TimeoutSec = timeoutSec;

            os::CreateThread(&m_WaitThread, [](void* p) {
                auto t = reinterpret_cast<TimeoutCancelable*>(p);
                os::SleepThread(TimeSpan::FromSeconds(t->m_TimeoutSec));
                t->Cancel();
            }, this, m_ThreadStack, sizeof(m_ThreadStack), os::DefaultThreadPriority);

            os::StartThread(&m_WaitThread);
        }

        ~TimeoutCancelable() NN_NOEXCEPT
        {
            os::DestroyThread(&m_WaitThread);
        }
    private:
        int m_TimeoutSec;
        os::ThreadType m_WaitThread = {};

        NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[8192];
    };
}

TEST_F(GetNsaIdTokenTest, Basic)
{
    nifm::NetworkConnection connection(os::EventClearMode_ManualClear);
    os::TimerEvent timeoutEvent(os::EventClearMode_ManualClear);

    static TimeoutCancelable tc(20);
    NNT_ASSERT_RESULT_SUCCESS(olsc::srv::util::SubmitAndWaitNetworkConnection(connection, nifm::RequirementPreset_InternetForSystemProcess, tc));

    nn::util::Cancelable cancelable;
    auto uid = GetFirstUserId();
    static char nsaIdTokenBuffer[account::NetworkServiceAccountIdTokenLengthMax] = {};
    size_t tokenSize;
    NNT_ASSERT_RESULT_SUCCESS(olsc::srv::util::GetNsaIdToken(&tokenSize, nsaIdTokenBuffer, sizeof(nsaIdTokenBuffer), uid, cancelable));
}
