﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/vi.h>
#include <nvn/nvn.h>
#include <nv/nv_MemoryManagement.h>

#include "Application.h"

extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char* name);
}

namespace {
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

    void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }
    void NvFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
}

AppBase* Application::m_pApp;
nn::vi::NativeWindowHandle Application::m_NativeWindow;
nn::vi::Display* Application::m_pDisplay;
nn::vi::Layer* Application::m_pLayer;
int Application::m_WindowWidth;
int Application::m_WindowHeight;
PFNNVNBOOTSTRAPLOADERPROC Application::m_pBootstrapLoader;

void Application::Run(AppBase* pApp)
{
    m_pApp = pApp;

    {
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);

        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }

    {
        nn::vi::Initialize();

        nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
        NN_ASSERT(result.IsSuccess());

        m_WindowWidth = 1280;
        m_WindowHeight = 720;

        result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
        NN_ASSERT(result.IsSuccess());

        nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);

        result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
    }

    {
        m_pBootstrapLoader = nvnBootstrapLoader;
    }

    m_pApp->OnInit(m_pBootstrapLoader, m_NativeWindow);

    while (true)
    {
        m_pApp->OnUpdate();

        if (m_pApp->IsExitRequested())
        {
            break;
        }

        m_pApp->OnRender();
    }

    m_pApp->OnDestroy();

    nn::vi::Finalize();
}
