﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/time/time_Result.h>
#include <nn/ntc/detail/service/core/ntc_SerializeTaskThread.h>

namespace
{
    int32_t g_TestCounter1 = 0;
    int32_t g_TestCounter2 = 0;

    // g_TestCounter1 を +1 して成功するタスク
    class ExecutorTask1 : public nn::ntc::detail::service::core::TaskExecutor
    {
        virtual nn::Result Execute() NN_NOEXCEPT NN_OVERRIDE
        {
            g_TestCounter1 += 1;
            NN_RESULT_SUCCESS;
        }
    };

    // g_TestCounter2 を +2 して成功するタスク
    class ExecutorTask2 : public nn::ntc::detail::service::core::TaskExecutor
    {
        virtual nn::Result Execute() NN_NOEXCEPT NN_OVERRIDE
        {
            g_TestCounter2 += 2;
            NN_RESULT_SUCCESS;
        }
    };

    // nn::time::ResultOutOfMemory() で失敗するタスク
    class ExecutorTask3 : public nn::ntc::detail::service::core::TaskExecutor
    {
        virtual nn::Result Execute() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_THROW(nn::time::ResultOutOfMemory());
        }
    };

    // 100 ミリ秒まって成功するタスク
    class ExecutorTask4 : public nn::ntc::detail::service::core::TaskExecutor
    {
        virtual nn::Result Execute() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NN_RESULT_SUCCESS;
        }
    };

    ExecutorTask1 g_ExecutorTask1;
    ExecutorTask2 g_ExecutorTask2;
    ExecutorTask3 g_ExecutorTask3;
    ExecutorTask4 g_ExecutorTask4;

    class TestTaskResouce
    {
    public:
        explicit TestTaskResouce(nn::ntc::detail::service::core::TaskExecutor* pTaskExecutor):
            m_Resource(nn::time::ResultNotStarted(), pTaskExecutor),
            m_TaskNode(&m_Resource)
        {
        }
        nn::ntc::detail::service::core::TaskResource m_Resource;
        nn::ntc::detail::service::core::TaskResourceNode m_TaskNode;
    };

    NN_OS_ALIGNAS_THREAD_STACK char g_StackBuffer[1024 * 16];
    nn::ntc::detail::service::core::SerializeTaskThread g_TestTaskThread;
}

TEST(TaskThread, Basic)
{
    g_TestCounter1 = 0;
    g_TestCounter2 = 0;

    g_TestTaskThread.StartThread(g_StackBuffer, sizeof(g_StackBuffer), 17, "TestTaskThread");

    TestTaskResouce task1(&g_ExecutorTask1);
    TestTaskResouce task2(&g_ExecutorTask2);
    TestTaskResouce task3(&g_ExecutorTask3);
    TestTaskResouce task4(&g_ExecutorTask4);

    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task1.m_TaskNode));
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task2.m_TaskNode));
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task3.m_TaskNode));
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));

    // task1
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task1.m_TaskNode));
    task1.m_Resource.m_SystemEvent.Wait();
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task1.m_TaskNode));
    NNT_EXPECT_RESULT_SUCCESS(task1.m_Resource.m_Result);
    EXPECT_EQ(1, g_TestCounter1);

    // task2
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task2.m_TaskNode));
    task2.m_Resource.m_SystemEvent.Wait();
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task2.m_TaskNode));
    NNT_EXPECT_RESULT_SUCCESS(task2.m_Resource.m_Result);
    EXPECT_EQ(2, g_TestCounter2);

    // task3
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task3.m_TaskNode));
    task3.m_Resource.m_SystemEvent.Wait();
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task3.m_TaskNode));
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOutOfMemory, task3.m_Resource.m_Result);

    // task4
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultNotStarted, task4.m_Resource.m_Result); // Result は初期状態のはず
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task4.m_TaskNode));
    // task4 は100ミリ秒待って完了するので、タスク完了までの状態をテスト
    EXPECT_FALSE(g_TestTaskThread.RegisterTask(task4.m_TaskNode)); // 重複 Register は false が返る
    EXPECT_TRUE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    EXPECT_FALSE(task4.m_Resource.m_SystemEvent.TryWait()); // 100 msec かかるので非シグナル状態のはず
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultProcessing, task4.m_Resource.m_Result);
    task4.m_Resource.m_SystemEvent.Wait();
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    NNT_EXPECT_RESULT_SUCCESS(task4.m_Resource.m_Result);

    // task4 もう一度実行
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task4.m_TaskNode)); // 完了後に再度 Register できる
    EXPECT_TRUE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    EXPECT_FALSE(task4.m_Resource.m_SystemEvent.TryWait()); // 100 msec かかるので非シグナル状態のはず
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultProcessing, task4.m_Resource.m_Result);
    task4.m_Resource.m_SystemEvent.Wait();
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    NNT_EXPECT_RESULT_SUCCESS(task4.m_Resource.m_Result);

    // task4 もう一度実行→キャンセル
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task4.m_TaskNode)); // 完了後に再度 Register できる
    EXPECT_TRUE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    EXPECT_FALSE(task4.m_Resource.m_SystemEvent.TryWait()); // 100 msec かかるので非シグナル状態のはず
    g_TestTaskThread.DoneTask(nn::time::ResultCanceled(), task4.m_TaskNode);
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    task4.m_Resource.m_SystemEvent.Wait();
    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultCanceled, task4.m_Resource.m_Result);

    // task4 もう一度タスク登録→すぐ登録解除
    EXPECT_TRUE(g_TestTaskThread.RegisterTask(task4.m_TaskNode)); // 完了後に再度 Register できる
    EXPECT_TRUE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));
    EXPECT_FALSE(task4.m_Resource.m_SystemEvent.TryWait()); // 100 msec かかるので非シグナル状態のはず
    g_TestTaskThread.UnregisterTaskResource(task4.m_TaskNode);
    EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task4.m_TaskNode));

    // task1, task2 以外で変わってないか
    EXPECT_EQ(1, g_TestCounter1);
    EXPECT_EQ(2, g_TestCounter2);

    g_TestTaskThread.StopThread(nn::time::ResultOutOfMemory()); // 適当なエラーを入れる
}

template<int TryCount>
class ClientThread
{
public:
    nn::os::ThreadType thread;
    NN_OS_ALIGNAS_THREAD_STACK char stack[1024 * 4];

    void Initialize(nn::ntc::detail::service::core::TaskExecutor* pTaskExecutor)
    {
        NNT_ASSERT_RESULT_SUCCESS( nn::os::CreateThread(&thread,
        [](void* pUserData) {
            nn::ntc::detail::service::core::TaskExecutor* pTaskExecutor = static_cast<nn::ntc::detail::service::core::TaskExecutor*>(pUserData);
            for(int i = 0 ; i < TryCount ; ++i)
            {
                TestTaskResouce task(pTaskExecutor);
                EXPECT_TRUE(g_TestTaskThread.RegisterTask(task.m_TaskNode));
                task.m_Resource.m_SystemEvent.Wait();
                EXPECT_FALSE(g_TestTaskThread.IsRegisteredTask(task.m_TaskNode));
                if(pTaskExecutor == &g_ExecutorTask3)
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::time::ResultOutOfMemory, task.m_Resource.m_Result);
                }
                else
                {
                    NNT_EXPECT_RESULT_SUCCESS(task.m_Resource.m_Result);
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
        },
        pTaskExecutor, stack, sizeof(stack), 20) );
    }

    void Start()
    {
        nn::os::StartThread(&thread);
    }

    void Stop()
    {
        nn::os::DestroyThread(&thread);
    }
};

TEST(TaskThread, MultiThreadUse)
{
    g_TestCounter1 = 0;
    g_TestCounter2 = 0;

    g_TestTaskThread.StartThread(g_StackBuffer, sizeof(g_StackBuffer), 17, "TestTaskThread");

    static ClientThread<30> clientThread1;
    static ClientThread<30> clientThread2;
    static ClientThread<30> clientThread3;
    static ClientThread<30> clientThread4;

    clientThread1.Initialize(&g_ExecutorTask1);
    clientThread2.Initialize(&g_ExecutorTask2);
    clientThread3.Initialize(&g_ExecutorTask3);
    clientThread4.Initialize(&g_ExecutorTask4);

    clientThread1.Start();
    clientThread2.Start();
    clientThread3.Start();
    clientThread4.Start();

    clientThread1.Stop();
    clientThread2.Stop();
    clientThread3.Stop();
    clientThread4.Stop();

    EXPECT_EQ(30, g_TestCounter1);
    EXPECT_EQ(60, g_TestCounter2);

    g_TestTaskThread.StopThread(nn::time::ResultOutOfMemory()); // 適当なエラーを入れる
}
