﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/detail/jwt/nsd_JwtParser.h>
#include <nn/nsd/detail/json/nsd_RapidJsonEventAccepterForClaims.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/nsd_TypesPrivate.h>

#include <nn/util/util_Base64.h>
#include <nn/util/util_TFormatString.h>

using namespace nn::nsd::detail::jwt;
using namespace nn::nsd::detail::json;

template <size_t StringBufferSize, size_t DocumentBufferSize>
class Stream : public InputStreamForRapidJson
{
private:
    char m_StringBuffer[StringBufferSize];
    char m_DocumentBuffer[DocumentBufferSize];

    const char* m_pSrcBuffer;
    const size_t m_SrcBufferSize;
    size_t m_PositionInSrcBuffer;

protected:
    virtual nn::Result RequestBufferImpl(size_t *pOutActualRead, void *buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_SrcBufferSize <= m_PositionInSrcBuffer )
        {
            *pOutActualRead = 0;
            NN_RESULT_SUCCESS;
        }

        size_t copySize = bufferSize;
        if( m_SrcBufferSize < m_PositionInSrcBuffer + copySize )
        {
            copySize  = m_SrcBufferSize - m_PositionInSrcBuffer;
        }
        std::memcpy(buffer, &m_pSrcBuffer[m_PositionInSrcBuffer], copySize);
        m_PositionInSrcBuffer += copySize;
        *pOutActualRead = copySize;
        NN_RESULT_SUCCESS;
    }

public:
    Stream(const char* pBuffer, size_t bufferSize):
        InputStreamForRapidJson(
            m_StringBuffer, StringBufferSize,
            m_DocumentBuffer, DocumentBufferSize),
        m_pSrcBuffer(pBuffer),
        m_SrcBufferSize(bufferSize),
        m_PositionInSrcBuffer(0)
    {}
};

namespace
{
    nn::nsd::SaveData g_SaveData = {};
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData1)
{
    // FqdnEntriesExpire: 2871763199 -> 2060/12/31 23:59:59. int32_t の範囲を超えるやつ.
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);
    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);
    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("hoge_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    EXPECT_STREQ("app1_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("app1_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("app2_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("app2_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }

}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData2)
{
    // FqdnEntriesExpire: 946684800 -> 2000/01/01 00:00:00. int32_t の範囲内のやつ.
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"deadbeef!!!\",\
    \"Expire\":946684800,\
    \"Version\":123}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("deadbeef!!!", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"deadbeef!!!"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(123, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);
    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);
    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("hoge_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(946684800LL, g_SaveData.expireUnixTime);
    EXPECT_STREQ("app1_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("app1_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("app2_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("app2_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[2].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[2].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData3)
{
    // FqdnEntriesExpire: 0

    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd3\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("test test test!", g_SaveData.settingName.value);
    EXPECT_STREQ("dd3", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);
    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);
    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    for(int i = 1 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(0LL, g_SaveData.expireUnixTime);
    for(int i = 0 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData4)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge2_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge3_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge4_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge5_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge6_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge7_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge8_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge9_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge10_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge11_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge12_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge13_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge14_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge15_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge16_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);
    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);
    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("hoge2_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[1].dst.value);
    EXPECT_STREQ("hoge3_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[2].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[2].dst.value);
    EXPECT_STREQ("hoge4_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[3].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[3].dst.value);
    EXPECT_STREQ("hoge5_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[4].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[4].dst.value);
    EXPECT_STREQ("hoge6_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[5].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[5].dst.value);
    EXPECT_STREQ("hoge7_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[6].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[6].dst.value);
    EXPECT_STREQ("hoge8_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[7].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[7].dst.value);
    EXPECT_STREQ("hoge9_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[8].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[8].dst.value);
    EXPECT_STREQ("hoge10_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[9].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[9].dst.value);
    EXPECT_STREQ("hoge11_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[10].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[10].dst.value);
    EXPECT_STREQ("hoge12_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[11].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[11].dst.value);
    EXPECT_STREQ("hoge13_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[12].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[12].dst.value);
    EXPECT_STREQ("hoge14_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[13].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[13].dst.value);
    EXPECT_STREQ("hoge15_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[14].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[14].dst.value);
    EXPECT_STREQ("hoge16_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[15].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[15].dst.value);

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    EXPECT_STREQ("app1_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("app1_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("app2_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("app2_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[2].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[2].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData5)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app0\",\"hoge\"],[\"app1\",\"hoge\"],[\"app2\",\"hoge\"],[\"app3\",\"hoge\"],[\"app4\",\"hoge\"],[\"app5\",\"hoge\"],[\"app6\",\"hoge\"],[\"app7\",\"hoge\"],[\"app8\",\"hoge\"],[\"app9\",\"hoge\"],\
            [\"app10\",\"hoge\"],[\"app11\",\"hoge\"],[\"app12\",\"hoge\"],[\"app13\",\"hoge\"],[\"app14\",\"hoge\"],[\"app15\",\"hoge\"],[\"app16\",\"hoge\"],[\"app17\",\"hoge\"],[\"app18\",\"hoge\"],[\"app19\",\"hoge\"],\
            [\"app20\",\"hoge\"],[\"app21\",\"hoge\"],[\"app22\",\"hoge\"],[\"app23\",\"hoge\"],[\"app24\",\"hoge\"],[\"app25\",\"hoge\"],[\"app26\",\"hoge\"],[\"app27\",\"hoge\"],[\"app28\",\"hoge\"],[\"app29\",\"hoge\"],\
            [\"app30\",\"hoge\"],[\"app31\",\"hoge\"],[\"app32\",\"hoge\"],[\"app33\",\"hoge\"],[\"app34\",\"hoge\"],[\"app35\",\"hoge\"],[\"app36\",\"hoge\"],[\"app37\",\"hoge\"],[\"app38\",\"hoge\"],[\"app39\",\"hoge\"],\
            [\"app40\",\"hoge\"],[\"app41\",\"hoge\"],[\"app42\",\"hoge\"],[\"app43\",\"hoge\"],[\"app44\",\"hoge\"],[\"app45\",\"hoge\"],[\"app46\",\"hoge\"],[\"app47\",\"hoge\"],[\"app48\",\"hoge\"],[\"app49\",\"hoge\"],\
            [\"app50\",\"hoge\"],[\"app51\",\"hoge\"],[\"app52\",\"hoge\"],[\"app53\",\"hoge\"],[\"app54\",\"hoge\"],[\"app55\",\"hoge\"],[\"app56\",\"hoge\"],[\"app57\",\"hoge\"],[\"app58\",\"hoge\"],[\"app59\",\"hoge\"],\
            [\"app60\",\"hoge\"],[\"app61\",\"hoge\"],[\"app62\",\"hoge\"],[\"app63\",\"hoge\"],[\"app64\",\"hoge\"],[\"app65\",\"hoge\"],[\"app66\",\"hoge\"],[\"app67\",\"hoge\"],[\"app68\",\"hoge\"],[\"app69\",\"hoge\"],\
            [\"app70\",\"hoge\"],[\"app71\",\"hoge\"],[\"app72\",\"hoge\"],[\"app73\",\"hoge\"],[\"app74\",\"hoge\"],[\"app75\",\"hoge\"],[\"app76\",\"hoge\"],[\"app77\",\"hoge\"],[\"app78\",\"hoge\"],[\"app79\",\"hoge\"],\
            [\"app80\",\"hoge\"],[\"app81\",\"hoge\"],[\"app82\",\"hoge\"],[\"app83\",\"hoge\"],[\"app84\",\"hoge\"],[\"app85\",\"hoge\"],[\"app86\",\"hoge\"],[\"app87\",\"hoge\"],[\"app88\",\"hoge\"],[\"app89\",\"hoge\"],\
            [\"app90\",\"hoge\"],[\"app91\",\"hoge\"],[\"app92\",\"hoge\"],[\"app93\",\"hoge\"],[\"app94\",\"hoge\"],[\"app95\",\"hoge\"],[\"app96\",\"hoge\"],[\"app97\",\"hoge\"],[\"app98\",\"hoge\"],[\"app99\",\"hoge\"],\
            [\"app100\",\"hoge\"],[\"app101\",\"hoge\"],[\"app102\",\"hoge\"],[\"app103\",\"hoge\"],[\"app104\",\"hoge\"],[\"app105\",\"hoge\"],[\"app106\",\"hoge\"],[\"app107\",\"hoge\"],[\"app108\",\"hoge\"],[\"app109\",\"hoge\"],\
            [\"app110\",\"hoge\"],[\"app111\",\"hoge\"],[\"app112\",\"hoge\"],[\"app113\",\"hoge\"],[\"app114\",\"hoge\"],[\"app115\",\"hoge\"],[\"app116\",\"hoge\"],[\"app117\",\"hoge\"],[\"app118\",\"hoge\"],[\"app119\",\"hoge\"],\
            [\"app120\",\"hoge\"],[\"app121\",\"hoge\"],[\"app122\",\"hoge\"],[\"app123\",\"hoge\"],[\"app124\",\"hoge\"],[\"app125\",\"hoge\"],[\"app126\",\"hoge\"],[\"app127\",\"hoge\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);
    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);
    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    for(int i = 1 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    for(int i = 0 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {

        char buffer[256] = {};
        nn::util::TSNPrintf(buffer, sizeof(buffer), "app%d", i);
        EXPECT_STREQ(buffer, g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("hoge", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData6)
{
    // NxAccount 以外の NAS サービスが入っている

    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"NewService1\":{\
                \"ClientId\":987654321,\
                \"RedirectUri\":\"new_service_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);

    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NewService1", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceName.value);
    EXPECT_EQ(987654321ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId);
    EXPECT_STREQ("new_service_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("hoge_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    EXPECT_STREQ("app1_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("app1_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("app2_fqdn_lp1", g_SaveData.applicationSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("app2_fqdn_dd1", g_SaveData.applicationSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData7)
{
    // ApplicationSettings/FqdnEntries 空

    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"NewService1\":{\
                \"ClientId\":987654321,\
                \"RedirectUri\":\"new_service_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);

    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("NewService1", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceName.value);
    EXPECT_EQ(987654321ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId);
    EXPECT_STREQ("new_service_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("nx_baas_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("nx_baas_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    EXPECT_STREQ("hoge_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[1].src.value);
    EXPECT_STREQ("hoge_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[1].dst.value);
    for(int i = 2 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    for(int i = 0 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }
}

TEST(NsdUnitTest_JwtClaimSetParser, CorrectData8)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service1\":{\
                \"ClientId\":111111111,\
                \"RedirectUri\":\"service1.com\"\
            },\
            \"Service2\":{\
                \"ClientId\":222222222,\
                \"RedirectUri\":\"service2.com\"\
            },\
            \"Service3\":{\
                \"ClientId\":333333333,\
                \"RedirectUri\":\"service3.com\"\
            },\
            \"Service4\":{\
                \"ClientId\":444444444,\
                \"RedirectUri\":\"service4.com\"\
            },\
            \"Service5\":{\
                \"ClientId\":555555555,\
                \"RedirectUri\":\"service5.com\"\
            },\
            \"Service6\":{\
                \"ClientId\":666666666,\
                \"RedirectUri\":\"service6.com\"\
            },\
            \"Service7\":{\
                \"ClientId\":777777777,\
                \"RedirectUri\":\"service7.com\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"hoge1_fqdn_lp1\",\"hoge1_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_ASSERT_RESULT_SUCCESS( Parse(&stream, &handler) );

    EXPECT_STREQ("setting name test.", g_SaveData.settingName.value);
    EXPECT_STREQ("dd1", g_SaveData.environmentIdentifier.value);
    EXPECT_STREQ("0123456789abcde", g_SaveData.deviceId.value);
    nn::nsd::DeviceId expectDeviceId = {{"0123456789abcde"}};
    EXPECT_EQ(expectDeviceId, g_SaveData.deviceId);
    EXPECT_EQ(1, g_SaveData.version);

    EXPECT_STREQ("nas.request.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasRequestServiceFqdn.value);
    EXPECT_STREQ("nas.api.fqdn", g_SaveData.backboneSettings.nasServiceSettings.nasApiServiceFqdn.value);

    EXPECT_STREQ("NxAccount", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceName.value);
    EXPECT_EQ(123456789ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.clientId);
    EXPECT_STREQ("nx_account_redirect_uri", g_SaveData.backboneSettings.nasServiceSettings.entries[0].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service1", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceName.value);
    EXPECT_EQ(111111111ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.clientId);
    EXPECT_STREQ("service1.com", g_SaveData.backboneSettings.nasServiceSettings.entries[1].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service2", g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceName.value);
    EXPECT_EQ(222222222ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.clientId);
    EXPECT_STREQ("service2.com", g_SaveData.backboneSettings.nasServiceSettings.entries[2].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service3", g_SaveData.backboneSettings.nasServiceSettings.entries[3].nasServiceName.value);
    EXPECT_EQ(333333333ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[3].nasServiceSetting.clientId);
    EXPECT_STREQ("service3.com", g_SaveData.backboneSettings.nasServiceSettings.entries[3].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service4", g_SaveData.backboneSettings.nasServiceSettings.entries[4].nasServiceName.value);
    EXPECT_EQ(444444444ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[4].nasServiceSetting.clientId);
    EXPECT_STREQ("service4.com", g_SaveData.backboneSettings.nasServiceSettings.entries[4].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service5", g_SaveData.backboneSettings.nasServiceSettings.entries[5].nasServiceName.value);
    EXPECT_EQ(555555555ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[5].nasServiceSetting.clientId);
    EXPECT_STREQ("service5.com", g_SaveData.backboneSettings.nasServiceSettings.entries[5].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service6", g_SaveData.backboneSettings.nasServiceSettings.entries[6].nasServiceName.value);
    EXPECT_EQ(666666666ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[6].nasServiceSetting.clientId);
    EXPECT_STREQ("service6.com", g_SaveData.backboneSettings.nasServiceSettings.entries[6].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("Service7", g_SaveData.backboneSettings.nasServiceSettings.entries[7].nasServiceName.value);
    EXPECT_EQ(777777777ULL, g_SaveData.backboneSettings.nasServiceSettings.entries[7].nasServiceSetting.clientId);
    EXPECT_STREQ("service7.com", g_SaveData.backboneSettings.nasServiceSettings.entries[7].nasServiceSetting.redirectUri.value);

    EXPECT_STREQ("hoge1_fqdn_lp1", g_SaveData.backboneSettings.fqdnEntries[0].src.value);
    EXPECT_STREQ("hoge1_fqdn_dd1", g_SaveData.backboneSettings.fqdnEntries[0].dst.value);
    for(int i = 1 ; i < g_SaveData.backboneSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.backboneSettings.fqdnEntries[i].dst.value);
    }

    EXPECT_EQ(2871764000LL, g_SaveData.expireUnixTime);
    for(int i = 0 ; i < g_SaveData.applicationSettings.FqdnEntryCountMax ; ++i)
    {
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].src.value);
        EXPECT_STREQ("", g_SaveData.applicationSettings.fqdnEntries[i].dst.value);
    }
}


TEST(NsdUnitTest_JwtClaimSetParser, ResultMultipleItem1)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd3\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"NxAccount\":{\
                \"ClientId\":987654321,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultMultipleItem , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound1)
{
    const char* json = "{\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound2)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound3)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound4)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"Expire\":0,\
    \"DeviceId\": \"0123456789abcde\"}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound5)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound6)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound7)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\"\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound8)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound9)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound10)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\"\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultRequiredItemNotFound11)
{
    const char* json = "{\
    \"SettingName\": \"test test test!\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":0,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough1)
{
    const char* json = "{\
    \"SettingName\": \"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough2)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough3)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough4)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough5)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough6)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough7)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1_long\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough8)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcdef\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough10)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemLengthBufferNotEnough11)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service1_aaaaaaa\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultStringTooLong, Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemOverCapacity1)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"hoge1_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge2_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge3_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge4_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge5_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge6_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge7_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge8_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge9_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge10_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge11_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge12_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge13_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge14_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge15_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge16_fqdn_lp1\",\"hoge_fqdn_dd1\"],\
            [\"hoge17_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultFqdnEntryOverCapacity , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemOverCapacity2)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"hoge1_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app0\",\"hoge\"],[\"app1\",\"hoge\"],[\"app2\",\"hoge\"],[\"app3\",\"hoge\"],[\"app4\",\"hoge\"],[\"app5\",\"hoge\"],[\"app6\",\"hoge\"],[\"app7\",\"hoge\"],[\"app8\",\"hoge\"],[\"app9\",\"hoge\"],\
            [\"app10\",\"hoge\"],[\"app11\",\"hoge\"],[\"app12\",\"hoge\"],[\"app13\",\"hoge\"],[\"app14\",\"hoge\"],[\"app15\",\"hoge\"],[\"app16\",\"hoge\"],[\"app17\",\"hoge\"],[\"app18\",\"hoge\"],[\"app19\",\"hoge\"],\
            [\"app20\",\"hoge\"],[\"app21\",\"hoge\"],[\"app22\",\"hoge\"],[\"app23\",\"hoge\"],[\"app24\",\"hoge\"],[\"app25\",\"hoge\"],[\"app26\",\"hoge\"],[\"app27\",\"hoge\"],[\"app28\",\"hoge\"],[\"app29\",\"hoge\"],\
            [\"app30\",\"hoge\"],[\"app31\",\"hoge\"],[\"app32\",\"hoge\"],[\"app33\",\"hoge\"],[\"app34\",\"hoge\"],[\"app35\",\"hoge\"],[\"app36\",\"hoge\"],[\"app37\",\"hoge\"],[\"app38\",\"hoge\"],[\"app39\",\"hoge\"],\
            [\"app40\",\"hoge\"],[\"app41\",\"hoge\"],[\"app42\",\"hoge\"],[\"app43\",\"hoge\"],[\"app44\",\"hoge\"],[\"app45\",\"hoge\"],[\"app46\",\"hoge\"],[\"app47\",\"hoge\"],[\"app48\",\"hoge\"],[\"app49\",\"hoge\"],\
            [\"app50\",\"hoge\"],[\"app51\",\"hoge\"],[\"app52\",\"hoge\"],[\"app53\",\"hoge\"],[\"app54\",\"hoge\"],[\"app55\",\"hoge\"],[\"app56\",\"hoge\"],[\"app57\",\"hoge\"],[\"app58\",\"hoge\"],[\"app59\",\"hoge\"],\
            [\"app60\",\"hoge\"],[\"app61\",\"hoge\"],[\"app62\",\"hoge\"],[\"app63\",\"hoge\"],[\"app64\",\"hoge\"],[\"app65\",\"hoge\"],[\"app66\",\"hoge\"],[\"app67\",\"hoge\"],[\"app68\",\"hoge\"],[\"app69\",\"hoge\"],\
            [\"app70\",\"hoge\"],[\"app71\",\"hoge\"],[\"app72\",\"hoge\"],[\"app73\",\"hoge\"],[\"app74\",\"hoge\"],[\"app75\",\"hoge\"],[\"app76\",\"hoge\"],[\"app77\",\"hoge\"],[\"app78\",\"hoge\"],[\"app79\",\"hoge\"],\
            [\"app80\",\"hoge\"],[\"app81\",\"hoge\"],[\"app82\",\"hoge\"],[\"app83\",\"hoge\"],[\"app84\",\"hoge\"],[\"app85\",\"hoge\"],[\"app86\",\"hoge\"],[\"app87\",\"hoge\"],[\"app88\",\"hoge\"],[\"app89\",\"hoge\"],\
            [\"app90\",\"hoge\"],[\"app91\",\"hoge\"],[\"app92\",\"hoge\"],[\"app93\",\"hoge\"],[\"app94\",\"hoge\"],[\"app95\",\"hoge\"],[\"app96\",\"hoge\"],[\"app97\",\"hoge\"],[\"app98\",\"hoge\"],[\"app99\",\"hoge\"],\
            [\"app100\",\"hoge\"],[\"app101\",\"hoge\"],[\"app102\",\"hoge\"],[\"app103\",\"hoge\"],[\"app104\",\"hoge\"],[\"app105\",\"hoge\"],[\"app106\",\"hoge\"],[\"app107\",\"hoge\"],[\"app108\",\"hoge\"],[\"app109\",\"hoge\"],\
            [\"app110\",\"hoge\"],[\"app111\",\"hoge\"],[\"app112\",\"hoge\"],[\"app113\",\"hoge\"],[\"app114\",\"hoge\"],[\"app115\",\"hoge\"],[\"app116\",\"hoge\"],[\"app117\",\"hoge\"],[\"app118\",\"hoge\"],[\"app119\",\"hoge\"],\
            [\"app120\",\"hoge\"],[\"app121\",\"hoge\"],[\"app122\",\"hoge\"],[\"app123\",\"hoge\"],[\"app124\",\"hoge\"],[\"app125\",\"hoge\"],[\"app126\",\"hoge\"],[\"app127\",\"hoge\"],[\"app128\",\"hoge\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultFqdnEntryOverCapacity , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultSettingItemOverCapacity3)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Servcie1\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Servcie2\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service3\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service4\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service5\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service6\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service7\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            },\
            \"Service8\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"hoge1_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultBackbonesettingsNasServiceOverCapacity, Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultUnexpectedValue1)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"UnexpectedItem\": \"hoge\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultUnexpectedJsonItem , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultUnexpectedValue3)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"UnexpectedItem\": \"hoge\",\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultUnexpectedJsonItem , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultUnexpectedValue4)
{
    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultInvalidFqdnEntryFormat, Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultUnexpectedValue5)
{

    // Expire:9223372036854775808; -> std::numeric_limits<int64_t>::max() + 1

    const char* json = "{\
    \"SettingName\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"Expire\":9223372036854775808,\
    \"DeviceId\": \"0123456789abcde\",\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultOutOfRangeValue , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ResultUnexpectedValue6)
{
    const char* json = "{\
    \"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\": \"setting name test.\",\
    \"Environment\": \"dd1\",\
    \"BackboneSettings\": {\
        \"Nas\":{\
            \"RequestFqdn\":\"nas.request.fqdn\",\
            \"ApiFqdn\":\"nas.api.fqdn\",\
            \"NxAccount\":{\
                \"ClientId\":123456789,\
                \"RedirectUri\":\"nx_account_redirect_uri\"\
            }\
        },\
        \"FqdnEntries\":[\
            [\"nx_baas_fqdn_lp1\",\"nx_baas_fqdn_dd1\"],\
            [\"hoge_fqdn_lp1\",\"hoge_fqdn_dd1\"]\
        ]\
    },\
    \"ApplicationSettings\":{\
        \"FqdnEntries\":[\
            [\"app1_fqdn_lp1\",\"app1_fqdn_dd1\"],\
            [\"app2_fqdn_lp1\",\"app2_fqdn_dd1\"]\
            ]\
    },\
    \"DeviceId\": \"0123456789abcde\",\
    \"Expire\":2871764000,\
    \"Version\":1}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultJsonParsingTerminated , Parse(&stream, &handler) );
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0000)
{
    const char* json = "{\
    \"errors\" : [\
        {\
            \"code\" : \"0000\",\
            \"message\" : \"Missing parameter in request\"\
        }\
    ]\
    }";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0000", handler.GetServerErrorCode());
    EXPECT_STREQ("Missing parameter in request", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0001)
{
    const char* json = "{\
    \"errors\" : [\
        {\
            \"code\" : \"0001\",\
            \"message\" : \"Invalid parameter in request\"\
        }\
    ]\
    }";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0001", handler.GetServerErrorCode());
    EXPECT_STREQ("Invalid parameter in request", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0002)
{
    const char* json = "{\"errors\" : [{\"code\" : \"0002\",\"message\" : \"Passcode not found\"}]}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0002", handler.GetServerErrorCode());
    EXPECT_STREQ("Passcode not found", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0003)
{
    const char* json = "{\
    \"errors\" : [\
        {\
            \"code\" : \"0003\",\
            \"message\" : \"Forbidden\"\
        }\
    ]\
    }";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0003", handler.GetServerErrorCode());
    EXPECT_STREQ("Forbidden", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0004)
{
    const char* json = "{\"errors\" : [{\"code\" : \"0004\",\"message\" : \"Method not allow\"}]}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0004", handler.GetServerErrorCode());
    EXPECT_STREQ("Method not allow", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0005)
{
    const char* json = "{\"errors\" : [{\"code\" : \"0005\",\"message\" : \"API version not found\"}]}";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0005", handler.GetServerErrorCode());
    EXPECT_STREQ("API version not found", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0500)
{
    const char* json = "{\
    \"errors\" : [\
        {\
            \"code\" : \"0500\",\
            \"message\" : \"Internal Server Error\"\
        }\
    ]\
    }";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0500", handler.GetServerErrorCode());
    EXPECT_STREQ("Internal Server Error", handler.GetServerErrorMessage());
}

TEST(NsdUnitTest_JwtClaimSetParser, ServerErrorResponse0503)
{
    const char* json = "{\
    \"errors\" : [\
        {\
            \"code\" : \"0503\",\
            \"message\" : \"Under Maintenance\"\
        }\
    ]\
    }";

    RapidJsonEventAccepterForClaims handler(&g_SaveData);
    Stream<256u, 256u> stream(json, strlen(json));
    NNT_EXPECT_RESULT_SUCCESS(Parse(&stream, &handler) );
    EXPECT_TRUE(handler.HasServerError());
    EXPECT_STREQ("0503", handler.GetServerErrorCode());
    EXPECT_STREQ("Under Maintenance", handler.GetServerErrorMessage());
}
