﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>

namespace {

    class SystemUpdateTestWithoutNetwork : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
    private:
        NN_ALIGNAS(4096) static uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

    protected:
        virtual void SetUp()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::ncm::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::nim::InitializeForNetworkInstallManager();
#endif
        }

        virtual void TearDown()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::nim::FinalizeForNetworkInstallManager();
#endif
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NNT_EXPECT_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                             nn::socket::DefaultSocketMemoryPoolSize,
                                                             nn::socket::DefaultSocketAllocatorSize,
                                                             nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
            nn::fs::UnmountHostRoot();
#endif
        }

        std::vector<nn::ncm::ContentMetaKey> GetSystemContentMetaList()
        {
            nn::ncm::ContentMetaKey array[] = {
                nn::ncm::ContentMetaKey::Make(0x0000000000000004, 0, nn::ncm::ContentMetaType::SystemUpdate),
                nn::ncm::ContentMetaKey::Make(0x0000000000000005, 0, nn::ncm::ContentMetaType::SystemData),
                nn::ncm::ContentMetaKey::Make(0x0000000000000006, 0, nn::ncm::ContentMetaType::SystemData)
            };

            return std::vector<nn::ncm::ContentMetaKey>(array, std::end(array));
        }
    };

#if defined( NN_BUILD_CONFIG_OS_WIN )
    NN_ALIGNAS(4096) uint8_t SystemUpdateTestWithoutNetwork::s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize] = {0};
#endif
}

TEST_F(SystemUpdateTestWithoutNetwork, SystemUpdateWithoutNetwork)
{
    auto list = GetSystemContentMetaList();
    for (auto& key : list)
    {
        DeleteAllContents(key.id, nn::ncm::StorageId::BuildInSystem);
    }

    nn::ns::Initialize();

    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());
        EXPECT_EQ( control.HasDownloaded(), false );
        auto progress = control.GetDownloadProgress();
        EXPECT_EQ( progress.loaded, 0 );
        EXPECT_EQ( progress.total, 0 );
    }

    nn::ns::Finalize();
} // NOLINT(impl/function_size)
