﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nnt/nntest.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_MaxCount.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>
#include "libraries/testNs_CommandOptions.h"


#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {
    // Case Setup
    class NupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::ns::Initialize();
        }

        virtual void TearDown()
        {
            nn::ns::Finalize();
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
            // .ymlで指定した option を格納している
            testns::InitializeCommandLineOptions();
        }

        static void TearDownTestCase()
        {
            testns::FinalizeCommandLineOptions();
        }
    };

    // AsyncUtils
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef TControl    ControlType;
        typedef THandle     HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        nn::Result WaitResponse() NN_NOEXCEPT
        {
            const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            auto result = OnWaitResponse( m_Handle );
            m_ElapsedTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT
            : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( HandleType& handle ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    // CheckLatestUpdate
    class NewlyUpdateInquirer : public RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate >
    {
    public:
        static const nn::ns::LatestSystemUpdate INVALID_LATEST_VALUE = static_cast< nn::ns::LatestSystemUpdate >( -1 );
        typedef RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate > BaseType;

        explicit NewlyUpdateInquirer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control ), m_LatestStatus( INVALID_LATEST_VALUE )
        {
        }

        const nn::ns::LatestSystemUpdate& GetLatestStatus() const NN_NOEXCEPT
        {
            return m_LatestStatus;
        }

        const char* GetLatestStatusString() const NN_NOEXCEPT
        {
            switch ( m_LatestStatus )
            {
            case nn::ns::LatestSystemUpdate::Downloaded:    return "Downloaded";
            case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
            case nn::ns::LatestSystemUpdate::UpToDate:      return "UpToDate";
            default:                                        return "Unknown";
            }
        }

    protected:
        virtual nn::Result OnWaitResponse( HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                LOG_OUT( "Wating check latest update...\n" );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            return handle.Get( &m_LatestStatus );
        }

    private:
        nn::ns::LatestSystemUpdate m_LatestStatus;
    };


    // CheckProgramList
    class ProgramList
    {
    public:
        static bool FindId( long inspectSystemUpdateId, long inspectSystemDataId, unsigned int inspectVersion )
        {
            // SystemUpdate と SystemData が2つとも存在したかチェックする用
            int idMutchCount = 0;

            nn::ncm::ContentMetaDatabase db;
            nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem );

            std::vector<nn::ncm::ContentMetaKey> list;
            list.resize(nn::ncm::SystemMaxContentMetaCount);
            auto listCount = db.ListContentMeta(list.data(), static_cast<int>(list.size()), nn::ncm::ContentMetaType::Unknown);
            list.resize(static_cast<size_t>(listCount.listed));
            if (list.size() > 0)
            {
                LOG_OUT("%d system content meta(s) found.\n", list.size());
                LOG_OUT("id                        ver.  type\n");
                //      0x0123456789abcdef  4294967295  SystemProgram
                for (auto& meta : list)
                {
                    LOG_OUT("0x%016llx  %10u  %d  \n", meta.id, meta.version, meta.type);
                    // systemUpdate のチェック
                    if ((meta.id == inspectSystemUpdateId )
                     && (meta.version == inspectVersion   )
                     && (meta.type == nn::ncm::ContentMetaType::SystemUpdate ))
                    {
                        LOG_OUT("match SystemUpdate\n");
                        idMutchCount++;
                    }

                    // systemData のチェック
                    if ((meta.id == inspectSystemDataId )
                     && (meta.version == inspectVersion )
                     && (meta.type == nn::ncm::ContentMetaType::SystemData ))
                    {
                        LOG_OUT("match SystemData\n");
                        idMutchCount++;
                    }
                }
            }
            // systemUpdateId と systemDataId の 2つが存在すれば true
            return ( idMutchCount == 2 );
        }

        // SystemUpdateId の存在確認をタイムアウトするまでチェックする
        static bool FindIdTimeOutCheck( long inspectSystemUpdateId, long inspectSystemDataId, unsigned int inspectVersion, int timeoutSeconds )
        {
            const int sleepTime = 15; // IDチェック後 15秒待つ
            int sumTime = 0;
            bool discoveredSystemUpdate = false;
            while ( !discoveredSystemUpdate && ( timeoutSeconds > sumTime) )
            {
                discoveredSystemUpdate = FindId( inspectSystemUpdateId, inspectSystemDataId, inspectVersion );
                if ( !discoveredSystemUpdate )
                {
                    // 少し待つ
                    nn::os::SleepThread( nn::TimeSpan::FromSeconds( sleepTime ) );
                    sumTime += sleepTime;
                }
            }
            return discoveredSystemUpdate;
        }
    };

}

TEST_F( NupCase, RestartNotRequiredNup )
{
    char pCaseValue[ 32 ] = "";
    char pSystemUpdateId[ 32 ] = "";
    char pSystemDataId[ 32 ] = "";
    char pLatestKind[ 32 ] = "";
    char pVersion[ 16 ] = "";
    char pBackgroundState[ 16 ] = "";

    long inspectSystemUpdateId = 0;
    long inspectSystemDataId = 0;
    unsigned int inspectVersion = 0;

    // オプションを取得
    ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_BgnupCase", sizeof( pCaseValue ), pCaseValue ) );
    LOG_OUT( "==============================================\n"  );
    LOG_OUT( "Specified the test case [ %s ].\n", pCaseValue  );
    LOG_OUT( "==============================================\n\n"  );

    if ( 0 == std::strcmp( pCaseValue, "programList" ) )
    {
        ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_systemUpdateID", sizeof( pSystemUpdateId ), pSystemUpdateId ) );
        ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_systemDataID", sizeof( pSystemDataId ), pSystemDataId ) );
        ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_Version", sizeof( pVersion ), pVersion ) );

        LOG_OUT( "==============================================\n"  );
        LOG_OUT( "Specified the SystemUpdateId [ %s ].\n", pSystemUpdateId  );
        LOG_OUT( "Specified the SystemDataId [ %s ].\n", pSystemDataId  );
        LOG_OUT( "Specified the Version [ %s ].\n", pVersion  );
        LOG_OUT( "==============================================\n\n"  );

        inspectSystemUpdateId = strtol(pSystemUpdateId, NULL, 16);
        inspectSystemDataId = strtol( pSystemDataId, NULL, 16 );
        inspectVersion = atoi(pVersion);

        // systemprogram listを実行して、SystemUpdateId と SystemDataId と version が一致することを確認。
        LOG_OUT( "SystemUpdateId check after request-bgnup.\n" );
        int timeoutSeconds = 10 * 60;
        ASSERT_EQ( true, ProgramList::FindIdTimeOutCheck( inspectSystemUpdateId, inspectSystemDataId, inspectVersion, timeoutSeconds ));

    }
    else if ( 0 == std::strcmp( pCaseValue, "check-latest" ) )
    {
        // check-latest の値を確認。

        ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_LatestKind", sizeof( pLatestKind ), pLatestKind ) );

        LOG_OUT( "==============================================\n"  );
        LOG_OUT( "Specified the LatestKind [ %s ].\n", pLatestKind  );
        LOG_OUT( "==============================================\n\n"  );

        // ネットワーク接続確立.
        LOG_OUT( "Request network connection ...\n" );
        nn::nifm::SubmitNetworkRequestAndWait();

        // 本体更新制御の占有開始.
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

        // 最新の更新の検出. RequestCheckLatestUpdate
        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        LOG_OUT( "Start latest update checking...\n" );
        NewlyUpdateInquirer inquirer( control );
        NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
        NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
        latest = inquirer.GetLatestStatus();
        // latest を確認
        LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );
        if ( 0 == std::strcmp( pLatestKind, "UpToDate" ) )
        {
            ASSERT_EQ( nn::ns::LatestSystemUpdate::UpToDate, latest );
        }
        else if ( 0 == std::strcmp( pLatestKind, "NeedsDownload" ) )
        {
            ASSERT_EQ( nn::ns::LatestSystemUpdate::NeedsDownload, latest );
        }
        else
        {
            LOG_OUT( "A pattern that does not exist..\n");
            NN_SDK_ASSERT( false );
        }
    }
    else if ( 0 == std::strcmp( pCaseValue, "get-bgstate" ) )
    {
        // get-background-state を実行して、None か InProgress かをチェックする。

        ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_state", sizeof( pBackgroundState ), pBackgroundState ) );

        LOG_OUT( "==============================================\n"  );
        LOG_OUT( "Specified the Background-State [ %s ].\n", pBackgroundState  );
        LOG_OUT( "==============================================\n\n"  );

        const int timeoutSeconds = 60;
        const int sleepTime = 5;
        int sumTime = 0;
        bool checkBackgroundState = false;

        while (sumTime < timeoutSeconds)
        {
            auto state = nn::ns::GetBackgroundNetworkUpdateState();
            std::string stateString;
            switch (state)
            {
            case nn::ns::BackgroundNetworkUpdateState::None: stateString = "None"; break;
            case nn::ns::BackgroundNetworkUpdateState::InProgress: stateString = "InProgress"; break;
            case nn::ns::BackgroundNetworkUpdateState::Ready: stateString = "Ready"; break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            LOG_OUT("%s\n", stateString.c_str());

            if (( (state == nn::ns::BackgroundNetworkUpdateState::None) && ( 0 == std::strcmp( pBackgroundState, "None" ) ) )
              ||( (state == nn::ns::BackgroundNetworkUpdateState::InProgress) && ( 0 == std::strcmp( pBackgroundState, "InProgress" ) ) ) )
            {
                checkBackgroundState = true;
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(sleepTime));
            sumTime += sleepTime;
        }
        ASSERT_EQ( true, checkBackgroundState );
    }

    else
    {
        LOG_OUT( "A pattern that does not exist..\n");
        NN_SDK_ASSERT( false );
    }


} // NOLINT(impl/function_size)
