﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <utility>
#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_CommandOptions.h"
#include "libraries/testNs_InstantMessageThread.h"
#include "testNs_NcmStorageVerify.h"

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( & )[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Result出力.
    //!--------------------------------------------------------------------------------------
    static void LogoutResultValue( const nn::Result& result, const char* const pTitle )
    {
        if ( nullptr != pTitle )
        {
            LOG_OUT( "%s\n", pTitle );
        }
        LOG_OUT( "-------- Result ---------\n" );
        LOG_OUT( "module = %u\n", result.GetModule() );
        LOG_OUT( "description = %u\n", result.GetDescription() );
        LOG_OUT( "inner value = 0x%08x\n", result.GetInnerValueForDebug() );
        LOG_OUT( "-------------------------\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! 並列キャンセラ
    //!--------------------------------------------------------------------------------------
    static const size_t CancellerThreadStackSize = nn::os::ThreadStackAlignment * 4;
    typedef testns::InstantMessenger< CancellerThreadStackSize >::ThreadContext ParallelCanceller;

    //!--------------------------------------------------------------------------------------
    //! fwdbg settings utilities.
    //!--------------------------------------------------------------------------------------
    class FirmwareDebugSettings
    {
    public:
        static const std::pair< const char*, const char* > NetworkConnectionTest;

        //!--------------------------------------------------------------------------------------
        //! @brief 取得
        //!--------------------------------------------------------------------------------------
        template< typename ValueType >
        static const ValueType GetSettings( const std::pair< const char*, const char* >& item, bool* const pOutResult = nullptr ) NN_NOEXCEPT
        {
            ValueType value;
            const auto outSize = nn::settings::fwdbg::GetSettingsItemValue( &value, sizeof( value ), item.first, item.second );
            if ( nullptr == pOutResult )
            {
                NN_ASSERT_EQUAL( outSize, sizeof( value ) );
            }
            else
            {
                *pOutResult = ( outSize == sizeof( value ) );
            }
            return value;
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 設定
        //!--------------------------------------------------------------------------------------
        template< typename ValueType >
        static void SetSettings( const std::pair< const char*, const char* >& item, const ValueType& value ) NN_NOEXCEPT
        {
            nn::settings::fwdbg::SetSettingsItemValue( item.first, item.second, &value, sizeof( value ) );
        }

        //!--------------------------------------------------------------------------------------
        //! network connection controller
        //!--------------------------------------------------------------------------------------
        class NetworkConnectionController
        {
        public:
            NetworkConnectionController() NN_NOEXCEPT
                : m_PreSetting( FirmwareDebugSettings::GetSettings< bool >( FirmwareDebugSettings::NetworkConnectionTest ) )
            {
                FirmwareDebugSettings::SetSettings( FirmwareDebugSettings::NetworkConnectionTest, true );
            }

            ~NetworkConnectionController() NN_NOEXCEPT
            {
                NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetWirelessCommunicationEnabledForTest( true ) );
                NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetEthernetCommunicationEnabledForTest( true ) );
                FirmwareDebugSettings::SetSettings( FirmwareDebugSettings::NetworkConnectionTest, m_PreSetting );
            }

            void ChangeConnection( const bool isEnabled ) NN_NOEXCEPT
            {
                NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetWirelessCommunicationEnabledForTest( isEnabled ) );
                NNT_EXPECT_RESULT_SUCCESS( nn::nifm::SetEthernetCommunicationEnabledForTest( isEnabled ) );
            }

        private:
            const bool m_PreSetting;
        };
    };
    const std::pair< const char*, const char* > FirmwareDebugSettings::NetworkConnectionTest =
        std::pair< const char*, const char* >( "nifm", "is_communication_control_enabled_for_test" );

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class NupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();

            auto& cancellerInstance = nn::util::Get( m_ParallelCanceller );
            new ( &cancellerInstance ) ParallelCanceller();
            cancellerInstance.Initialize();
        }

        virtual void TearDown()
        {
            nn::util::Get( m_ParallelCanceller ).Finalize();
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
            testns::InitializeCommandLineOptions();
        }

        static void TearDownTestCase()
        {
            testns::FinalizeCommandLineOptions();
            testns::UnmountHostFileSystem();
        }

    protected:
        nn::util::TypedStorage< ParallelCanceller, sizeof( ParallelCanceller ), NN_ALIGNOF( ParallelCanceller ) > m_ParallelCanceller;
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;

    public:
        NcmStorageVerifier& GetNcmStorageVerifier() NN_NOEXCEPT
        {
            return nn::util::Get( m_NcmStorageVerifier );
        }

        ParallelCanceller& GetParallelCanceller() NN_NOEXCEPT
        {
            return nn::util::Get( m_ParallelCanceller );
        }

        void DoDownloadAction( std::function< bool ( NupCase& context, nn::ns::SystemUpdateControl& control ) > doRunnable ) NN_NOEXCEPT;
    };



    //!--------------------------------------------------------------------------------------
    //! RequestSynchronizer
    //!--------------------------------------------------------------------------------------
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef ParallelCanceller                               CancellerType;
        typedef TControl                                        ControlType;
        typedef THandle                                         HandleType;
        typedef RequestSynchronizer< HandleType, ControlType >  MyType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse() NN_NOEXCEPT
        {
            const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            auto result = OnWaitResponse( m_Control, m_Handle );
            m_ElapsedTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

        void CancelThere( const int64_t expectedElapsedMillis ) NN_NOEXCEPT
        {
            CancellerType* pCanceller;
            if ( nullptr != ( pCanceller = m_pCanceller ) )
            {
                m_OverTimeForCancel = nn::os::GetSystemTick() + nn::os::ConvertToTick( nn::TimeSpan( nn::TimeSpanType::FromMilliSeconds( expectedElapsedMillis ) ) );
                pCanceller->Start( []( testns::MessageThreadContext* pThread, const testns::ThreadMessage* pMessage )->void
                {
                    auto* const pThis = reinterpret_cast< MyType* >( pMessage->GetUserContext() );
                    const nn::os::Tick diff = pThis->m_OverTimeForCancel - nn::os::GetSystemTick();
                    if ( diff.GetInt64Value() > 0 )
                    {
                        const nn::TimeSpan sleeping = nn::os::ConvertToTimeSpan( diff );
                        LOG_OUT( "Sleep at [ %lld ] ms.\n", sleeping.GetMilliSeconds() );
                        nn::os::SleepThread( sleeping );
                    }
                    pThis->CallParallelCance();
                }, this );
            }
        }

    protected:
        explicit RequestSynchronizer( ControlType& control, CancellerType* pCanceller = nullptr ) NN_NOEXCEPT
            : m_Control( control ), m_pCanceller( pCanceller )
        {
        }

        ~RequestSynchronizer() NN_NOEXCEPT
        {
            CancellerType* pCanceller;
            if ( nullptr != ( pCanceller = m_pCanceller ) )
            {
                m_pCanceller = nullptr;
                pCanceller->CancelAndWait();
            }
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT = 0;

        virtual void OnRequestParallelCance( ControlType& control, HandleType& handle ) NN_NOEXCEPT
        {
            NN_UNUSED( control );
            NN_UNUSED( handle );
        }

        void CallParallelCance() NN_NOEXCEPT
        {
            OnRequestParallelCance( m_Control, m_Handle );
        }

    private:
        ControlType&    m_Control;
        CancellerType*  m_pCanceller;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
        nn::os::Tick    m_OverTimeForCancel;
    };

    //!--------------------------------------------------------------------------------------
    //! CheckLatestUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInquirer : public RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate >
    {
    public:
        static const nn::ns::LatestSystemUpdate INVALID_LATEST_VALUE = static_cast< nn::ns::LatestSystemUpdate >( -1 );
        typedef RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate > BaseType;

        explicit NewlyUpdateInquirer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control ), m_LatestStatus( INVALID_LATEST_VALUE )
        {
        }

        const nn::ns::LatestSystemUpdate& GetLatestStatus() const NN_NOEXCEPT
        {
            return m_LatestStatus;
        }

        const char* GetLatestStatusString() const NN_NOEXCEPT
        {
            switch ( m_LatestStatus )
            {
            case nn::ns::LatestSystemUpdate::Downloaded:    return "Downloaded";
            case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
            case nn::ns::LatestSystemUpdate::UpToDate:      return "UpToDate";
            default:                                        return "Unknown";
            }
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                LOG_OUT( "Wating check latest update...\n" );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            return handle.Get( &m_LatestStatus );
        }

    private:
        nn::ns::LatestSystemUpdate m_LatestStatus;
    };

    //!--------------------------------------------------------------------------------------
    //! ApplyDownloadedUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            control.ApplyDownloadedUpdate();
            NN_RESULT_SUCCESS;
        }
    };

    //!--------------------------------------------------------------------------------------
    //! Cancel Configuration
    //!--------------------------------------------------------------------------------------
    template< typename KindCategoryEnumrate, typename KindElapsedContainer = uint32_t >
    class CancelConfiguration
    {
    public:
        typedef KindCategoryEnumrate    CategoryEnumrateType;   // for enumrate.
        typedef KindElapsedContainer    ElapsedContainerType;   // for millisecond span.

        explicit CancelConfiguration( const CategoryEnumrateType& category, const ElapsedContainerType& elapsedTime ) NN_NOEXCEPT
            : m_ElapsedTime( elapsedTime ),
            m_Category( category )
        {
        }

        const CategoryEnumrateType& GetCategory() const NN_NOEXCEPT
        {
            return m_Category;
        }

        const ElapsedContainerType& GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

    private:
        const ElapsedContainerType  m_ElapsedTime;
        const CategoryEnumrateType  m_Category;
    };

    //!--------------------------------------------------------------------------------------
    //! DownloadLatestUpdate ( cancel by async handle )
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateDownloader : public RequestSynchronizer< nn::ns::AsyncResult >
    {
    public:
        enum class CancelCategory : nn::Bit8
        {
            None,
            InDownloadPrepared,
            InDownloading,
        };
        typedef CancelConfiguration< CancelCategory > CancelConfigurationType;
        typedef RequestSynchronizer< nn::ns::AsyncResult > BaseType;
        typedef nn::ns::SystemUpdateProgress ProgressValueType;

        explicit NewlyUpdateDownloader( ControlType& control, CancellerType* pCanceller = nullptr,
            const CancelConfigurationType& cancelConfig = CancelConfigurationType( CancelCategory::None, 0 ) ) NN_NOEXCEPT
            : BaseType( control, pCanceller ), cancelConfiguration( cancelConfig )
        {
        }

        static const char* GetBackStateString( const nn::ns::BackgroundNetworkUpdateState state ) NN_NOEXCEPT
        {
            switch ( state )
            {
            case nn::ns::BackgroundNetworkUpdateState::None:        return "None";
            case nn::ns::BackgroundNetworkUpdateState::InProgress:  return "InProgress";
            case nn::ns::BackgroundNetworkUpdateState::Ready:       return "Ready";
            default:                                                return "Unknown";
            }
        }

        static const char* GetBackStateString() NN_NOEXCEPT
        {
            return GetBackStateString( nn::ns::GetBackgroundNetworkUpdateState() );
        }

        inline const ProgressValueType& GetCancelledProgressValue() const NN_NOEXCEPT
        {
            return cancelledProgressValue;
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            bool cancelled = false;
            while ( !handle.TryWait() )
            {
                const auto progress = control.GetDownloadProgress();
                LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );

                if ( false == cancelled )
                {
                    auto& config = cancelConfiguration;
                    switch ( config.GetCategory() )
                    {
                    case CancelCategory::InDownloadPrepared:
                        if ( 0 == progress.total )
                        {
                            cancelledProgressValue = progress;
                            CancelThere( static_cast< int64_t >( config.GetElapsedTime() ) );
                            cancelled = true;
                        }
                        break;
                    case CancelCategory::InDownloading:
                        if ( 0 < progress.total )
                        {
                            cancelledProgressValue = progress;
                            CancelThere( static_cast< int64_t >( config.GetElapsedTime() ) );
                            cancelled = true;
                        }
                        break;
                    default:
                        cancelled = true;
                        break;
                    }
                }
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            const auto progress = control.GetDownloadProgress();
            LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
            return handle.Get();
        }

        virtual void OnRequestParallelCance( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( control );
            LOG_OUT( "Cancel download request detected.\n" );
            handle.Cancel();
        }

    protected:
        const CancelConfigurationType cancelConfiguration;
        ProgressValueType cancelledProgressValue;
    };

    //!--------------------------------------------------------------------------------------
    //! DownloadLatestUpdate ( cancel by async handle, for SIGLO-50315 )
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateDownloaderUntilMetaPrepared : public NewlyUpdateDownloader
    {
    public:
        explicit NewlyUpdateDownloaderUntilMetaPrepared( ControlType& control, CancellerType* pCanceller = nullptr,
            const ProgressValueType& cancelTarget = ProgressValueType() ) NN_NOEXCEPT
            : NewlyUpdateDownloader( control, pCanceller, CancelConfigurationType( CancelCategory::None, 0 ) ),
            cancelTargetProgress( cancelTarget )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            const auto target = cancelTargetProgress;
            bool cancelled = false;
            while ( !handle.TryWait() )
            {
                if ( false == cancelled )
                {
                    const auto progress = control.GetDownloadProgress();
                    if ( progress.total > 0 || ( target.total == progress.total && target.loaded != progress.loaded ) )
                    {
                        cancelledProgressValue = progress;
                        LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
                        CancelThere( 0 );
                        cancelled = true;
                    }
                }
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 50 ) );
            }
            const auto progress = control.GetDownloadProgress();
            LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
            return handle.Get();
        }

    protected:
        const ProgressValueType cancelTargetProgress;
    };

    //!--------------------------------------------------------------------------------------
    //! DownloadLatestUpdate ( abort by network )
    //!--------------------------------------------------------------------------------------
    class DownloaderWithNetworkAbort : public NewlyUpdateDownloader, public FirmwareDebugSettings::NetworkConnectionController
    {
    public:
        explicit DownloaderWithNetworkAbort( ControlType& control, CancellerType* pCanceller = nullptr,
            const CancelConfigurationType& cancelConfig = CancelConfigurationType( CancelCategory::None, 0 ) ) NN_NOEXCEPT
            : NewlyUpdateDownloader( control, pCanceller, cancelConfig )
        {
        }

        virtual void OnRequestParallelCance( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( handle );
            NN_UNUSED( control );
            LOG_OUT( "Abort download request detected.\n" );
            NetworkConnectionController::ChangeConnection( false );
        }
    };



    //!--------------------------------------------------------------------------------------
    //! Network connection request.
    //! 本クライアントアプリケーションがネットワークリソースを占有する事を要求します。
    //!--------------------------------------------------------------------------------------
    static bool SubmitConnectionNetwork() NN_NOEXCEPT
    {
        LOG_OUT( "Request network connection ...\n" );
        const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
        nn::nifm::SubmitNetworkRequestAndWait();
        const nn::TimeSpan spanOnConnection = nn::os::GetSystemTick().ToTimeSpan() - before;
        LOG_OUT( "Response wait : %lld ms.\n", spanOnConnection.GetMilliSeconds() );
        const auto availableNetwork = nn::nifm::IsNetworkAvailable();
        LOG_OUT( "Network connection %s.\n", ( availableNetwork ? "succeeded" : "failed" ) );
        if ( false == availableNetwork )
        {
            LogoutResultValue( nn::nifm::HandleNetworkRequestResult(), "Network connection error." );
        }
        return availableNetwork;
    }

    //!--------------------------------------------------------------------------------------
    //! Cancel connection request.
    //! 本クライアントアプリケーションがネットワークリソースの占有を放棄する事を要求します。
    //!--------------------------------------------------------------------------------------
    static bool ReleaseConnectionNetwork() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();
        const auto availableNetwork = nn::nifm::IsNetworkAvailable();
        LOG_OUT( ">>=== Release network connection ... %s.\n", ( false == availableNetwork ) ? "succeeded" : "failed" );
        return !availableNetwork;
    }

    //!--------------------------------------------------------------------------------------
    //! 期待する利用状態に遷移するまで待機します。
    //!--------------------------------------------------------------------------------------
    static void WaitExpectNetworkAvailable( const bool expectAvailable, unsigned timeOutSeconds = 20 ) NN_NOEXCEPT
    {
        for ( ; expectAvailable != nn::nifm::IsNetworkAvailable() && timeOutSeconds > 0; --timeOutSeconds )
        {
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! Install procedure
    //!--------------------------------------------------------------------------------------
    static void ExecuteInstall( bool& outInstallSuccess, nn::ns::SystemUpdateControl& control, const bool bExpectLimitOver ) NN_NOEXCEPT
    {
        // インストール. ApplyDownloadedUpdate
        LOG_OUT( "Start install downloaded update... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate]\n" );
        NewlyUpdateInstaller installer( control );
        const auto result = installer.WaitResponse();
        outInstallSuccess = result.IsSuccess();
        if ( false == bExpectLimitOver )
        {
            NNT_ASSERT_RESULT_SUCCESS( result );
        }
        else if ( outInstallSuccess )
        {
            outInstallSuccess = false;
            GTEST_FAIL() << "Unexpected success on [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate].";
        }
        else
        {
            outInstallSuccess = true;
            LogoutResultValue( result, "Happened an expected failure on [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate]." );
        }
        LOG_OUT( "Complete install downloaded update.\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! Cancelled download procedure
    //!--------------------------------------------------------------------------------------
    template< typename DownloaderType, typename ConfigurationType = NewlyUpdateDownloader::CancelConfigurationType,
        typename ResultType = NewlyUpdateDownloader::ProgressValueType >
    static void ExecuteCancelledDownload( nn::ns::SystemUpdateControl& control,
        NewlyUpdateDownloader::CancellerType& canceller,
        const ConfigurationType config, ResultType* pOutResultValue = nullptr ) NN_NOEXCEPT
    {
        // システムアップデートダウンロードタスクを破棄する。
        control.Relieve();
        NNT_EXPECT_RESULT_SUCCESS( nn::ns::DestroySystemUpdateTask() );
        control.Occupy();

        DownloaderType downloader( control, &canceller, config );
        NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
        const auto result = downloader.WaitResponse();
        if ( false == result.IsSuccess() )
        {
            if ( nullptr != pOutResultValue )
            {
                *pOutResultValue = downloader.GetCancelledProgressValue();
            }
            LogoutResultValue( result, "Happened an expected failure on [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]." );
        }
        else
        {
            GTEST_FAIL() << "Unexpected success on [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate].";
        }
    }

    //!--------------------------------------------------------------------------------------
    //! バックグラウンドダウンロード完了待ち.
    //! ダウンロード進捗中に１度だけネットワーク切断を行います。
    //!--------------------------------------------------------------------------------------
    static const bool WaitResponseForBackgroundDownload( nn::ns::SystemUpdateControl& control,
        unsigned abortElapsedSeconds = 20, const unsigned reportIntervalSeconds = 10 ) NN_NOEXCEPT
    {
        bool downloaded = false;

        // バックグラウンド用占有解除.
        control.Relieve();
        // クライアントネットワークリソース占有解除.
        EXPECT_EQ( true, ReleaseConnectionNetwork() );

        // バックグラウンド待ち
        const nn::TimeSpan beforeInProgress = nn::os::GetSystemTick().ToTimeSpan();
        while ( NN_STATIC_CONDITION( true ) )
        {
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( static_cast< int64_t >( reportIntervalSeconds ) ) );

            // progress チェック
            control.Occupy();
            const auto progress = control.GetDownloadProgress();
            control.Relieve();
            const auto state = nn::ns::GetBackgroundNetworkUpdateState();
            const auto spanInProgress = nn::os::GetSystemTick().ToTimeSpan() - beforeInProgress;
            const auto spanSeconds = spanInProgress.GetSeconds();
            LOG_OUT( "Background download state( %s ), progress( loaded %lld, total %lld ), at %lld seconds later.\n",
                NewlyUpdateDownloader::GetBackStateString( state ),
                progress.loaded, progress.total,
                spanSeconds
            );
            if ( nn::ns::BackgroundNetworkUpdateState::InProgress != state )
            {
                downloaded = ( nn::ns::BackgroundNetworkUpdateState::Ready == state );
                break;
            }
            // 強制中断
            if ( abortElapsedSeconds > 0 && spanSeconds > static_cast< int64_t >( abortElapsedSeconds ) )
            {
                LOG_OUT( ">>=== Abort background download by the network disconnection.\n" );
                {
                    FirmwareDebugSettings::NetworkConnectionController network;
                    network.ChangeConnection( false );
                    WaitExpectNetworkAvailable( false );
                    nn::os::SleepThread( nn::TimeSpan::FromSeconds( 5 ) );  // 接続OFF状態を5秒維持。
                }
                LOG_OUT( ">>=== Repair the network disconnection, Resume the background downloading.\n" );
                abortElapsedSeconds = 0;    // 中断は１度だけ.
            }
        }

        // クライアントネットワークリソース占有再要求.
        EXPECT_EQ( true, SubmitConnectionNetwork() );
        // 検証用に占有復帰.
        control.Occupy();

        return downloaded;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ダウンロード部以外の共通化
    //! @param[in] doRunnable   ダウンロード部の実装処理を指定します。
    //!                         呼び出し時は Occupy() 済であることが保証されます。
    //!                         処理中に Relieve()を行った場合、処理終了時には Occupy() 状態に復帰しておく必要があります。
    //!--------------------------------------------------------------------------------------
    void NupCase::DoDownloadAction( std::function< bool( NupCase& context, nn::ns::SystemUpdateControl& control ) > doRunnable ) NN_NOEXCEPT
    {
        // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
        auto& verifier = GetNcmStorageVerifier();
        verifier.PrepareUnexpectedContentOnStorage();

        // ネットワーク接続確立.
        ASSERT_EQ( true, SubmitConnectionNetwork() );

        // 本体更新制御の占有開始.
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        {
            // 最新の更新の検出. RequestCheckLatestUpdate
            LOG_OUT( "Start latest update checking... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
            NewlyUpdateInquirer inquirer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
            latest = inquirer.GetLatestStatus();
            EXPECT_EQ( nn::ns::LatestSystemUpdate::NeedsDownload, latest );
            LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );
        }

        bool newlyInstalled = false;
        if ( nn::ns::LatestSystemUpdate::NeedsDownload == latest && nullptr != doRunnable )
        {
            // download 成功及び完了したら true が返される想定。
            if ( true == doRunnable( *this, control ) )
            {
                // インストール. ApplyDownloadedUpdate
                ExecuteInstall( newlyInstalled, control, false );
            }
        }
        else if ( nn::ns::LatestSystemUpdate::Downloaded == latest )
        {
            // インストール. ApplyDownloadedUpdate
            ExecuteInstall( newlyInstalled, control, false );
        }

        // 無事インストールできたなら本体更新完了確認( latest status, already installed, ncmベリファイ )を実施。
        if ( true == newlyInstalled )
        {
            latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
            {
                // インストール済確認. RequestCheckLatestUpdate == nn::ns::LatestSystemUpdate::UpToDate
                LOG_OUT( "Start verify latest update status... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
                NewlyUpdateInquirer inquirer( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
                latest = inquirer.GetLatestStatus();
                EXPECT_EQ( nn::ns::LatestSystemUpdate::UpToDate, latest );
                LOG_OUT( "Complete verify latest update status [ %s ].\n", inquirer.GetLatestStatusString() );
            }

            if ( nn::ns::LatestSystemUpdate::UpToDate == latest )
            {
                // NCMデータベースベリファイ
                verifier.VerifyHasContentOnStorage();
                verifier.VerifyUnnecessaryPlaceHolderOnStorage();
                verifier.VerifyUnexpectedContentOnStorage();

                // ダウンロード済コンテンツの消去確認. RequestDownloadLatestUpdate != nn::ns::ResultAlreadyUpToDate
                LOG_OUT( "Verify whether the downloaded contents be clean... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_FAILURE( nn::ns::ResultAlreadyUpToDate, downloader.WaitResponse() );
                LOG_OUT( "Complete, verify whether the downloaded contents be clean.\n" );
            }
        }
    }

} // NOLINT(impl/function_size)


//!--------------------------------------------------------------------------------------
//!
//! - コマンドラインオプションで使用する DoDownloadAction を分岐させる。
//! - DoDownloadAction 単位でインストール→ベリファイが完了する。
//! - doRunnable 内で中断、及びダウンロード完了を完了させる。
//! - バックタスクでのダウンロード確認も doRunnable での中断後、doRunnable内で行う。
//! - std::funtion を利用しているので、分岐では std::function作成にして DoDownloadActionは最後に呼び出すだけでよい。
//!   但しラムダ引数部 ( [] )がタイミング依存なので、引数追加する場合は注意。
//!--------------------------------------------------------------------------------------
TEST_F( NupCase, AbortOnUpdate )
{
    char pCaseValue[ 32 ] = "\0";
    ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_AbortCase", sizeof( pCaseValue ), pCaseValue ) );

    if ( 0 == std::strcmp( pCaseValue, "CancelByApi" ) )
    {
        // ハンドルオブジェクト経由の Cancel() 呼び出しテスト。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )
        {
            bool downloaded = false;

            // キャンセラスレッド準備
            auto& canceller = context.GetParallelCanceller();

            // ダウンロード準備中キャンセル. RequestDownloadLatestUpdate
            // 準備中のキャンセルは、次のダウンロードAPIで再度最初からの準備。
            LOG_OUT( "===== Start update downloading with cancel at prepared... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader::CancelConfigurationType configPrepared( NewlyUpdateDownloader::CancelCategory::InDownloadPrepared, 1000 );
            ExecuteCancelledDownload< NewlyUpdateDownloader >( control, canceller, configPrepared );
            LOG_OUT( "Complete update downloading with cancel at prepared.\n" );

            // SIGLO-50315 メタ取得中の連続キャンセル
            LOG_OUT( "===== Start update downloading with cancel at prepared in meta downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloaderUntilMetaPrepared::ProgressValueType configTarget = NewlyUpdateDownloaderUntilMetaPrepared::ProgressValueType();
            const int repeatTimes = 5;
            for ( int i = 1; 0 == configTarget.total && i <= repeatTimes; ++i )
            {
                LOG_OUT( "===== Execute index [ %d / %d ] of cancel at prepared in meta downloading... =====\n", i, repeatTimes );
                ExecuteCancelledDownload< NewlyUpdateDownloaderUntilMetaPrepared, NewlyUpdateDownloaderUntilMetaPrepared::ProgressValueType >( control, canceller, configTarget, &configTarget );
            }
            LOG_OUT( "Complete update downloading with cancel at prepared in meta downloading.\n" );

            // ダウンロード中キャンセル. RequestDownloadLatestUpdate
            // ダウンロード中のキャンセルは、次のダウンロードAPIでダウンロード済の途中から再開。
            LOG_OUT( "===== Start update downloading with cancel at downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader::CancelConfigurationType configDownloding( NewlyUpdateDownloader::CancelCategory::InDownloading, 5000 );
            ExecuteCancelledDownload< NewlyUpdateDownloader >( control, canceller, configDownloding );
            LOG_OUT( "Complete update downloading with cancel at downloading.\n" );

            // ダウンロード正常. RequestDownloadLatestUpdate
            LOG_OUT( "===== Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader downloader( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
            LOG_OUT( "Complete latest update downloading.\n" );
            downloaded = control.HasDownloaded();
            EXPECT_EQ( true, downloaded );
            return downloaded;
        } );
    }
    else if ( 0 == std::strcmp( pCaseValue, "NetworkPrepare" ) )
    {
        // ネットワーク切断( foreground download with prepare ) 呼び出しテスト。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )
        {
            // ダウンロード準備中キャンセル. RequestDownloadLatestUpdate
            // 準備中のキャンセルは、次のダウンロードAPIで再度最初からの準備。
            LOG_OUT( "===== Start update downloading with abort at prepared... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader::CancelConfigurationType config( NewlyUpdateDownloader::CancelCategory::InDownloadPrepared, 2000 );
            ExecuteCancelledDownload< DownloaderWithNetworkAbort >( control, context.GetParallelCanceller(), config );
            LOG_OUT( "Complete update downloading with abort at prepared.\n" );

            return WaitResponseForBackgroundDownload( control, 20 );
        } );
    }
    else if ( 0 == std::strcmp( pCaseValue, "NetworkDownload" ) )
    {
        // ネットワーク切断( foreground download with actual progress ) 呼び出しテスト。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )
        {
            // ダウンロード準備中キャンセル. RequestDownloadLatestUpdate
            // 準備中のキャンセルは、次のダウンロードAPIで再度最初からの準備。
            LOG_OUT( "===== Start update downloading with abort at downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader::CancelConfigurationType config( NewlyUpdateDownloader::CancelCategory::InDownloading, 5000 );
            ExecuteCancelledDownload< DownloaderWithNetworkAbort >( control, context.GetParallelCanceller(), config );
            LOG_OUT( "Complete update downloading with abort at downloading.\n" );

            return WaitResponseForBackgroundDownload( control, 30 );
        } );
    }
    else
    {
        LOG_OUT( "Not specify the test case." );
    }

} // NOLINT(impl/function_size)
