﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <utility>
#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_FsUtilities.h"
#include "libraries/testNs_CommandOptions.h"
#include "libraries/testNs_InstantMessageThread.h"
#include "testNs_NcmStorageVerify.h"

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( & )[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Result出力.
    //!--------------------------------------------------------------------------------------
    static void LogoutResultValue( const nn::Result& result, const char* const pTitle )
    {
        if ( nullptr != pTitle )
        {
            LOG_OUT( "%s\n", pTitle );
        }
        LOG_OUT( "-------- Result ---------\n" );
        LOG_OUT( "module = %u\n", result.GetModule() );
        LOG_OUT( "description = %u\n", result.GetDescription() );
        LOG_OUT( "inner value = 0x%08x\n", result.GetInnerValueForDebug() );
        LOG_OUT( "-------------------------\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! 並列通知用スレッド
    //!--------------------------------------------------------------------------------------
    static const size_t CancellerThreadStackSize = nn::os::ThreadStackAlignment * 4;
    typedef testns::InstantMessenger< CancellerThreadStackSize >::ThreadContext ParallelMessenger;

    //!--------------------------------------------------------------------------------------
    //! @brief 並列要求用
    //!--------------------------------------------------------------------------------------
    class ParallelNotifier
    {
    public:
        typedef ParallelMessenger MessengerType;

        void CancelThere( const int64_t expectedElapsedMillis ) NN_NOEXCEPT
        {
            MessengerType* pMessenger;
            if ( nullptr != ( pMessenger = m_pMessenger ) )
            {
                m_OverTimeForCancel = nn::os::GetSystemTick() + nn::os::ConvertToTick( nn::TimeSpan( nn::TimeSpanType::FromMilliSeconds( expectedElapsedMillis ) ) );
                pMessenger->Start( []( testns::MessageThreadContext* pThread, const testns::ThreadMessage* pMessage )->void
                {
                    auto* const pThis = reinterpret_cast< ParallelNotifier* >( pMessage->GetUserContext() );
                    const nn::os::Tick diff = pThis->m_OverTimeForCancel - nn::os::GetSystemTick();
                    if ( diff.GetInt64Value() > 0 )
                    {
                        const nn::TimeSpan sleeping = nn::os::ConvertToTimeSpan( diff );
                        LOG_OUT( "Sleep at [ %lld ] ms.\n", sleeping.GetMilliSeconds() );
                        nn::os::SleepThread( sleeping );
                    }
                    pThis->OnRequestByParallel();
                }, this );
            }
        }

    protected:
        explicit ParallelNotifier( MessengerType* pMessenger = nullptr ) NN_NOEXCEPT
            : m_pMessenger( pMessenger )
        {
        }

        ~ParallelNotifier() NN_NOEXCEPT
        {
            MessengerType* pMessenger;
            if ( nullptr != ( pMessenger = m_pMessenger ) )
            {
                m_pMessenger = nullptr;
                pMessenger->CancelAndWait();
            }
        }

        virtual void OnRequestByParallel() NN_NOEXCEPT
        {
        }

    private:
        MessengerType*  m_pMessenger;
        nn::os::Tick    m_OverTimeForCancel;
    };

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class NupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();

            auto& messengerInstance = nn::util::Get( m_ParallelMessenger );
            new ( &messengerInstance ) ParallelMessenger();
            messengerInstance.Initialize();
        }

        virtual void TearDown()
        {
            nn::util::Get( m_ParallelMessenger ).Finalize();
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
            testns::InitializeCommandLineOptions();
        }

        static void TearDownTestCase()
        {
            testns::FinalizeCommandLineOptions();
            testns::UnmountHostFileSystem();
        }

    protected:
        nn::util::TypedStorage< ParallelMessenger, sizeof( ParallelMessenger ), NN_ALIGNOF( ParallelMessenger ) > m_ParallelMessenger;
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;

    public:
        NcmStorageVerifier& GetNcmStorageVerifier() NN_NOEXCEPT
        {
            return nn::util::Get( m_NcmStorageVerifier );
        }

        ParallelMessenger& GetParallelMessenger() NN_NOEXCEPT
        {
            return nn::util::Get( m_ParallelMessenger );
        }

        void DoInstallAction( std::function< void ( NupCase& context, nn::ns::SystemUpdateControl& control ) > doRunnable ) NN_NOEXCEPT;
        void DoVerifyAction() NN_NOEXCEPT;
    };



    //!--------------------------------------------------------------------------------------
    //! RequestSynchronizer
    //!--------------------------------------------------------------------------------------
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef TControl    ControlType;
        typedef THandle     HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse() NN_NOEXCEPT
        {
            const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            auto result = OnWaitResponse( m_Control, m_Handle );
            m_ElapsedTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT
            : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    //!--------------------------------------------------------------------------------------
    //! CheckLatestUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInquirer : public RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate >
    {
    public:
        static const nn::ns::LatestSystemUpdate INVALID_LATEST_VALUE = static_cast< nn::ns::LatestSystemUpdate >( -1 );
        typedef RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate > BaseType;

        explicit NewlyUpdateInquirer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control ), m_LatestStatus( INVALID_LATEST_VALUE )
        {
        }

        const nn::ns::LatestSystemUpdate& GetLatestStatus() const NN_NOEXCEPT
        {
            return m_LatestStatus;
        }

        static const char* GetLatestStatusString( const nn::ns::LatestSystemUpdate status ) NN_NOEXCEPT
        {
            switch ( status )
            {
            case nn::ns::LatestSystemUpdate::Downloaded:    return "Downloaded";
            case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
            case nn::ns::LatestSystemUpdate::UpToDate:      return "UpToDate";
            default:                                        return "Unknown";
            }
        }

        const char* GetLatestStatusString() const NN_NOEXCEPT
        {
            return GetLatestStatusString( m_LatestStatus );
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                LOG_OUT( "Wating check latest update...\n" );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            return handle.Get( &m_LatestStatus );
        }

    private:
        nn::ns::LatestSystemUpdate m_LatestStatus;
    };

    //!--------------------------------------------------------------------------------------
    //! ApplyDownloadedUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            control.ApplyDownloadedUpdate();
            NN_RESULT_SUCCESS;
        }
    };

    //!--------------------------------------------------------------------------------------
    //! DownloadLatestUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateDownloader : public RequestSynchronizer< nn::ns::AsyncResult >
    {
    public:
        typedef RequestSynchronizer< nn::ns::AsyncResult > BaseType;

        explicit NewlyUpdateDownloader( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

        static const char* GetBackStateString( const nn::ns::BackgroundNetworkUpdateState state ) NN_NOEXCEPT
        {
            switch ( state )
            {
            case nn::ns::BackgroundNetworkUpdateState::None:        return "None";
            case nn::ns::BackgroundNetworkUpdateState::InProgress:  return "InProgress";
            case nn::ns::BackgroundNetworkUpdateState::Ready:       return "Ready";
            default:                                                return "Unknown";
            }
        }

        static const char* GetBackStateString() NN_NOEXCEPT
        {
            return GetBackStateString( nn::ns::GetBackgroundNetworkUpdateState() );
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                const auto progress = control.GetDownloadProgress();
                LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            const auto progress = control.GetDownloadProgress();
            LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
            return handle.Get();
        }
    };



    //!--------------------------------------------------------------------------------------
    //! Network connection request.
    //! 本クライアントアプリケーションがネットワークリソースを占有する事を要求します。
    //!--------------------------------------------------------------------------------------
    static bool SubmitConnectionNetwork() NN_NOEXCEPT
    {
        LOG_OUT( "Request network connection ...\n" );
        const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
        nn::nifm::SubmitNetworkRequestAndWait();
        const nn::TimeSpan spanOnConnection = nn::os::GetSystemTick().ToTimeSpan() - before;
        LOG_OUT( "Response wait : %lld ms.\n", spanOnConnection.GetMilliSeconds() );
        const auto availableNetwork = nn::nifm::IsNetworkAvailable();
        LOG_OUT( "Network connection %s.\n", ( availableNetwork ? "succeeded" : "failed" ) );
        if ( false == availableNetwork )
        {
            LogoutResultValue( nn::nifm::HandleNetworkRequestResult(), "Network connection error." );
        }
        return availableNetwork;
    }

    //!--------------------------------------------------------------------------------------
    //! Cancel connection request.
    //! 本クライアントアプリケーションがネットワークリソースの占有を放棄する事を要求します。
    //!--------------------------------------------------------------------------------------
    static bool ReleaseConnectionNetwork() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();
        const auto availableNetwork = nn::nifm::IsNetworkAvailable();
        LOG_OUT( ">>=== Release network connection ... %s.\n", ( false == availableNetwork ) ? "succeeded" : "failed" );
        return !availableNetwork;
    }

    //!--------------------------------------------------------------------------------------
    //! バックグラウンドダウンロード完了待ち.
    //!--------------------------------------------------------------------------------------
    static const bool WaitResponseForBackgroundDownload( nn::ns::SystemUpdateControl& control, const unsigned reportIntervalSeconds = 10 ) NN_NOEXCEPT
    {
        bool downloaded = false;

        // バックグラウンド用占有解除.
        control.Relieve();
        // クライアントネットワークリソース占有解除.
        EXPECT_EQ( true, ReleaseConnectionNetwork() );

        // バックグラウンド待ち
        const nn::TimeSpan beforeInProgress = nn::os::GetSystemTick().ToTimeSpan();
        while ( NN_STATIC_CONDITION( true ) )
        {
            nn::os::SleepThread( nn::TimeSpan::FromSeconds( static_cast< int64_t >( reportIntervalSeconds ) ) );

            // progress チェック
            control.Occupy();
            const auto progress = control.GetDownloadProgress();
            control.Relieve();
            const auto state = nn::ns::GetBackgroundNetworkUpdateState();
            const auto spanInProgress = nn::os::GetSystemTick().ToTimeSpan() - beforeInProgress;
            const auto spanSeconds = spanInProgress.GetSeconds();
            LOG_OUT( "Background download state( %s ), progress( loaded %lld, total %lld ), at %lld seconds later.\n",
                NewlyUpdateDownloader::GetBackStateString( state ),
                progress.loaded, progress.total,
                spanSeconds
            );
            if ( nn::ns::BackgroundNetworkUpdateState::InProgress != state )
            {
                downloaded = ( nn::ns::BackgroundNetworkUpdateState::Ready == state );
                break;
            }
        }

        // クライアントネットワークリソース占有再要求.
        EXPECT_EQ( true, SubmitConnectionNetwork() );
        // 検証用に占有復帰.
        control.Occupy();

        return downloaded;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief インストール部以外の共通化
    //!         ncm ベリファイなし
    //!--------------------------------------------------------------------------------------
    void NupCase::DoInstallAction( std::function< void ( NupCase& context, nn::ns::SystemUpdateControl& control ) > doRunnable ) NN_NOEXCEPT
    {
        // ネットワーク接続確立.
        ASSERT_EQ( true, SubmitConnectionNetwork() );

        // 本体更新制御の占有開始.
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        {
            // 最新の更新の検出. RequestCheckLatestUpdate
            LOG_OUT( "Start latest update checking... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
            NewlyUpdateInquirer inquirer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
            latest = inquirer.GetLatestStatus();
            EXPECT_EQ( nn::ns::LatestSystemUpdate::NeedsDownload, latest );
            LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );
        }

        if ( nn::ns::LatestSystemUpdate::NeedsDownload == latest )
        {
            bool downloaded = false;
            {
                LOG_OUT( "Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
                LOG_OUT( "Complete latest update downloading.\n" );
                downloaded = control.HasDownloaded();
                EXPECT_EQ( true, downloaded );
            }
            // download 成功及び完了したら true が返される想定。
            if ( true == downloaded && nullptr != doRunnable )
            {
                doRunnable( *this, control );
            }
        }
        else if ( nn::ns::LatestSystemUpdate::Downloaded == latest && nullptr != doRunnable )
        {
            doRunnable( *this, control );
        }
    } // NOLINT(impl/function_size)

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void NupCase::DoVerifyAction() NN_NOEXCEPT
    {
        // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
        auto& verifier = GetNcmStorageVerifier();
        verifier.PrepareUnexpectedContentOnStorage();

        // ネットワーク接続確立.
        ASSERT_EQ( true, SubmitConnectionNetwork() );

        // 本体更新制御の占有開始.
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        {
            // 最新の更新の検出. RequestCheckLatestUpdate
            LOG_OUT( "Start latest update checking... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
            NewlyUpdateInquirer inquirer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
            latest = inquirer.GetLatestStatus();
            LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );
        }

        if ( nn::ns::LatestSystemUpdate::NeedsDownload == latest )
        {
            // バックグラウンドで走ってる？
            if ( true == WaitResponseForBackgroundDownload( control ) )
            {
                latest = nn::ns::LatestSystemUpdate::Downloaded;
            }
            else
            {
                // 正しく中断されてると判断して、ダウンロード再要求。
                LOG_OUT( "Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
                LOG_OUT( "Complete latest update downloading.\n" );
                EXPECT_EQ( true, control.HasDownloaded() );
            }
        }

        bool newlyInstalled = false;
        if ( true == control.HasDownloaded() )
        {
            LOG_OUT( "===== Start install... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate] =====\n" );
            NewlyUpdateInstaller installer( control );
            const auto result = installer.WaitResponse();
            newlyInstalled = result.IsSuccess();
            NNT_EXPECT_RESULT_SUCCESS( result );
        }
        else
        {
            LOG_OUT( "Unexpected procedure latest status : [ %s ], HasDownloaded( %s ).\n",
                NewlyUpdateInquirer::GetLatestStatusString( latest ),
                control.HasDownloaded() ? "true" : "false"
            );
            newlyInstalled = true;
        }

        // 無事インストールできたなら本体更新完了確認( latest status, already installed, ncmベリファイ )を実施。
        if ( true == newlyInstalled )
        {
            latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
            {
                // インストール済確認. RequestCheckLatestUpdate == nn::ns::LatestSystemUpdate::UpToDate
                LOG_OUT( "Start verify latest update status... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
                NewlyUpdateInquirer inquirer( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
                latest = inquirer.GetLatestStatus();
                EXPECT_EQ( nn::ns::LatestSystemUpdate::UpToDate, latest );
                LOG_OUT( "Complete verify latest update status [ %s ].\n", inquirer.GetLatestStatusString() );
            }

            if ( nn::ns::LatestSystemUpdate::UpToDate == latest )
            {
                // NCMデータベースベリファイ
                verifier.VerifyHasContentOnStorage();
                verifier.VerifyUnnecessaryPlaceHolderOnStorage();
                verifier.VerifyUnexpectedContentOnStorage();

                // ダウンロード済コンテンツの消去確認. RequestDownloadLatestUpdate != nn::ns::ResultAlreadyUpToDate
                LOG_OUT( "Verify whether the downloaded contents be clean... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_FAILURE( nn::ns::ResultAlreadyUpToDate, downloader.WaitResponse() );
                LOG_OUT( "Complete, verify whether the downloaded contents be clean.\n" );
            }
        }
    } // NOLINT(impl/function_size)

    //#define _USE_ADVISORY_FILE_
    //!--------------------------------------------------------------------------------------
    //! @brief 通知用
    //!--------------------------------------------------------------------------------------
    class InstallCanceller : public ParallelNotifier
    {
    public:
        explicit InstallCanceller( MessengerType& messengerContext ) NN_NOEXCEPT
            : ParallelNotifier( &messengerContext )
        {
#if defined( _USE_ADVISORY_FILE_ )
            m_OutPath[ 0 ] = '\0';
            if ( false == testns::MakeHostFileSystemPath( m_OutPath, "/InstallRun.advisory.txt" ) )
            {
                LOG_OUT( "Can not make the advisory file as install running on the host file system, because did not mount the host file system.\n" );
            }
            else
            {
                // アドバイザリファイルが生成されるまで待機
                const nn::TimeSpan begin = nn::os::GetSystemTick().ToTimeSpan();
                while ( false == fsutil::IsExistPath( m_OutPath ) )
                {
                    nn::os::SleepThread( nn::TimeSpan::FromSeconds( 2 ) );
                    const nn::TimeSpan elapsedTime = nn::os::GetSystemTick().ToTimeSpan() - begin;
                    if ( elapsedTime.GetSeconds() > 60 )
                    {
                        LOG_OUT( "Can not detect the advisory file on the host file system, time out.\n" );
                        break;
                    }
                }
            }
#endif
        }

    protected:
        virtual void OnRequestByParallel() NN_NOEXCEPT NN_OVERRIDE
        {
#if defined( _USE_ADVISORY_FILE_ )
            // ホストのアドバイザリファイルを削除して、外部で検知。
            // アドバイザリファイルの生成はスクリプト側の準備完了時に行います。
            NNT_EXPECT_RESULT_SUCCESS( nn::fs::DeleteFile( m_OutPath ) );
#endif

            LOG_OUT( ">>> Request force process exit: On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate] <<<\n" );
            SUCCEED();
        }

#if defined( _USE_ADVISORY_FILE_ )
    private:
        char m_OutPath[ 256 ];
#endif

    };
}

//!--------------------------------------------------------------------------------------
//! @brief テスト本体
//!--------------------------------------------------------------------------------------
TEST_F( NupCase, AbortOnUpdate )
{
    char pCaseValue[ 32 ] = "";
    ASSERT_EQ( true, testns::QueryCommandLineOptions( "--testns_AbortCase", sizeof( pCaseValue ), pCaseValue ) );
    LOG_OUT( "==============================================\n"  );
    LOG_OUT( "Specified the test case [ %s ].\n", pCaseValue  );
    LOG_OUT( "==============================================\n\n"  );

    if ( 0 == std::strcmp( pCaseValue, "Install" ) )
    {
        // 通常ダウンロード
        DoInstallAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )
        {
            //InstallCanceller canceller( context.GetParallelMessenger() );
            //canceller.CancelThere( 0 );        // 指定ミリ秒数後にキャンセラ起動.
            LOG_OUT( "===== Start install... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate] =====\n" );

            // スクリプト側で RunOnTaget をログマッチで終了させ、Power-offコマンドを発行させます.
            LOG_OUT( ">>> Request force process exit: On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate] <<<\n" );
            SUCCEED();

            // ローカル環境( SDEV MP + 1000BASE + 5mケーブル )
            // ControlTarget power-off 所要時間 3.65 ms.
            // 3.0s で、再起動後 UpToDate
            // 3.3s 以上で 再起動後 NeedsDownload
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 3300 ) );

            control.ApplyDownloadedUpdate();    // AbortOnUpdate のNUPコンテンツ( 64MiB * 7 )だと大体 1500 ms で完了する. しびあー.
            LOG_OUT( "===== Complete install... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate] =====\n" );
        } );
    }
    else if ( 0 == std::strcmp( pCaseValue, "Verify" ) )
    {
        DoVerifyAction();
    }
    else if ( 0 == std::strcmp( pCaseValue, "ScriptTest" ) )
    {
        InstallCanceller canceller( GetParallelMessenger() );
        canceller.CancelThere( 300 );       // 指定ミリ秒数後にキャンセラ起動.
        nn::os::SleepThread( nn::TimeSpan::FromMinutes( 5 ) );
    }
    else
    {
        LOG_OUT( "Not specify the test case.\n" );
    }

} // NOLINT(impl/function_size)
