﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using CsTestAssistants;

namespace testNs_CupSimpleAuto
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    [TestClass]
    public class TestCupAutoboot : TestClassBase
    {
        // 定数
        private const string LogPrefix = "[CupSimpleAuto]";
        private const string CupNspKey = "CupNsp";
        private const string AppNspKey = "AppNsp";
        private readonly TimeSpan CupTimeout = new TimeSpan(0, 2, 0);

        // プロパティ
        static public string InputCupConfigFile { get; set; }
        static public ID64 ApplicationId { get; set; } = new ID64(0x01001a500005e020);
        static public ID64 UpdateId { get; set; } = new ID64(0x01001a500005e021);
        static public ID64 ProgramId { get; set; } = new ID64(0x01001a500005e022);
        static public string TestOption { get; set; }

        // 定義
        class LogServie
        {
            public System.Net.IPAddress Address { get; }
            public int Port { get; }

            public LogServie(string address, string port)
            {
                Address = System.Net.IPAddress.Parse(address);
                Port = int.Parse(port);
            }

            public override string ToString()
            {
                return $"{Address}:{Port}";
            }
        }

        public TestContext TestContext { get; set; }

        // テストクラス初期処理
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);

            InputCupConfigFile = GetProperties(context, "InputCupConfigFile", "");
            TestOption = GetProperties(context, "TestOption", "");
            var id = GetProperties(context, "ApplicationId", "");
            if (!string.IsNullOrEmpty(id))
            {
                ApplicationId = new ID64(id);
            }

            Log.WriteLine($"=== Test Parameter ===");
            Log.WriteLine($"InputCupConfigFile : {InputCupConfigFile}");
            Log.WriteLine($"ApplicationId      : {ApplicationId}");
            Log.WriteLine($"UpdateId           : {UpdateId}");
            Log.WriteLine($"ProgramId          : {ProgramId}");
            Log.WriteLine($"TestOption         : {TestOption}");
        }

        // 要素取得
        private static T GetProperties<T>(TestContext context, string key, T @default)
        {
            T value = @default;
            if (context.Properties.Contains(key))
            {
                value = (T)context.Properties[key];
            }
            return value;
        }

        // 作業ディレクトリ作成
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        [TestMethod]
        public void TestWriteCard()
        {
            var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
            var intermediate = GenerateIntermediateDirectoryAsMethod(executor);
            try
            {
                // ゲームカード書き込み
                Log.WriteLine($"Phase: write gamecard...");
                WriteCard(executor);

                // CUP 適用前の再起動
                Log.WriteLine($"Phase: reset and connect...");
                ExpectEvaluator.IsTrue(executor.ControlTarget($"connect --reset"));

                // 上記再起動から CUP 適用中まで
                if (string.Compare(TestOption, "RestartDuringUpdate") == 0)
                {
                    // ログ待機（想定：CUP 適用中）
                    Log.WriteLine($"Phase: wait for system update progress...");
                    var service = GetLogServie(executor);
                    var pattern = $"\\[GameCardManager\\] SystemUpdateProgress";
                    ExpectEvaluator.IsTrue(WaitLogMessage(service, pattern, CupTimeout));

                    // CUP 適用中の再起動
                    Log.WriteLine($"Phase: restart during update...");
                    ExpectEvaluator.IsTrue(executor.ControlTarget($"connect --reset"));
                }

                // CUP 適用完了後の自動再起動まで
                {
                    // ログ待機（想定：CUP 適用完了）
                    Log.WriteLine($"Phase: wait for system update closing...");

                    var service = GetLogServie(executor);
                    var pattern = $"\\[SystemUpdateInterfaceServer\\] Close system update control";
                    ExpectEvaluator.IsTrue(WaitLogMessage(service, pattern, CupTimeout));

                    // CUP 適用完了後に即座に再起動すわけではないので少し待機
                    var wait = 10;
                    Log.WriteLine($"Phase: wait {wait} sec");
                    System.Threading.Thread.Sleep(wait * 1000);
                }

                // 上記再起動からアプリ起動まで
                {
                    // CUP 適用直後に再起動されるので再接続
                    Log.WriteLine($"Phase: connect after restart...");
                    ExpectEvaluator.IsTrue(executor.ControlTarget($"connect"));

                    // ログ待機（想定：アプリ起動）
                    Log.WriteLine($"Phase: wait for application start...");
                    var service = GetLogServie(executor);
                    var pattern = $"ApplicationID  : 0x{ApplicationId}";
                    ExpectEvaluator.IsTrue(WaitLogMessage(service, pattern, CupTimeout));

                    // CUP 適用後の再起動
                    Log.WriteLine($"Phase: reset after updated...");
                    ExpectEvaluator.IsTrue(executor.ControlTarget($"connect --reset"));
                }

                // 上記再起動からアプリ起動まで
                {
                    // # ログ待機（想定：アプリ起動 ※CUP適用済なので途中に再起動はない）
                    Log.WriteLine($"Phase: wait for application start...");
                    var service = GetLogServie(executor);
                    var pattern = $"ApplicationID  : 0x{ApplicationId}";
                    ExpectEvaluator.IsTrue(WaitLogMessage(service, pattern, CupTimeout));
                }

            }
            catch (Exception e)
            {
                Log.WriteLine($"Phase: catch exception");
                Log.WriteLineAsIs(e.Message);
                Log.WriteLineAsIs(e.StackTrace);

                // ターゲットの接続不良を想定してリセット＆再接続
                Log.WriteLine($"Phase: reset");
                executor.ControlTarget($"connect --reset");

                Log.WriteLine($"Phase: wait {CupTimeout.TotalSeconds} sec");
                System.Threading.Thread.Sleep((int)CupTimeout.TotalMilliseconds);

                // 上記の再起動中に CUP されて切断されることを考慮して再接続
                Log.WriteLine($"Phase: connect...");
                executor.ControlTarget($"connect");

                throw;
            }
            finally
            {
                // 自動起動のまま残ると他のテストに悪影響をもたらすので必ず削除
                Log.WriteLine($"Phase: erase gamecard...");
                executor.RunDevMenuCommandSystem($"gamecard erase ");

                if (IsUnitTestOnIDE)
                {
                    // IDE 起動による検証時はインストールしたコンテンツを削除
                    Log.WriteLine($"Phase: erase system update...");
                    executor.RunDevMenuCommandSystem($"systemprogram uninstall 0x{UpdateId}");
                    executor.RunDevMenuCommandSystem($"systemprogram uninstall 0x{ProgramId}");
                }

                Log.WriteLine($"Phase: reset and connect...");
                executor.ControlTarget($"connect --reset");
            }
        }

        private void WriteCard(SigloHelper.CommodityExecutor.Context executor)
        {
            // 事前条件
            ExpectEvaluator.IsTrue(System.IO.File.Exists(InputCupConfigFile));

            // 設定ファイル読み込み
            var config = new Dictionary<string, string>();
            using (var reader = new System.IO.StreamReader(InputCupConfigFile))
            {
                while (!reader.EndOfStream)
                {
                    var line = reader.ReadLine();
                    if (!string.IsNullOrEmpty(line))
                    {
                        var array = line.Split("=".ToCharArray());
                        ExpectEvaluator.IsTrue(array.Length == 2);

                        var key = array[0];
                        var value = array[1].Trim(" \"".ToCharArray());
                        config.Add(key, value);
                    }
                }
            }

            // 対象ファイル確認
            ExpectEvaluator.IsTrue(config.ContainsKey(AppNspKey) && System.IO.File.Exists(config[AppNspKey]));
            ExpectEvaluator.IsTrue(config.ContainsKey(CupNspKey) && System.IO.File.Exists(config[CupNspKey]));

            // カード書き込み
            ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                $"gamecard write \"{config[AppNspKey]}\" --update-partition \"{config[CupNspKey]}\" --auto"
            ));
        }

        // ログサービスのアドレス、ポート番号を取得
        private LogServie GetLogServie(SigloHelper.CommodityExecutor.Context executor)
        {
            const string AddressKey = "address";
            const string PortKey = "port";

            //ExpectEvaluator.IsTrue(executor.ControlTarget($"connect"));
            ExpectEvaluator.IsTrue(executor.ControlTargetNoTarget($"list-service"));

            var pattern = $"^{executor.TargetName}.+@Log\\t(?<{AddressKey}>[\\d\\.]+):(?<{PortKey}>\\d+)";
            var output = executor.OutputStream.Standard.ToString();
            var match = Regex.Match(output, pattern, RegexOptions.Multiline | RegexOptions.IgnoreCase);
            ExpectEvaluator.IsTrue(match.Success);

            return new LogServie(match.Groups[AddressKey].Value, match.Groups[PortKey].Value);
        }

        // 指定ログメッセージを待機
        private bool WaitLogMessage(LogServie service, string pattern, TimeSpan timeout)
        {
            Log.WriteLineAsIs($"connect {service}");
            var ip = new System.Net.IPEndPoint(service.Address, service.Port);
            var client = new System.Net.Sockets.TcpClient();
            client.Connect(service.Address, service.Port);

            var matched = false;
            using (var stream = client.GetStream())
            using (var reader = new System.IO.StreamReader(stream))
            {
                var msec = (int)timeout.TotalMilliseconds;
                if (msec > 0)
                {
                    stream.ReadTimeout = msec;
                }

                Regex regex = null;
                if (!string.IsNullOrEmpty(pattern))
                {
                    Log.WriteLineAsIs($"waiting... {pattern}");
                    regex = new Regex(pattern, RegexOptions.IgnoreCase);
                }

                var wather = new Retry.Watcher(0, (int)timeout.TotalSeconds);
                while (true)
                {
                    // タイムアウト確認
                    if (!wather.IsContinuable)
                    {
                        Log.WriteLineAsIs($"timeout... {service}");
                        break;
                    }

                    // 切断確認
                    var socket = client.Client;
                    var readable = socket.Poll(1000, System.Net.Sockets.SelectMode.SelectRead);
                    var received = socket.Available > 0;
                    if (readable && !received)
                    {
                        Log.WriteLineAsIs($"disconnected... {service}");
                        break;
                    }

                    // 受信中なら1行読み込む
                    var line = "";
                    if (received)
                    {
                        line = reader.ReadLine();
                    }
                    if (string.IsNullOrEmpty(line))
                    {
                        continue;
                    }
                    Log.WriteLineAsIs(line);

                    // パターン一致確認
                    if (regex != null && regex.IsMatch(line))
                    {
                        matched = true;
                        break;
                    }
                }
            }
            return matched;
        }
    }
}
