﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <string>
#include <nn/os.h>

namespace testns {

//!--------------------------------------------------------------------------------------
//! @brief 区間処理速度計測ユーティリティ. ( 軽量版 )
//!--------------------------------------------------------------------------------------
class SectionPerformanceMeasureLite
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測開始
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static const nn::os::Tick Begin() NN_NOEXCEPT
    {
        return nn::os::GetSystemTick();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測終了.
    //! @return ミリ秒を返却します。
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static const int64_t End( const nn::os::Tick& beginTick ) NN_NOEXCEPT
    {
        return ( nn::os::GetSystemTick() - beginTick ).ToTimeSpan().GetMilliSeconds();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測終了.
    //! @details 計測結果をミリ秒文字列に変換して出力します。
    //!--------------------------------------------------------------------------------------
    static std::string& End( const nn::os::Tick& beginTick, std::string& outReport ) NN_NOEXCEPT;
};

//!--------------------------------------------------------------------------------------
//! @brief 区間処理速度計測ユーティリティ.
//!--------------------------------------------------------------------------------------
class SectionPerformanceMeasure
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief デフォルトコンストラクタ
    //!--------------------------------------------------------------------------------------
    SectionPerformanceMeasure() NN_NOEXCEPT : m_AccumAverage( 0 ) {}

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測開始
    //!--------------------------------------------------------------------------------------
    nn::os::Tick Begin() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測終了
    //!--------------------------------------------------------------------------------------
    nn::os::Tick End() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測平均値取得.
    //! @details Begin() ⇔ End() で計測された区間速度の累積指数平均を TimeSpan型で取得します.
    //!--------------------------------------------------------------------------------------
    nn::TimeSpan GetAverage() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief 区間計測平均値をログに報告.
    //! @details Begin() ⇔ End() で計測された区間速度の累積指数平均をミリ秒でログに出力します.
    //! @param[in] reportInterval レポート間隔をミリ秒単位で指定します.
    //!--------------------------------------------------------------------------------------
    void ReportAverage( const uint32_t reportInterval = 0, const char* prefix = "SectionPerformanceMeasure average" ) NN_NOEXCEPT;

private:
    nn::os::Tick    m_BeginTick;
    nn::os::Tick    m_ReportedTick;
    int64_t         m_AccumAverage;
};

//!--------------------------------------------------------------------------------------
//! @brief FPSユーティリティ
//!--------------------------------------------------------------------------------------
class FpsMeasure
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief デフォルトコンストラクタ
    //!--------------------------------------------------------------------------------------
    FpsMeasure() NN_NOEXCEPT : m_NowTick( 0 ), m_AccumAverage( 0 ) {}

    //!--------------------------------------------------------------------------------------
    //! @brief ポーリング計測
    //!--------------------------------------------------------------------------------------
    const float Measure() NN_NOEXCEPT;

    //!--------------------------------------------------------------------------------------
    //! @brief Tickカウントから FPSを算出します。
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static const float ComputeValue( const int64_t tickValue ) NN_NOEXCEPT
    {
        return static_cast< float >( 1000000.0 / nn::os::Tick( tickValue ).ToTimeSpan().GetMicroSeconds() );
    }

private:
    nn::os::Tick    m_NowTick;
    int64_t         m_AccumAverage;
};

} // ::testns
