﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include "testNs_PerformanceMeasure.h"

#define LOG_OUT( ... )      NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

//=======================================================================================
namespace testns {
//=======================================================================================

//!--------------------------------------------------------------------------------------
//! @brief 区間計測開始
//!--------------------------------------------------------------------------------------
nn::os::Tick SectionPerformanceMeasure::Begin() NN_NOEXCEPT
{
    nn::os::Tick result;
    m_BeginTick = result = nn::os::GetSystemTick();
    return result;
}

//!--------------------------------------------------------------------------------------
//! @brief 区間計測終了
//!--------------------------------------------------------------------------------------
nn::os::Tick SectionPerformanceMeasure::End() NN_NOEXCEPT
{
    const nn::os::Tick pre = m_BeginTick;
    if ( pre.GetInt64Value() > 0 )
    {
        const nn::os::Tick now = nn::os::GetSystemTick();
        const int64_t average = static_cast< int64_t >( ( m_AccumAverage * 0.9 ) + ( ( now - pre ).GetInt64Value() * 0.1 ) );
        m_AccumAverage = average;
        m_BeginTick = now;
        return now;
    }
    return nn::os::Tick();
}

//!--------------------------------------------------------------------------------------
//! @brief 区間計測平均値取得.
//! @details Begin() ⇔ End() で計測された区間速度の累積指数平均を TimeSpan型で取得します.
//!--------------------------------------------------------------------------------------
nn::TimeSpan SectionPerformanceMeasure::GetAverage() NN_NOEXCEPT
{
    return nn::os::ConvertToTimeSpan( nn::os::Tick( m_AccumAverage ) );
}

//!--------------------------------------------------------------------------------------
//! @brief 区間計測平均値をログに報告.
//! @details Begin() ⇔ End() で計測された区間速度の累積指数平均をミリ秒でログに出力します.
//! @param[in] reportInterval レポート間隔をミリ秒単位で指定します.
//!--------------------------------------------------------------------------------------
void SectionPerformanceMeasure::ReportAverage( const uint32_t reportInterval, const char* prefix ) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    const nn::os::Tick interval = nn::os::ConvertToTick( nn::TimeSpan( nn::TimeSpanType::FromMilliSeconds( reportInterval ) ) );
    const nn::os::Tick now = nn::os::GetSystemTick();
    if ( ( now - m_ReportedTick ) > interval )
    {
        const double millis = GetAverage().GetMicroSeconds() / 1000.0;
        LOG_OUT( "%s[ %zf ms ]\n", prefix, millis );
        m_ReportedTick = now;
    }

#else

    NN_UNUSED( reportInterval );
    NN_UNUSED( prefix );

#endif
}

//!--------------------------------------------------------------------------------------
//! @brief 区間計測終了.
//! @details 計測結果をミリ秒文字列に変換して出力します。
//!--------------------------------------------------------------------------------------
std::string& SectionPerformanceMeasureLite::End( const nn::os::Tick& beginTick, std::string& outReport ) NN_NOEXCEPT
{
    char buf[ 64 ];
    const double millis = ( nn::os::GetSystemTick() - beginTick ).ToTimeSpan().GetMicroSeconds() / 1000.0;
    nn::util::SNPrintf( buf, sizeof( buf ), "[ %zf ms ]", millis );
    outReport = buf;
    return outReport;
}

//!--------------------------------------------------------------------------------------
//! @brief ポーリング監視
//!--------------------------------------------------------------------------------------
const float FpsMeasure::Measure() NN_NOEXCEPT
{
    const nn::os::Tick now = nn::os::GetSystemTick();
    const nn::os::Tick pre = m_NowTick;
    m_NowTick = now;

    const int64_t average = static_cast< int64_t >( ( m_AccumAverage * 0.9 ) + ( ( now - pre ).GetInt64Value() * 0.1 ) );
    m_AccumAverage = average;
    return ComputeValue( average );
}

//=======================================================================================
} // ::testns
//=======================================================================================
