﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <cstdlib>
#include <mutex>
#include <nn/os.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_IntrusiveList.h>
#include "testNs_MemoryAllocator.h"

#if defined( NN_SDK_BUILD_LIBRARY )
#include <nn/nn_SdkAssert.h>
#define TESTNS_INTERNAL_ASSERT NN_SDK_ASSERT
#else
#include <nn/nn_Assert.h>
#define TESTNS_INTERNAL_ASSERT NN_ASSERT
#endif

namespace testns {

//!--------------------------------------------------------------------------------------
//! @brief      任意オブジェクト型インスタンスのリサイクル機構を補助するヘルパーモジュールの基底クラスです。
//! @details    本クラスは抽象クラスです、事前のプールオブジェクトは準備されません。@n
//!             本クラスを継承し、派生先クラスにてプールコンテナの管理をお願いいたします。@n
//!             また、本クラスで管理するオブジェクト型は、BasedNodeクラスを継承してください。@n
//!             スレッドセーフです。
//!--------------------------------------------------------------------------------------
template< typename TObject >
class ObjectRecyclerBase : public nn::util::IntrusiveList< TObject, nn::util::IntrusiveListBaseNodeTraits< TObject > >
{
public:
    typedef TObject ObjectType;
    typedef nn::util::IntrusiveList< ObjectType, nn::util::IntrusiveListBaseNodeTraits< ObjectType > > CollectionImpl;

    //!--------------------------------------------------------------------------------------
    //! @brief      リサイクルオブジェクトベースノード。
    //! @details    本リサイクラで管理するオブジェクトは、本ノードクラスを継承しているものとして扱われます。
    //!--------------------------------------------------------------------------------------
    class BasedNode : public nn::util::IntrusiveListBaseNode< TObject >
    {
    public:
        //!--------------------------------------------------------------------------------------
        //! @brief      リサイクルを実施します。
        //! @details    本メソッドを呼び出す時は本クラスを継承した派生クラスインスタンスの運用が終了していることを保証し、
        //!             以後アクセスが発生しない事を保証しなければなりません。
        //!--------------------------------------------------------------------------------------
        void Recycle() NN_NOEXCEPT
        {
            ObjectRecyclerBase< TObject >* pOwnerRecycler;
            if ( nullptr != ( pOwnerRecycler = m_pOwnerRecycler ) )
            {
                pOwnerRecycler->Recycle( static_cast< TObject& >( *this ) );
            }
        }

    protected:
        //!--------------------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //!--------------------------------------------------------------------------------------
        explicit BasedNode( ObjectRecyclerBase< TObject >* pOwnerRecycler ) NN_NOEXCEPT
            : m_pOwnerRecycler( pOwnerRecycler )
        {
        }

    private:
        ObjectRecyclerBase< TObject >* const m_pOwnerRecycler;
    };

    //!--------------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!--------------------------------------------------------------------------------------
    explicit ObjectRecyclerBase() NN_NOEXCEPT : m_Lock( true )
    {
        nn::os::InitializeConditionVariable( &m_LockCondition );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //!--------------------------------------------------------------------------------------
    virtual ~ObjectRecyclerBase() NN_NOEXCEPT
    {
        nn::os::FinalizeConditionVariable( &m_LockCondition );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief      明示的終了処理です。
    //! @details    デストラクタ以外での終了処理が存在する場合は、本メソッドで処理します。
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void TryCanObtain() const NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_Lock ) > lock( m_Lock );
        return DoTryCanObtain( m_Collection );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void WaitCanObtain() const NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_Lock ) > lock( m_Lock );

        while ( false == DoTryCanObtain( m_Collection ) )
        {
            nn::os::WaitConditionVariable( &m_LockCondition, m_Lock.GetBase() );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    ObjectType* TryObtain() NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_Lock ) > lock( m_Lock );
        return DoTryObtain( m_Collection );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    ObjectType& Obtain() NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_Lock ) > lock( m_Lock );

        ObjectType* pResult;
        while ( nullptr == ( pResult = DoTryObtain( m_Collection ) ) )
        {
            nn::os::WaitConditionVariable( &m_LockCondition, m_Lock.GetBase() );
        }
        return *pResult;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void Recycle( ObjectType& object ) NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_Lock ) > lock( m_Lock );

        if ( true == DoRecycle( object ) )
        {
            nn::os::SignalConditionVariable( &m_LockCondition );
        }
    }

protected:
    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    virtual ObjectType* OnCreateNewObject() NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    virtual const bool CanCreateNewObject() const NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const bool DoTryCanObtain( const CollectionImpl& collection ) const NN_NOEXCEPT
    {
        return ( false == collection.empty() || CanCreateNewObject() );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    ObjectType* DoTryObtain( CollectionImpl& collection ) NN_NOEXCEPT
    {
        if ( false == collection.empty() )
        {
            ObjectType& result = collection.front();
            collection.pop_front();
            return &result;
        }
        return OnCreateNewObject();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const bool DoRecycle( ObjectType& object ) NN_NOEXCEPT
    {
        if ( false == object.IsLinked() )
        {
            m_Collection.push_back( object );
            return true;
        }
        return false;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE CollectionImpl& GetCollection() NN_NOEXCEPT
    {
        return m_Collection;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE nn::os::Mutex& GetMutex() NN_NOEXCEPT
    {
        return m_Lock;
    }

private:
    CollectionImpl                          m_Collection;
    mutable nn::os::Mutex                   m_Lock;
    mutable nn::os::ConditionVariableType   m_LockCondition;
};

//!--------------------------------------------------------------------------------------
//! @brief      ヒープからプールコンテナを取得するオブジェクトリサイクラ。
//! @details    初期化時に指定数の有限コンテナをヒープから確保します。
//!             有限コンテナを全て使い切るとnullptrが返されます。
//!--------------------------------------------------------------------------------------
template< typename TObject, const unsigned ReservedBlockCount = 16 >
class HeapObjectRecycler : public ObjectRecyclerBase< TObject >
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief 親型
    //!--------------------------------------------------------------------------------------
    typedef ObjectRecyclerBase< TObject > ParentType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンテナ型
    //!--------------------------------------------------------------------------------------
    typedef typename ParentType::ObjectType ObjectType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit HeapObjectRecycler() NN_NOEXCEPT
        : m_pReservedObjects( nullptr ), m_RemainFreeObject( ReservedBlockCount )
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!--------------------------------------------------------------------------------------
    ~HeapObjectRecycler() NN_NOEXCEPT NN_OVERRIDE
    {
        Finalize();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        void* pReservedMemory;
        if ( nullptr != ( pReservedMemory = m_pReservedObjects ) )
        {
            m_pReservedObjects = nullptr;
            ::aligned_free( pReservedMemory );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 生成
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        TESTNS_INTERNAL_ASSERT( nullptr == m_pReservedObjects, "This recycler has been initialized already." );
        void* const pReservedMemory = ::aligned_alloc( NN_ALIGNOF( ObjectType ), sizeof( ObjectType ) * ReservedBlockCount );
        m_pReservedObjects = reinterpret_cast< ObjectType* >( pReservedMemory );
        m_RemainFreeObject = ReservedBlockCount;
    }

protected:
    //!--------------------------------------------------------------------------------------
    //! @brief 新オブジェクト生成可否
    //!--------------------------------------------------------------------------------------
    virtual const bool CanCreateNewObject() const NN_NOEXCEPT NN_OVERRIDE
    {
        return ( m_RemainFreeObject > 0 && nullptr != m_pReservedObjects );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 新オブジェクト生成
    //!--------------------------------------------------------------------------------------
    virtual ObjectType* OnCreateNewObject() NN_NOEXCEPT NN_OVERRIDE
    {
        ObjectType* const pObjects = m_pReservedObjects;
        const unsigned n = m_RemainFreeObject;
        if ( n > 0 && nullptr != pObjects )
        {
            m_RemainFreeObject = n - 1;
            return new ( &pObjects[ n - 1 ] ) ObjectType( this );
        }
        return nullptr;
    }

private:
    ObjectType*     m_pReservedObjects;
    unsigned        m_RemainFreeObject;
};

//!--------------------------------------------------------------------------------------
//! @brief      クラスメンバに有限プールコンテナを確保するオブジェクトリサイクラ。
//! @details    有限コンテナを全て使い切るとnullptrが返されます。
//!--------------------------------------------------------------------------------------
template< typename TObject, const unsigned ReservedBlockCount = 16 >
class StaticObjectRecycler : public ObjectRecyclerBase< TObject >
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief 親型
    //!--------------------------------------------------------------------------------------
    typedef ObjectRecyclerBase< TObject > ParentType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンテナ型
    //!--------------------------------------------------------------------------------------
    typedef typename ParentType::ObjectType ObjectType;

    //!--------------------------------------------------------------------------------------
    //! @brief 確保ブロック型
    //!--------------------------------------------------------------------------------------
    typedef nn::util::TypedStorage< TObject, sizeof( TObject ), NN_ALIGNOF( TObject ) > ObjectStorageType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit StaticObjectRecycler() NN_NOEXCEPT : m_RemainFreeObject( ReservedBlockCount )
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 生成
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        m_RemainFreeObject = ReservedBlockCount;
    }

protected:
    //!--------------------------------------------------------------------------------------
    //! @brief 新オブジェクト生成可否
    //!--------------------------------------------------------------------------------------
    virtual const bool CanCreateNewObject() const NN_NOEXCEPT NN_OVERRIDE
    {
        return ( m_RemainFreeObject > 0 );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 新オブジェクト生成
    //!--------------------------------------------------------------------------------------
    virtual ObjectType* OnCreateNewObject() NN_NOEXCEPT NN_OVERRIDE
    {
        unsigned n;
        if ( ( n = m_RemainFreeObject ) > 0 )
        {
            m_RemainFreeObject = n - 1;
            return new ( &m_pReservedObjects[ n - 1 ] ) ObjectType( this );
        }
        return nullptr;
    }

private:
    ObjectStorageType   m_pReservedObjects[ ReservedBlockCount ];
    unsigned            m_RemainFreeObject;
};



//!--------------------------------------------------------------------------------------
//! @brief      メッセージレシーバークラス。
//! @details    メッセージスレッドからのコールバックを、任意のスレッド上で受信するためのメッセージレシーバーです。
//!--------------------------------------------------------------------------------------
template< const unsigned MessageQueueCapacity >
class MessageReceiver
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    MessageReceiver() NN_NOEXCEPT
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief      デストラクタ
    //!--------------------------------------------------------------------------------------
    virtual ~MessageReceiver() NN_NOEXCEPT
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 初期化
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        nn::os::InitializeMessageQueue( &m_Context, m_pMessage, MessageQueueCapacity );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了
    //!--------------------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT
    {
        nn::os::FinalizeMessageQueue( &m_Context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージキューコンテキスト取得
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE const nn::os::MessageQueueType& GetQueueContext() const NN_NOEXCEPT
    {
        return m_Context;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージキューサイズ
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE const int GetQueueCapacity() const NN_NOEXCEPT
    {
        return MessageQueueCapacity;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージキュー領域取得
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE uintptr_t* GetQueueBuffer() NN_NOEXCEPT
    {
        return m_pMessage;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信
    //!--------------------------------------------------------------------------------------
    virtual void Send( const uintptr_t message ) NN_NOEXCEPT
    {
        nn::os::SendMessageQueue( &m_Context, message );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信 ( 非待機 )
    //!--------------------------------------------------------------------------------------
    virtual bool TrySend( const uintptr_t message ) NN_NOEXCEPT
    {
        return nn::os::TrySendMessageQueue( &m_Context, message );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ受信
    //!--------------------------------------------------------------------------------------
    virtual void Receive( uintptr_t& receiveData ) NN_NOEXCEPT
    {
        nn::os::ReceiveMessageQueue( &receiveData, &m_Context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ受信 ( 非待機 )
    //!--------------------------------------------------------------------------------------
    virtual bool TryReceive( uintptr_t& receiveData ) NN_NOEXCEPT
    {
        return nn::os::TryReceiveMessageQueue( &receiveData, &m_Context );
    }

private:
    nn::os::MessageQueueType    m_Context;
    uintptr_t                   m_pMessage[ MessageQueueCapacity ];
};



//!--------------------------------------------------------------------------------------
//! @brief メッセージスレッドコンテキストインタフェース 前方宣言
//!--------------------------------------------------------------------------------------
class MessageThreadContext;

//!--------------------------------------------------------------------------------------
//! @brief メッセージコンテナ基底
//!--------------------------------------------------------------------------------------
class ThreadMessage
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief Runnableインタフェース
    //!--------------------------------------------------------------------------------------
    typedef void ( *RunnableFunctionType )( MessageThreadContext* pThread, const ThreadMessage* pMessage );

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit ThreadMessage() NN_NOEXCEPT : m_pContext( nullptr ), m_pRunnable( nullptr ), m_CancelLock( true ), m_Cancelled( false )
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void Initialize( RunnableFunctionType pRunnable, void* pUserContext ) NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_CancelLock ) > lock( m_CancelLock );
        m_pContext = pUserContext;
        m_pRunnable = pRunnable;
        m_Cancelled = false;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief キャンセル要求
    //!--------------------------------------------------------------------------------------
    void Cancel() NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_CancelLock ) > lock( m_CancelLock );
        m_Cancelled = true;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief キャンセル要求確認
    //!--------------------------------------------------------------------------------------
    const bool IsCancelled() const NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_CancelLock ) > lock( m_CancelLock );
        return m_Cancelled;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    RunnableFunctionType GetRunnable() const NN_NOEXCEPT
    {
        return m_pRunnable;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void* GetUserContext() const NN_NOEXCEPT
    {
        return m_pContext;
    }

private:
    void* volatile          m_pContext;
    RunnableFunctionType    m_pRunnable;
    mutable nn::os::Mutex   m_CancelLock;
    volatile bool           m_Cancelled;
};

//!--------------------------------------------------------------------------------------
//! @brief メッセージスレッドコンテキストインタフェース
//!--------------------------------------------------------------------------------------
class MessageThreadContext
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief      強制終了要求
    //! @details    処理中のメッセージ及び、メッセージキューにプール中の各メッセージへキャンセルを通達し
    //!             エントリポイントのメッセージループ終了メッセージを発行する。
    //!             各メッセージのキャンセルの扱いは、実装部に依存。
    //!--------------------------------------------------------------------------------------
    virtual void Finish() NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief スレッドコンテキスト取得
    //!--------------------------------------------------------------------------------------
    virtual const nn::os::ThreadType& GetThreadContext() const NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信
    //!--------------------------------------------------------------------------------------
    virtual ThreadMessage& Send( ThreadMessage& message ) NN_NOEXCEPT = 0;
};

//!--------------------------------------------------------------------------------------
//! @brief メッセージスレッド基底型
//!--------------------------------------------------------------------------------------
template< const unsigned MessageQueueCapacity >
class MessageThreadBase : public MessageReceiver< MessageQueueCapacity >, public MessageThreadContext
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief スタックサイズ単位型
    //!--------------------------------------------------------------------------------------
    typedef char StackUnitType;

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージレシーバー型
    //!--------------------------------------------------------------------------------------
    typedef MessageReceiver< MessageQueueCapacity > MessageReceiverType;

    //!--------------------------------------------------------------------------------------
    //! @brief デフォルトコンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit MessageThreadBase() NN_NOEXCEPT
        : m_QuitLock( true ), m_ActiveMessage( nullptr ), m_RequiredFinish( false )
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ
    //!--------------------------------------------------------------------------------------
    virtual ~MessageThreadBase() NN_NOEXCEPT
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief コンテキスト解放処理
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT
    {
        Finish();
        nn::os::WaitThread( &m_Context );
        nn::os::DestroyThread( &m_Context );
        MessageReceiverType::Finalize();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief スタックサイズ取得
    //!--------------------------------------------------------------------------------------
    virtual const size_t GetStackSize() const NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief スタック領域取得
    //!--------------------------------------------------------------------------------------
    virtual StackUnitType* GetStackRoot() NN_NOEXCEPT = 0;

    //!--------------------------------------------------------------------------------------
    //! @brief スレッドコンテキスト取得
    //!--------------------------------------------------------------------------------------
    virtual const nn::os::ThreadType& GetThreadContext() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Context;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief      強制終了要求
    //! @details    処理中のメッセージ及び、メッセージキューにプール中の各メッセージへキャンセルを通達し
    //!             エントリポイントのメッセージループ終了メッセージを発行する。
    //!             各メッセージのキャンセルの扱いは、実装部に依存。
    //!--------------------------------------------------------------------------------------
    virtual void Finish() NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard< decltype( m_QuitLock ) > lock( m_QuitLock );
        if ( false == m_RequiredFinish )
        {
            auto* pMessage = m_ActiveMessage;
            m_ActiveMessage = nullptr;
            m_RequiredFinish = true;
            OnFinishRequired( pMessage );
            MessageReceiverType::Send( 0U );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了要求確認
    //!--------------------------------------------------------------------------------------
    const bool IsFinishRequired() const NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_QuitLock ) > lock( m_QuitLock );
        return m_RequiredFinish;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信
    //!--------------------------------------------------------------------------------------
    virtual void Send( const uintptr_t message ) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard< decltype( m_QuitLock ) > lock( m_QuitLock );
        if ( false == m_RequiredFinish )
        {
            MessageReceiverType::Send( message );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信 ( 非待機 )
    //!--------------------------------------------------------------------------------------
    virtual bool TrySend( const uintptr_t message ) NN_NOEXCEPT NN_OVERRIDE
    {
        std::lock_guard< decltype( m_QuitLock ) > lock( m_QuitLock );
        return ( false == m_RequiredFinish ) ? MessageReceiverType::TrySend( message ) : false;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ送信
    //!--------------------------------------------------------------------------------------
    virtual ThreadMessage& Send( ThreadMessage& message ) NN_NOEXCEPT NN_OVERRIDE
    {
        Send( reinterpret_cast< const uintptr_t >( &message ) );
        return message;
    }

protected:
    //!--------------------------------------------------------------------------------------
    //! @brief      生成及び待機
    //! @param[in]  relativeAmount 優先度相対値。
    //! @details    本メソッドを呼び出したスレッド優先度から引数指定分の優先度変化を加えた優先度で新しいスレッドを生成します。
    //!--------------------------------------------------------------------------------------
    void InitializeWithRelativePriority( const int relativeAmount = 0 ) NN_NOEXCEPT
    {
        int requiredPriority = nn::os::GetThreadPriority( nn::os::GetCurrentThread() ) + relativeAmount;
        if ( nn::os::HighestThreadPriority > requiredPriority )
        {
            requiredPriority = nn::os::HighestThreadPriority;
        }
        else if ( nn::os::LowestThreadPriority < requiredPriority )
        {
            requiredPriority = nn::os::LowestThreadPriority;
        }
        Initialize( requiredPriority );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 生成及び待機
    //!--------------------------------------------------------------------------------------
    void Initialize( const int threadPriority = nn::os::DefaultThreadPriority ) NN_NOEXCEPT
    {
        StackUnitType* const pStackRoot = GetStackRoot();
        TESTNS_INTERNAL_ASSERT( nullptr != pStackRoot );
        m_ActiveMessage = 0U;
        m_RequiredFinish = false;
        MessageReceiverType::Initialize();
        nn::os::CreateThread( &m_Context, EntryPoint, this, pStackRoot, GetStackSize(), threadPriority );
        nn::os::StartThread( &m_Context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ受信通知
    //!--------------------------------------------------------------------------------------
    virtual void OnMessageReceived( ThreadMessage* pMessage ) NN_NOEXCEPT
    {
        if ( nullptr == pMessage )
        {
            return; // 終了要求
        }
        ThreadMessage::RunnableFunctionType pRunnable;
        if ( nullptr != ( pRunnable = pMessage->GetRunnable() ) )
        {
            pRunnable( this, pMessage );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了要求検出通知
    //! @details Finish() を呼び出したスレッド上で呼び出されます。
    //!--------------------------------------------------------------------------------------
    virtual void OnFinishRequired( ThreadMessage* pMessage ) NN_NOEXCEPT
    {
        if ( nullptr != pMessage )
        {
            pMessage->Cancel();
        }
    }

private:
    //!--------------------------------------------------------------------------------------
    //! @brief スレッドエントリ
    //!--------------------------------------------------------------------------------------
    static void EntryPoint( void* pArguments ) NN_NOEXCEPT
    {
        uintptr_t receiveData;
        MessageThreadBase* const pThis = static_cast< MessageThreadBase* >( pArguments );
        do
        {
            pThis->MessageReceiverType::Receive( receiveData );
            ThreadMessage* pMessage = reinterpret_cast< ThreadMessage* >( receiveData );

            pThis->SetActiveMessage( pMessage );
            pThis->OnMessageReceived( pMessage );
            pThis->SetActiveMessage( nullptr );
        } while ( 0U != receiveData );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief アクティブメッセージ設定
    //!--------------------------------------------------------------------------------------
    void SetActiveMessage( ThreadMessage* pMessage ) NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_QuitLock ) > lock( m_QuitLock );
        m_ActiveMessage = pMessage;
    }

    nn::os::ThreadType          m_Context;
    mutable nn::os::Mutex       m_QuitLock;
    ThreadMessage* volatile     m_ActiveMessage;
    volatile bool               m_RequiredFinish;   //!< 終了要求の有無
};



//!--------------------------------------------------------------------------------------
//! @brief メッセージスレッド基底( スタック, キュー確保版 )
//!
//! スタックはアドレスアライメントの制約( ThreadStackAlignment, GuardedStackAlignment )があり、
//! クラスメンバに持たせるとクラスパディングが無駄になる事から動的アロケータ方式にしています。
//!--------------------------------------------------------------------------------------
template< const size_t StackByteSize = nn::os::ThreadStackAlignment * 2, const int QueueBufferCapacity = 16 >
class MessageThread : public MessageThreadBase< QueueBufferCapacity >
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief 基底型
    //!--------------------------------------------------------------------------------------
    typedef MessageThreadBase< QueueBufferCapacity > ThreadBaseType;

    //!--------------------------------------------------------------------------------------
    //! @brief スタック単位型
    //!--------------------------------------------------------------------------------------
    typedef typename ThreadBaseType::StackUnitType StackUnitType;

    //!--------------------------------------------------------------------------------------
    //! @brief デフォルトコンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit MessageThread() NN_NOEXCEPT : m_pStack( nullptr ) {}

    //!--------------------------------------------------------------------------------------
    //! @brief 生成及び待機
    //!--------------------------------------------------------------------------------------
    void InitializeWithRelativePriority( const int relativeAmount = 0 ) NN_NOEXCEPT
    {
        TESTNS_INTERNAL_ASSERT( nullptr == m_pStack, "This thread has been initialized already." );
        m_pStack = reinterpret_cast< StackUnitType* >( ::aligned_alloc( nn::os::ThreadStackAlignment, StackByteSize ) );
        ThreadBaseType::InitializeWithRelativePriority( relativeAmount );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 生成及び待機
    //!--------------------------------------------------------------------------------------
    void Initialize( const int threadPriority = nn::os::DefaultThreadPriority ) NN_NOEXCEPT
    {
        TESTNS_INTERNAL_ASSERT( nullptr == m_pStack, "This thread has been initialized already." );
        m_pStack = reinterpret_cast< StackUnitType* >( ::aligned_alloc( nn::os::ThreadStackAlignment, StackByteSize ) );
        ThreadBaseType::Initialize( threadPriority );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        ThreadBaseType::Finalize();
        StackUnitType* pStackRoot;
        if ( nullptr != ( pStackRoot = m_pStack ) )
        {
            m_pStack = nullptr;
            ::aligned_free( pStackRoot );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief スタックサイズ取得
    //!--------------------------------------------------------------------------------------
    virtual const size_t GetStackSize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return StackByteSize;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief スタック領域取得
    //!--------------------------------------------------------------------------------------
    virtual StackUnitType* GetStackRoot() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pStack;
    }

private:
    StackUnitType*  m_pStack;
};

} // ::testns

#undef TESTNS_INTERNAL_ASSERT
