﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include "testNs_CommandOptions.h"

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <unordered_map>
#include <functional>   // std::hash
#include <string>       // std::hash< std::string >

#define LOG_OUT( ... )      NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace testns { namespace {

    //!--------------------------------------------------------------------------------------
    //! @brief コレクション
    //!--------------------------------------------------------------------------------------
    class OptionCollection : public std::unordered_map< std::string, std::string >
    {
        NN_DISALLOW_COPY( OptionCollection );
        NN_DISALLOW_MOVE( OptionCollection );

    public:
        typedef std::unordered_map< std::string, std::string > MapParentType;

        //!--------------------------------------------------------------------------------------
        //! @brief コンストラクタ
        //!--------------------------------------------------------------------------------------
        explicit OptionCollection( const size_t reserveCount = 8 ) NN_NOEXCEPT
        {
            MapParentType::reserve( reserveCount );
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 登録
        //!--------------------------------------------------------------------------------------
        void Entry( const char* const pArgument ) NN_NOEXCEPT
        {
            if ( nullptr != pArgument && '\0' != pArgument[ 0 ] )
            {
                std::string data( pArgument );
                std::string::size_type separatorIndex;
                if ( std::string::npos != ( separatorIndex = data.find_first_of( '=' ) ) && separatorIndex > 0 )
                {
                    std::string value, key( data.substr( 0, separatorIndex ) );
                    if ( ( separatorIndex + 1 ) < data.length() )
                    {
                        value = data.substr( separatorIndex + 1 );
                    }
                    ( *this )[ key ] = value;
                }
                else
                {
                    std::string value( "" );
                    ( *this )[ data ] = value;
                }
            }
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 検索
        //!--------------------------------------------------------------------------------------
        const bool Find( const char* const pKey, std::string& outMappedValue ) NN_NOEXCEPT
        {
            if ( nullptr != pKey && '\0' != pKey[ 0 ] )
            {
                std::string key( pKey );
                auto it = MapParentType::find( key );
                if ( it != MapParentType::end() )
                {
                    outMappedValue = it->second;
                    return true;
                }
            }
            return false;
        }
    };

    static OptionCollection* g_pOptionCollection;
}

//!--------------------------------------------------------------------------------------
//! @brief 起動コマンド引数コレクションの解放
//!--------------------------------------------------------------------------------------
void FinalizeCommandLineOptions() NN_NOEXCEPT
{
    OptionCollection* pCollection;
    if ( nullptr != ( pCollection = g_pOptionCollection ) )
    {
        delete pCollection;
        g_pOptionCollection = nullptr;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief 起動コマンド引数の検出
//! @details "--" で始まるキーワードを列挙します。
//! 空白を挟まず "=" で区切られた場合は値としてマッピングされます。
//!--------------------------------------------------------------------------------------
void InitializeCommandLineOptions() NN_NOEXCEPT
{
    FinalizeCommandLineOptions();

    const int argc = ::nnt::GetHostArgc();
    if ( argc > 1 )
    {
        OptionCollection* const pCollection = new OptionCollection();
        char** const argv = ::nnt::GetHostArgv();
        for ( int i = 1; i < argc; ++i )
        {
            pCollection->Entry( argv[ i ] );
        }
        g_pOptionCollection = pCollection;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief コマンド引数問い合わせ
//! @return 指定キーが存在した場合は true を返します。
//! @param[in]  pKey            オプションキー文字列( ex: "--testns_Option=Value" の場合、"--testns_Option" が該当します。 )
//! @param[in]  outBufferSize   pMappedValue の格納可能容量( byte )を指定します。
//! @param[out] pMappedValue    指定キーに紐づく値( ex: "--testns_Option=Value" の場合、"Value" が該当します。 )
//!--------------------------------------------------------------------------------------
const bool QueryCommandLineOptions( const char* const pKey, const size_t outBufferSize, char* const pMappedValue ) NN_NOEXCEPT
{
    std::string value;
    OptionCollection* pCollection;
    if ( nullptr != ( pCollection = g_pOptionCollection ) && pCollection->Find( pKey, value ) )
    {
        if ( nullptr != pMappedValue && outBufferSize > 0 )
        {
            const size_t n = ( value.size() < outBufferSize ) ? value.size() : ( outBufferSize - 1 );
            std::strncpy( pMappedValue, value.c_str(), n );
            pMappedValue[ n ] = '\0';
        }
        return true;
    }
    return false;
}


} // namespace testns
