﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted RebootTarget.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the wait seconds for a boot sequence complete.
    [string] $WaitBootSequence = "20",

    [string]
    $TargetName,

    [string]
    $TargetAddress
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\HostBridge"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"

if ([string]::IsNullOrEmpty($TargetAddress)) {
    if ([string]::IsNullOrEmpty($TargetName)) {
        throw "TargetAddress and TargetName not specified"
    } else {
        $TargetAddress = Get-TargetAddressFromSerial $TargetName
    }
}

# リブートを実行する
function Execution-Reboot( [string]$defSdkRoot, [string]$expectNsp )
{
    try {
        # ターゲットボックスへの接続要求
        Write-Host "start connection...:  $TargetAddress"
        Invoke-Expression "& `"${defSdkRoot}\Tools\CommandLineTools\ControlTarget.exe`" connect -t ${TargetAddress}"
        if ( $LastExitCode -eq 0 )
        {
            Write-Host "Connected. Wait a boot sequence complete."

            # 起動完了しログが安定しないと期待しない文字列がRunOnTargetから返されてしまうので、起動完了まで 30秒待ちます。
            Start-Sleep -s 30
            $result = Invoke-Expression "& `"${defSdkRoot}\Tools\CommandLineTools\RunOnTargetPrivate.exe`" run `"--target`" `"${TargetAddress}`" `"${expectNsp}`" `"--`" `"power`" `"reboot`""
            return [System.String] $result
        }
        else 
        {
            throw "Target connection failed. ExitCode = $LastExitCode"
        }
    } catch [Exception] {
        Write-ErrorRecord $_
    }
    return "0000000000000000 [FAILED]"
}

# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    Write-Host "============== Script configuration. =============="
    Write-Host "Platform        : ${Platform}"
    Write-Host "BuildType       : ${BuildType}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"

    # DevMenuCommandSystem.nsp の検索
    [System.String] $expectNsp = $(Search-AvailableDevMenuCommandSystem ${NintendoSdkRootPath} ${Platform} ${BuildType})
    Write-Host "DevMenuCommandSystem: ${expectNsp}"

    # リブートを実行
    $defResultDevMenuCom = $(Execution-Reboot ${NintendoSdkRootPath} ${expectNsp})
    
    # Interval for wait a boot sequence complete.
    Start-Sleep -s ${WaitBootSequence}

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode
