﻿<#
    .SYNOPSIS
        Private utilities module

    .DESCRIPTION
        This file defines functions for private operations
#>

# 基本パス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${MyScriptDirectoryPath}\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\SdkToolUtilities"

# XMLファイル出力( utf-8 )
function Out-WriteXml( [ref]$defXmlDoc, [string]$defFilePath )
{
    $xmlWriter = New-Object System.Xml.XmlTextWriter( ${defFilePath}, [System.Text.Encoding]::UTF8 )
    $xmlWriter.Formatting = [System.Xml.Formatting]::Indented
    $defXmlDoc.Value.Save( $xmlWriter )
    $xmlWriter.Close()
}
Export-ModuleMember -Function Out-WriteXml

# システムプログラムメタXML生成
# 返値: ファイルベース名
function Out-MakeSystemProgramMeta( [string]$defIntermediateWorkingDirectory, [string]$defSystemProgramId, [int]$defVersion, [string]$defArchSignature )
{
    # メタファイルベース名
    [string] $defFileNameBase = $(Get-ContentNspFileBaseName ${defSystemProgramId} ${defVersion} ${defArchSignature})
    [string] $defIntermediateFile = "${defIntermediateWorkingDirectory}\${defFileNameBase}.nmeta"
    if ( $(Test-Path ${defIntermediateFile}) )
    {
        Remove-Item ${defIntermediateFile} | Out-Null
    }

    # デフォルトは"arch64.lp64" とする
    [string] $defIs64BitInstruction = "True"
    [string] $defProcessAddressSpace = "AddressSpace64Bit"
    if ( $defArchSignature -eq "arch32.ilp32" )
    {
        [string] $defIs64BitInstruction = "False"
        [string] $defProcessAddressSpace = "AddressSpace32Bit"
    }
    elseif ( $defArchSignature -eq "arch64.ilp32" )
    {
        [string] $defIs64BitInstruction = "True"
        [string] $defProcessAddressSpace = "AddressSpace32Bit"
    }

    # ベースファイルを基に代入
    $defXmlDoc = [xml](Get-Content "${MyScriptDirectoryPath}\BaseSystemProgram.nmeta")
    $defXmlDoc.NintendoSdkMeta.Core.ProgramId = "0x${defSystemProgramId}"
    $defXmlDoc.NintendoSdkMeta.Core.Is64BitInstruction = ${defIs64BitInstruction}
    $defXmlDoc.NintendoSdkMeta.Core.ProcessAddressSpace = ${defProcessAddressSpace}
    $defXmlDoc.NintendoSdkMeta.SystemProgram.Id = "0x${defSystemProgramId}"
    $defXmlDoc.NintendoSdkMeta.SystemProgram.Version = [string] "${defVersion}"

    # 出力
    [ref] $refDoc = ${defXmlDoc}
    $(Out-WriteXml ${refDoc} ${defIntermediateFile})
    return ${defFileNameBase}
}
Export-ModuleMember -Function Out-MakeSystemProgramMeta

# システムアップデートメタXML生成
# 返値: ファイルベース名
function Out-MakeSystemUpdateMeta( [string]$defIntermediateWorkingDirectory, [string]$defSystemUpdateId, [int]$defVersion, [string]$defSystemProgramId, [string]$defSystemDataId )
{
    # メタファイルベース名
    [string] $defFileNameBase = $(Get-ContentNspFileBaseName ${defSystemUpdateId} ${defVersion} "")
    [string] $defIntermediateFile = "${defIntermediateWorkingDirectory}\${defFileNameBase}.nmeta"
    $(Remove-ExistFile ${defIntermediateFile})

    # ベースファイルを基に代入
    $defXmlDoc = [xml](Get-Content "${MyScriptDirectoryPath}\BaseSystemUpdate.nmeta")
    $defXmlDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.Id = [string] "0x${defSystemUpdateId}"
    $defXmlDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.Version = [string] "${defVersion}"

    # SystemProgram ディレクティブ
    if ( ${defSystemProgramId}.Length -gt 0 )
    {
        $defContentMeta = $defXmlDoc.CreateElement( "ContentMeta" )
        $defDocType = $defXmlDoc.CreateElement( "Type" )
        $defDocId = $defXmlDoc.CreateElement( "Id" )
        $defDocVersion = $defXmlDoc.CreateElement( "Version" )
        $defDocType.set_InnerText( "SystemProgram" )
        $defDocId.set_InnerText( [string] "0x${defSystemProgramId}" )
        $defDocVersion.set_InnerText( [string] "${defVersion}" )
        $defContentMeta.AppendChild( ${defDocType} ) | Out-Null
        $defContentMeta.AppendChild( ${defDocId} ) | Out-Null
        $defContentMeta.AppendChild( ${defDocVersion} ) | Out-Null
        $defXmlDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.AppendChild( ${defContentMeta} ) | Out-Null
    }

    # SystemData ディレクティブ
    if ( ${defSystemDataId}.Length -gt 0 )
    {
        $defContentMeta = $defXmlDoc.CreateElement( "ContentMeta" )
        $defDocType = $defXmlDoc.CreateElement( "Type" )
        $defDocId = $defXmlDoc.CreateElement( "Id" )
        $defDocVersion = $defXmlDoc.CreateElement( "Version" )
        $defDocType.set_InnerText( "SystemData" )
        $defDocId.set_InnerText( [string] "0x${defSystemDataId}" )
        $defDocVersion.set_InnerText( [string] "${defVersion}" )
        $defContentMeta.AppendChild( ${defDocType} ) | Out-Null
        $defContentMeta.AppendChild( ${defDocId} ) | Out-Null
        $defContentMeta.AppendChild( ${defDocVersion} ) | Out-Null
        $defXmlDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.AppendChild( ${defContentMeta} ) | Out-Null
    }

    # 出力
    [ref] $refDoc = ${defXmlDoc}
    $(Out-WriteXml ${refDoc} ${defIntermediateFile})
    return ${defFileNameBase}
}
Export-ModuleMember -Function Out-MakeSystemUpdateMeta

# ROPS submit 用入力XML生成
# 返値: ファイルパス
function Out-MakeRopsSubmitXml( [string]$defIntermediateWorkingDirectory, [string]$defMetaId, [int]$defReleaseVersion, [int]$defSubmitVersion )
{
    # ファイルベース名
    [string] $defFileNameBase =  "${defMetaId}_v${defReleaseVersion}-${defSubmitVersion}.submit"
    [string] $defIntermediateFile = "${defIntermediateWorkingDirectory}\${defFileNameBase}.xml"
    $(Remove-ExistFile ${defIntermediateFile})

    # ベースファイルを基に代入
    $defXmlDoc = [xml](Get-Content "${MyScriptDirectoryPath}\RopsSubmitBase.xml")
    $defXmlDoc.submit_request_entity.release_version = [string] "${defReleaseVersion}"
    $defXmlDoc.submit_request_entity.submit_version = [string] "${defSubmitVersion}"

    # 出力
    [ref] $refDoc = ${defXmlDoc}
    $(Out-WriteXml ${refDoc} ${defIntermediateFile})
    return ${defIntermediateFile}
}
Export-ModuleMember -Function Out-MakeRopsSubmitXml

# -----------------------------------------------------------------------
# ncm content database verify 用入力XMLオブジェクトノード生成
# -----------------------------------------------------------------------
function New-MakeMetaKeyCollectionNode( [ref]$defNodeDocRef, [string]$defId, [string]$defType, [string]$defVersion )
{
    $defId.Trim() | Out-Null
    $defType.Trim() | Out-Null
    $defVersion.Trim() | Out-Null
    [string] $defId = $(Select-ExpectStringMetaId ${defId})

    $defRoot = $defNodeDocRef.Value
    $defMetaKeyItem = $defRoot.CreateElement( "MetaKey" )
    $defMetaKeyItem.SetAttribute( "type", [string] "${defType}" ) | Out-Null
    $defMetaKeyItem.SetAttribute( "id", [string] "${defId}" ) | Out-Null
    $defMetaKeyItem.SetAttribute( "version", [string] "${defVersion}" ) | Out-Null
    return [System.Xml.XmlElement] ${defMetaKeyItem}
}

# -----------------------------------------------------------------------
# ncm content database verify 用入力XMLオブジェクト生成
# 返値: ファイルパス
# defMetaKeyCollection:
#   キー表現文字列書式 "${ID}:${VERSION}:${TYPE}"
#   ':' で分離します。
# -----------------------------------------------------------------------
function New-MakeMetaKeyCollectionXml( [string[]]$defMetaKeyCollection )
{
    [xml] $defXmlDoc = New-Object xml
    [ref] $refXmlDoc = ${defXmlDoc}
    $defDocRoot = $defXmlDoc.CreateElement( "MetaKeyCollection" )
    $defXmlDoc.AppendChild( ${defDocRoot} ) | Out-Null
    foreach( $defKeyString in $defMetaKeyCollection )
    {
        $defKeyValues = ${defKeyString} -csplit ":"
        if ( $defKeyValues.Length -gt 2 )
        {
            [string] $defId      = $defKeyValues[0]
            [string] $defVersion = $defKeyValues[1]
            [string] $defType    = $defKeyValues[2]
            $defMetaKeyItem = $(New-MakeMetaKeyCollectionNode ${refXmlDoc} ${defId} ${defType} ${defVersion})
            $defDocRoot.AppendChild( ${defMetaKeyItem} ) | Out-Null
        }
    }
    return $defXmlDoc
}
Export-ModuleMember -Function New-MakeMetaKeyCollectionXml

# -----------------------------------------------------------------------
# ncm content database verify 用入力XML生成
# 返値: ファイルパス
# defMetaKeyCollection:
#   キー表現文字列書式 "${ID}:${VERSION}:${TYPE}"
#   ':' で分離します。
# -----------------------------------------------------------------------
function Out-MakeMetaKeyCollectionXml( [string]$defIntermediateWorkingDirectory, [string]$defFileName, [string[]]$defMetaKeyCollection )
{
    # ファイルベース名
    [string] $defIntermediateFile = "${defIntermediateWorkingDirectory}\${defFileName}"
    $(Remove-ExistFile ${defIntermediateFile})

    # XMLオブジェクト生成
    [xml] $defXmlDoc = $(New-MakeMetaKeyCollectionXml ${defMetaKeyCollection})

    # 出力
    [ref] $refDoc = ${defXmlDoc}
    $(Out-WriteXml ${refDoc} ${defIntermediateFile})
    return ${defIntermediateFile}
}
Export-ModuleMember -Function Out-MakeMetaKeyCollectionXml

# -----------------------------------------------------------------------
# ncm content database verify 用入力XMLオブジェクト生成
# 返値: ファイルパス
# defSystemUpdateMetaFilePath:
#   AuthoringTool creatensp に渡す、SystemUpdateMetaのメタ情報を記載したメタXMLファイルパス
# -----------------------------------------------------------------------
function New-MakeMetaKeyCollectionXmlFromMetaXml( [string]$defSystemUpdateMetaFilePath )
{
    # 入力メタファイルの存在確認
    $(Test-ExistFile ${defSystemUpdateMetaFilePath})

    # 出力オブジェクト
    [xml] $defXmlDoc = New-Object xml
    [ref] $refXmlDoc = ${defXmlDoc}
    $defDocRoot = $defXmlDoc.CreateElement( "MetaKeyCollection" )
    $defXmlDoc.AppendChild( ${defDocRoot} ) | Out-Null

    # SystemUpdate : ContentMeta
    [xml] $defSourceMetaDoc = [xml](Get-Content ${defSystemUpdateMetaFilePath})
    [string] $defId = $defSourceMetaDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.Id
    [string] $defType = $defSourceMetaDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.Type
    [string] $defVersion = $defSourceMetaDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.Version
    $defMetaKeyItem = $(New-MakeMetaKeyCollectionNode ${refXmlDoc} ${defId} ${defType} ${defVersion})
    $defDocRoot.AppendChild( ${defMetaKeyItem} ) | Out-Null

    # SystemUpdate : ContentMeta : ContentMeta
    [array] $defSourceMetaNodes = $defSourceMetaDoc.NintendoSdkMeta.SystemUpdate.ContentMeta.ContentMeta
    ${defSourceMetaNodes} | % {
        [System.Xml.XmlElement] $defSourceMetaNode = $_
        [string] $defType    = $defSourceMetaNode.Type
        [string] $defVersion = $defSourceMetaNode.Version
        [string] $defId      = $defSourceMetaNode.Id
        $defMetaKeyItem = $(New-MakeMetaKeyCollectionNode ${refXmlDoc} ${defId} ${defType} ${defVersion})
        $defDocRoot.AppendChild( ${defMetaKeyItem} ) | Out-Null
    }
    return $defXmlDoc
}
Export-ModuleMember -Function New-MakeMetaKeyCollectionXmlFromMetaXml
