﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdlib>
#include <memory>

#include <nn/fs.h>
#include <nn/aoc.h>
#include <nn/aoc/aoc_SystemApi.h>

#include "testNs_Common.h"

namespace testns {

    //!--------------------------------------------------------------------------------------
    //! AddOnContent 関連のFSマウント系関数の Scoped クラス
    //!--------------------------------------------------------------------------------------
    class ScopedMountAoc
    {
        NN_DISALLOW_COPY( ScopedMountAoc );

    public:
        explicit ScopedMountAoc( const std::string& inMountName, nn::aoc::AddOnContentIndex inTargetIndex ) NN_NOEXCEPT
            : m_MountName( inMountName )
        {
            size_t cacheSize = 0;
            m_LastResult = nn::fs::QueryMountAddOnContentCacheSize( &cacheSize, inTargetIndex );
            if ( m_LastResult.IsFailure() )
            {
                // キャッシュサイズ読み込みで失敗
                return;
            }

            m_CacheBufferPtr.reset( new char[ cacheSize ] );
            // 追加コンテンツのマウント
            m_LastResult = nn::fs::MountAddOnContent( m_MountName.c_str(), inTargetIndex, m_CacheBufferPtr.get(), cacheSize );
            if ( m_LastResult.IsFailure() )
            {
                // マウントで失敗
                return;
            }
        }

        virtual ~ScopedMountAoc() NN_NOEXCEPT
        {
            if ( m_LastResult.IsSuccess() )
            {
                // 追加コンテンツのアンマウント
                nn::fs::Unmount( m_MountName.c_str() );
            }
        }

        nn::Result GetLastResult() const NN_NOEXCEPT
        {
            return m_LastResult;
        }

        bool IsAvailable() const NN_NOEXCEPT
        {
            return m_LastResult.IsSuccess();
        }

        const std::string MakeMountPath( const char* pFilePath ) const NN_NOEXCEPT
        {
            std::string result( m_MountName );
            return result.append( ":" ).append( pFilePath );
        }

    private:
        nn::Result m_LastResult;
        std::unique_ptr<char[]> m_CacheBufferPtr;
        std::string m_MountName;
    };

} // namespace testns
