﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Xml;

namespace CsTestAssistants
{
    //!----------------------------------------------------------------------------
    /// <summary>
    /// 64 bit値 構成識別子.
    /// </summary>
    /// <remarks>immutableクラスとして扱います.</remarks>
    //!----------------------------------------------------------------------------
    public sealed class ID64
    {
        /// <summary>
        /// 無効ID の実体値
        /// </summary>
        private const ulong INVALID_VALUE = 0;

        /// <summary>
        /// 無効ID
        /// </summary>
        public static readonly ID64 Invalid = new ID64( INVALID_VALUE );

        /// <summary>
        /// 実体64bit値
        /// </summary>
        public ulong Value { get; }

        /// <summary>
        /// 64bit値の16進数表記文字列
        /// </summary>
        /// <remarks>変換コストをコンストラクション時のみとするためのもの</remarks>
        private string m_StringValueX16 { get; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="value">識別子として扱う64bit値</param>
        public ID64( ulong value )
        {
            Value = value;
            m_StringValueX16 = value.ToString( "x16" );
        }

        /// <summary>
        /// コンストラクタ( 文字列指定 )
        /// </summary>
        /// <param name="value">数値化可能な入力文字列, radix = 16 の場合 "0x", "0X" を付与する事が可能.</param>
        /// <param name="radix">基数 ( 2, 8, 10, 16 )</param>
        /// <see cref="System.Convert.ToUInt64(string, int)"/>
        public ID64( string value, int radix = 16 )
            : this( System.Convert.ToUInt64( value, radix ) )
        {
        }

        public bool IsValid()
        {
            return ( INVALID_VALUE != Value );
        }

        /// <summary>
        /// 等価比較を行います.
        /// </summary>
        /// <param name="obj">比較対象オブジェクトインスタンス</param>
        /// <returns>
        /// 等価の場合 true が返されます.
        /// </returns>
        /// <see cref="System.NullReferenceException">自身がnullであれば呼び出しに失敗します.</see>
        public override bool Equals( object obj )
        {
            return ReferenceEquals( obj, this ) || ( null != obj && obj is ID64 && Value.Equals( ( ( ID64 )obj ).Value ) );
        }

        /// <summary>
        /// Hash値を返します.
        /// </summary>
        /// <returns>int 型の Hash 値.</returns>
        public override int GetHashCode()
        {
            return Value.GetHashCode();
        }

        /// <summary>
        /// 64bit値の16進数文字列を返します.
        /// </summary>
        /// <returns>64bit値の16進数文字列.</returns>
        public override string ToString()
        {
            return m_StringValueX16;
        }

        /// <summary>
        /// 等価比較を行います.
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        public static bool operator ==( ID64 lhs, ID64 rhs )
        {
            return ReferenceEquals( lhs, rhs ) || ( null != ( object )lhs && null != ( object )rhs && lhs.Value.Equals( rhs.Value ) );
        }

        /// <summary>
        /// 非等価比較を行います.
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        public static bool operator !=( ID64 lhs, ID64 rhs )
        {
            return !( lhs == rhs );
        }

        /// <summary>
        /// ID64 = ID64 + ID64
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator +( ID64 lhs, ID64 rhs )
        {
            return new ID64( lhs.Value + rhs.Value );
        }

        /// <summary>
        /// ID64 = ID64 + ulong
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator +( ID64 lhs, ulong rhs )
        {
            return new ID64( lhs.Value + rhs );
        }

        /// <summary>
        /// ID64 = ulong + ID64
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator +( ulong lhs, ID64 rhs )
        {
            return new ID64( lhs + rhs.Value );
        }

        /// <summary>
        /// ID64 = ID64 - ID64
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator -( ID64 lhs, ID64 rhs )
        {
            return new ID64( lhs.Value - rhs.Value );
        }

        /// <summary>
        /// ID64 = ID64 - ulong
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator -( ID64 lhs, ulong rhs )
        {
            return new ID64( lhs.Value - rhs );
        }

        /// <summary>
        /// ID64 = ulong - ID64
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        /// <see cref="System.NullReferenceException">比較オブジェクトがnullの場合に発生します.</see>
        public static ID64 operator -( ulong lhs, ID64 rhs )
        {
            return new ID64( lhs - rhs.Value );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// ID64依存コンテンツホルダーコンテナ。
        /// </summary>
        /// <typeparam name="TValue"></typeparam>
        //!----------------------------------------------------------------------------
        public class Holder<TValue>
        {
            public ID64 Identifier { get; }

            public TValue Value { get; set; }

            public Holder( ID64 id, TValue value )
            {
                Identifier = id;
                Value = value;
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// プロパティキーバリューコレクション型定義
    /// </summary>
    //!----------------------------------------------------------------------------
    public class PropertyCollection : Dictionary<string, string>
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="capacity"></param>
        public PropertyCollection( int capacity = 16 ) : base( capacity )
        {
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="other"></param>
        public PropertyCollection( PropertyCollection other ) : base( other )
        {
        }

        /// <summary>
        /// ToString
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            StringBuilder b = new StringBuilder( 128 );
            foreach ( var kv in this )
            {
                if ( 0 < b.Length )
                {
                    b.Append( ", " );
                }
                b.AppendFormat( "[ {0}={1} ]", kv.Key, kv.Value );
            }
            return b.ToString();
        }

        /// <summary>
        /// 指定のキーの値を追加/置換します。
        /// </summary>
        /// <param name="key">プロパティキー文字列</param>
        /// <param name="value">追加/置換する値文字列</param>
        /// <exception cref="System.ArgumentNullException">key が null です。</exception>
        public void PutValue( string key, string value )
        {
            if ( ContainsKey( key ) )
            {
                this[ key ] = value;
            }
            else
            {
                Add( key, value );
            }
        }

        /// <summary>
        /// 文字列としてキーの値を取得します。
        /// </summary>
        /// <param name="key">プロパティキー文字列</param>
        /// <returns>指定キーが存在しない場合は、null が返されます。それ以外は格納している値文字列を返します。</returns>
        public string GetValue( string key )
        {
            string outValue;
            return ( TryGetValue( key, out outValue ) ) ? outValue : null;
        }

        /// <summary>
        /// ID64値としてキーの値を取得します。
        /// </summary>
        /// <param name="key">プロパティキー文字列</param>
        /// <returns>指定キーが存在しない場合は、null が返されます。存在するがID64に変換できない場合は例外が発生します。</returns>
        /// <exception cref="System.FormatException">ID64に変換できない文字列です</exception>
        /// <exception cref="System.OverflowException">64bitの範囲を越える値を示す文字列です</exception>
        public ID64 GetValueAsID64( string key )
        {
            string outValue;
            return ( TryGetValue( key, out outValue ) ) ? new ID64( outValue ) : null;
        }

        /// <summary>
        /// 32bit符号なし整数値としてキーの値を取得します。
        /// </summary>
        /// <param name="key">プロパティキー文字列</param>
        /// <param name="defaultValue">キーが見つからない場合に返されるデフォルト値</param>
        /// <returns></returns>
        public uint GetValueAsUI32( string key, uint defaultValue = default( uint ) )
        {
            return GetValueAs( key, System.TypeCode.UInt32, defaultValue );
        }

        /// <summary>
        /// <see cref="System.TypeCode"/> で指定した型値としてキーの値を取得します。
        /// </summary>
        /// <typeparam name="T">値型</typeparam>
        /// <param name="key">プロパティキー文字列</param>
        /// <param name="typeCode">型コード</param>
        /// <param name="defaultValue">キーが見つからない場合に返されるデフォルト値</param>
        /// <returns></returns>
        public T GetValueAs<T>( string key, System.TypeCode typeCode, T defaultValue = default( T ) ) where T : struct
        {
            string outValue;
            return ( TryGetValue( key, out outValue ) ) ? ( T )System.Convert.ChangeType( outValue, typeCode ) : defaultValue;
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// コンテンツエンティティ構成
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class Content
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンテンツ種別
        /// </summary>
        /// <remarks>
        /// ncm_ContentType.h
        /// </remarks>
        //!----------------------------------------------------------------------------
        public enum Type : byte
        {
            Meta,
            Program,
            Data,
            Control,
            HtmlDocument,
            LegalInformation,
            DeltaFragment,
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// コンテンツメタ必須構成
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class ContentMeta
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンテンツメタ種別
        /// </summary>
        /// <remarks>
        /// ncm_ContentMetaTypes.h
        /// </remarks>
        //!----------------------------------------------------------------------------
        public enum Type : byte
        {
            Unknown = 0,
            SystemProgram,
            SystemData,
            SystemUpdate,
            BootImagePackage,
            BootImagePackageSafe,

            Application = 128,
            Patch,
            AddOnContent,
            Delta,
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// メタ情報構成パッケージ
        /// </summary>
        //!----------------------------------------------------------------------------
        public static class Contexture
        {
            //!----------------------------------------------------------------------------
            /// <summary>
            /// ベーシック構成
            /// </summary>
            /// <remarks>
            /// Type, Identifier, Version のメンバは必ず immutable( 不変 )にしてください。
            /// 上記メンバ以外の運用については制限しないため、sealed は宣言しません。
            /// </remarks>
            //!----------------------------------------------------------------------------
            public class Basic
            {
                /// <summary>
                /// コンテンツメタID。
                /// 本メンバが扱う値は派生先の実装依存です。
                /// 必ずしも <see cref="Type"/> メンバが示すコンテンツ種別に準ずるメタID値である保証はありません。
                /// </summary>
                public ID64 Identifier { get; }

                /// <summary>
                /// コンテンツバージョン。
                /// 本メンバが扱う値は派生先の実装依存です。
                /// 必ずしも <see cref="Type"/> メンバが示すコンテンツ種別に準ずるバージョン値である保証はありません。
                /// </summary>
                public int Version { get; }

                /// <summary>
                /// リリースバージョン。
                /// コンテンツバージョンの上位 16 bit です。
                /// </summary>
                public int ReleaseVersion
                {
                    get { return (Version >> 16); }
                }

                /// <summary>
                /// プライベートバージョン。
                /// コンテンツバージョンの下位 16 bit です。
                /// </summary>
                public int PrivateVersion
                {
                    get { return ( Version & 0xffff ); }
                }

                /// <summary>
                /// コンテンツ種別。
                /// 本メンバが扱う値は派生先の実装依存です。
                /// </summary>
                public Type Type { get; }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="id">コンテンツメタID</param>
                /// <param name="version">コンテンツバージョン</param>
                /// <param name="type">コンテンツ種別</param>
                public Basic( ID64 id, int version, Type type )
                {
                    Identifier = id;
                    Version = version;
                    Type = type;
                }

                /// <summary>
                /// コピーコンストラクタ
                /// </summary>
                /// <param name="other"></param>
                public Basic( Basic other )
                {
                    if ( null == other )
                    {
                        throw new System.ArgumentNullException( "Could not construct the new instance by the specified argument because the null object." );
                    }
                    Identifier = other.Identifier;
                    Version = other.Version;
                    Type = other.Type;
                }

                /// <summary>
                /// 等価比較を行います.
                /// </summary>
                /// <param name="obj">比較対象オブジェクトインスタンス</param>
                /// <returns>
                /// 等価の場合 true が返されます.
                /// </returns>
                /// <see cref="System.NullReferenceException">自身がnullであれば呼び出しに失敗します.</see>
                public override bool Equals( object obj )
                {
                    Basic o;
                    return ReferenceEquals( obj, this ) || (
                        null != obj && obj is Basic
                        && Identifier == ( o = ( Basic )obj ).Identifier
                        && Version == o.Version
                        && Type == o.Type
                    );
                }

                /// <summary>
                /// Hash値を返します.
                /// </summary>
                /// <returns>int 型の Hash 値.</returns>
                public override int GetHashCode()
                {
                    return ( Identifier.GetHashCode() ^ Version.GetHashCode() ^ Type.GetHashCode() );
                }

                /// <summary>
                /// ToString
                /// </summary>
                /// <returns></returns>
                public override string ToString()
                {
                    return string.Format( "{0} => [ id={1}, version={2}, type={3} ]", base.ToString(), Identifier, Version, Type );
                }
            }

            //!----------------------------------------------------------------------------
            /// <summary>
            /// nsp ( nca ) に含まれる、*.cnmt.xml レベルの詳細情報の格納を目的とした上位構成です。
            /// </summary>
            //!----------------------------------------------------------------------------
            public class Advance : Basic
            {
                /// <summary>
                /// プロパティコレクション用キー定義
                /// </summary>
                public enum PropertyKey : byte
                {
                    RequiredDownloadSystemVersion,
                    RequiredApplicationVersion,
                    RequiredSystemVersion,
                    ApplicationId,
                    PatchId,
                    Index,
                    Tag,
                }

                /// <summary>
                /// *.cnmt.xml に含まれる情報要素のKey/Valueマップ。
                /// キー対象は以下のようなエレメントディレクティブです。
                /// &lt;ApplicationId&gt;
                /// &lt;PatchId&gt;
                /// &lt;RequiredSystemVersion&gt;
                /// &lt;RequiredApplicationVersion&gt;
                /// &lt;RequiredDownloadSystemVersion&gt;
                /// など。
                /// 但し、対象のキーが必ず含まれる保証はありません。
                /// また、本パラメータは GetHashCode 及び Equals には反映されません。
                /// </summary>
                public PropertyCollection Properties { get; }

                /// <summary>
                /// *.cnmt.xml に含まれる &lt;Content&gt; 情報要素のプロパティコレクションリストです。
                /// リスト要素一つのプロパティコレクションに対するキー対象は以下のようなエレメントディレクティブです。
                /// &lt;Type&gt; => Value値候補は、Content.Type が該当します。
                /// &lt;Id&gt; => 16byte ContentId です。
                /// &lt;Size&gt; => エンティティサイズ( byte ) です。
                /// &lt;Hash&gt; => 32 byte Hash です。( 256 bit HASH )
                /// 但し、対象のキーが必ず含まれる保証はありません。
                /// また、本パラメータは GetHashCode 及び Equals には反映されません。
                /// </summary>
                public List<PropertyCollection> ContentProperties { get; }

                /// <summary>
                /// コンストラクタ
                /// </summary>
                /// <param name="id">コンテンツメタID</param>
                /// <param name="version">コンテンツバージョン</param>
                /// <param name="type">コンテンツ種別</param>
                public Advance( ID64 id, int version, Type type ) : base( id, version, type )
                {
                    Properties = new PropertyCollection();
                    ContentProperties = new List<PropertyCollection>( 8 );
                }

                /// <summary>
                /// コピーコンストラクタ
                /// </summary>
                /// <param name="other">コピー元のインスタンス</param>
                public Advance( Basic other ) : base( other )
                {
                    Properties = new PropertyCollection();
                    ContentProperties = new List<PropertyCollection>( 8 );
                }

                /// <summary>
                /// コピーコンストラクタ
                /// </summary>
                /// <param name="other">コピー元のインスタンス</param>
                public Advance( Advance other ) : base ( other )
                {
                    if ( null == other )
                    {
                        throw new System.ArgumentNullException( "Could not construct the new instance by the specified argument because the null object." );
                    }
                    Properties = new PropertyCollection( other.Properties );
                    ContentProperties = new List<PropertyCollection>( other.ContentProperties.Capacity );
                    foreach ( var prop in other.ContentProperties )
                    {
                        ContentProperties.Add( new PropertyCollection( prop ) );
                    }
                }
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// コンテンツ構成コンフィギュレーション
    /// </summary>
    //!----------------------------------------------------------------------------
    public class ContentConfiguration<TPatchContexture> : ContentMeta.Contexture.Basic, IEnumerable<TPatchContexture> where TPatchContexture : System.IComparable<TPatchContexture>
    {
        /// <summary>
        /// タイトルパッチ構成リスト
        /// </summary>
        /// <remarks>このインスタンスは外部へ出しません。</remarks>
        private List<TPatchContexture> m_Patches { get; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="id">タイトルメタID</param>
        /// <param name="version">タイトルバージョン</param>
        /// <param name="type">タイトル種別, Application, AddOnContent</param>
        /// <param name="initialPatchCapacity"></param>
        public ContentConfiguration( ID64 id, int version, ContentMeta.Type type, int initialPatchCapacity = 8 )
            : base( id, version, type )
        {
            m_Patches = new List<TPatchContexture>( initialPatchCapacity );
        }

        /// <summary>
        /// パッチ構成を追加します。
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public ContentConfiguration<TPatchContexture> AddPatch( TPatchContexture value )
        {
            // 追加時は必ず重複排他のため Distinct を行います.
            if ( false == m_Patches.Contains( value ) )
            {
                m_Patches.Add( value );
                m_Patches.Sort();   // 若いパッチ順に昇順ソート
            }
            return this;
        }

        public ContentConfiguration<TPatchContexture> AddPatches( params TPatchContexture[] values )
        {
            foreach( var value in values )
            {
                AddPatch( value );
            }
            return this;
        }


        public int Count()
        {
            return m_Patches.Count;
        }

        public bool Contains( TPatchContexture value )
        {
            return m_Patches.Contains( value );
        }

        public IEnumerator<TPatchContexture> GetEnumerator()
        {
            return m_Patches.GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// コンテンツ生成結果受信コンテナ.
    /// 1つの nsp ファイルが持つ情報を示します.
    /// </summary>
    /// <remarks>
    /// マルチコンテントメタ nsp の場合は、ベースインスタンスには代表( 最初に見つかった )のメタ情報が格納されます。
    /// それ以外の参照は <see cref="WithinContents"/> を参照します。
    /// 尚、<see cref="WithinContents"/> は代表メタも含まれるため、
    /// マルチコンテントメタ nsp かどうかの判断を行う場合は、<see cref="HasMultiContents"/> メソッドを利用してください。
    /// </remarks>
    //!----------------------------------------------------------------------------
    public class GeneratedContentResult : ContentMeta.Contexture.Advance, IEnumerable<ContentMeta.Contexture.Advance>
    {
        /// <summary>
        /// 本生成結果対象のnspに含まれるコンテントメタのコレクションです。
        /// </summary>
        public List<ContentMeta.Contexture.Advance> WithinContents { get; }

        /// <summary>
        /// 生成されたコンテンツの NSPファイルのフルパスを示します。
        /// </summary>
        public string NspPath { get; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="id"></param>
        /// <param name="version"></param>
        /// <param name="type"></param>
        /// <param name="path"></param>
        internal GeneratedContentResult( ID64 id, int version, ContentMeta.Type type, string path )
            : base( id, version, type )
        {
            NspPath = path;
            WithinContents = new List<ContentMeta.Contexture.Advance>( 8 );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="meta"></param>
        /// <param name="path"></param>
        internal GeneratedContentResult( ContentMeta.Contexture.Advance meta, string path )
            : base( meta )
        {
            NspPath = path;
            WithinContents = new List<ContentMeta.Contexture.Advance>( 8 );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="meta"></param>
        /// <param name="path"></param>
        internal GeneratedContentResult( ContentMeta.Contexture.Basic meta, string path )
            : base( meta )
        {
            NspPath = path;
            WithinContents = new List<ContentMeta.Contexture.Advance>( 8 );
        }

        /// <summary>
        /// ToString
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            return base.ToString() + $", [ NspPath=\"{NspPath}\" ]";
        }

        /// <summary>
        /// マルチコンテントメタnspとして生成されたか判断します。
        /// </summary>
        /// <returns>true の場合、本生成結果対象のnspに２個以上のコンテンツメタが含まれます。</returns>
        public bool HasMultiContents()
        {
            return WithinContents.Count > 1;
        }

        /// <summary>
        /// 列挙イテレータ支援
        /// </summary>
        /// <returns></returns>
        public IEnumerator<ContentMeta.Contexture.Advance> GetEnumerator()
        {
            return WithinContents.GetEnumerator();
        }

        /// <summary>
        /// 列挙イテレータ支援
        /// </summary>
        /// <returns></returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        public GeneratedContentResult CopyToDirectory(string path)
        {
            var dir = FileHelper.MakeDirectory(path);
            var dst = System.IO.Path.Combine(dir.FullName, System.IO.Path.GetFileName(NspPath));
            System.IO.File.Copy(NspPath, dst, true);
            return new GeneratedContentResult((ContentMeta.Contexture.Advance)this, dst);
        }

        /// <summary>
        /// 結果カタログ
        /// </summary>
        public class Catalog : List<GeneratedContentResult>
        {
            public Catalog( int initialCapacity = 16 ) : base( initialCapacity ) { }

            public Catalog( IEnumerable<GeneratedContentResult> collection ) : base( collection ) { }

            public List<string> ToNspPaths()
            {
                var result = new List<string>( Count );
                foreach( var value in this )
                {
                    result.Add( value.NspPath );
                }
                return result;
            }

            /// <summary>
            /// 指定のコンテンツメタIDを持つコンテンツコンテナを抽出します。
            /// ※現在、マルチコンテントメタには対応していません。
            /// </summary>
            /// <param name="expectIdentifier">探索対象コンテンツメタID</param>
            /// <returns>
            /// 見つかった場合は対象のコンテナインスタンス, 見つからない場合は、null が返されます。
            /// 厳密にはコンテナクラスの既定値( default( T ) )が返されますが、参照型想定のため nullと同意とします。
            /// </returns>
            public GeneratedContentResult FindFrom( ID64 expectIdentifier )
            {
                return Find( meta => { return ( meta.Identifier == expectIdentifier ); } );
            }

            /// <summary>
            /// 指定のコンテンツメタバージョンを持つコンテンツコンテナを抽出します。
            /// ※現在、マルチコンテントメタには対応していません。
            /// </summary>
            /// <param name="expectVersion">探索対象コンテンツメタバージョン( 32bit表現 )</param>
            /// <returns>
            /// 見つかった場合は対象のコンテナインスタンス, 見つからない場合は、null が返されます。
            /// 厳密にはコンテナクラスの既定値( default( T ) )が返されますが、参照型想定のため nullと同意とします。
            /// </returns>
            public GeneratedContentResult FindFromVersion( int expectVersion )
            {
                return Find( meta => { return ( meta.Version == expectVersion ); } );
            }

            /// <summary>
            /// 指定のコンテンツメタリリースバージョンを持つコンテンツコンテナを抽出します。
            /// ※現在、マルチコンテントメタには対応していません。
            /// </summary>
            /// <param name="expectReleaseVersion">探索対象コンテンツメタリリースバージョン( 上位16bit )
            /// 本引数のみ指定時のプライベートバージョン( 下位16bit ) は 0 として扱われます。
            /// </param>
            /// <param name="expectPrivateVersion">探索対象コンテンツメタプライベートバージョン( 下位16bit )</param>
            /// <returns>
            /// 見つかった場合は対象のコンテナインスタンス, 見つからない場合は、null が返されます。
            /// 厳密にはコンテナクラスの既定値( default( T ) )が返されますが、参照型想定のため nullと同意とします。
            /// </returns>
            public GeneratedContentResult FindFromReleaseVersion( int expectReleaseVersion, int expectPrivateVersion = 0 )
            {
                return FindFromVersion( expectReleaseVersion << 16 | ( expectPrivateVersion & 0xffff ) );
            }

            /// <summary>
            /// メタバージョンによる昇順ソートを行います。
            /// ソート対象のバージョンは32bitが利用されます。
            /// </summary>
            /// <returns>自身のインスタンス</returns>
            public Catalog SortByVersion()
            {
                Sort( ( lhs, rhs ) => { return lhs.Version.CompareTo( rhs.Version ); } );
                return this;
            }
        }

        /// <summary>
        /// メタタイプでカテゴライズされたカタログ
        /// ※現在、マルチコンテントメタには対応していません。
        /// </summary>
        public class TypeCategorizedCatalog : Dictionary<ContentMeta.Type, Catalog>
        {
            /// <summary>
            /// デフォルトコンストラクタ
            /// </summary>
            /// <param name="capacity"></param>
            public TypeCategorizedCatalog( int capacity = 16 ) : base( capacity )
            {
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="catalog"></param>
            public TypeCategorizedCatalog( Catalog catalog ) : base( catalog.Count )
            {
                foreach ( var c in catalog )
                {
                    Add( c );
                }
            }

            /// <summary>
            /// インスタンスが持つ、メタタイプごとのカタログを取得します。
            /// </summary>
            /// <param name="type">カタログが欲しいメタタイプ</param>
            /// <returns>対象メタタイプが存在しなければ null が返されます。</returns>
            public Catalog GetTypedCatalog( ContentMeta.Type type )
            {
                Catalog result;
                return ( TryGetValue( type, out result ) ) ? result : null;
            }

            /// <summary>
            /// コンテンツ追加
            /// </summary>
            /// <param name="content"></param>
            public TypeCategorizedCatalog Add( GeneratedContentResult content )
            {
                var type = content.Type;
                var values = GetTypedCatalog( type );
                if ( null == values )
                {
                    values = new Catalog();
                    this[ type ] = values;
                }
                values.Add( content );
                return this;
            }
        }

        /// <summary>
        /// タイトル( オーナーアプリケーション )単位のカテゴライズ。
        /// 値は依存関係を持つnsp( App, Patch, AoC ) のタイプ別マップ。
        /// MakeContents で生成した GeneratedContentResult.Catalog を入力として依存関係に基づいてグループ分け。
        /// </summary>
        public class TitleCategorizedCatalog : Dictionary<ID64, TypeCategorizedCatalog>
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="catalog">MakeContents で生成した GeneratedContentResult.Catalog インスタンス</param>
            public TitleCategorizedCatalog( Catalog catalog, bool sortByVersion = true ) : base( catalog.Count )
            {
                foreach ( var c in catalog )
                {
                    Add( c );
                }
                if ( sortByVersion )
                {
                    SortByVersion();
                }
            }

            /// <summary>
            /// 指定タイトルIDのメタタイプでカテゴライズされたカタログを取得します。
            /// 取得したカタログには、指定タイトルIDに依存するコンテンツのみが含まれます。
            /// </summary>
            /// <param name="ownerApplicationId">タイトルID( オーナーアプリケーションID )</param>
            /// <returns>対象IDが存在しなければ null が返されます。</returns>
            public TypeCategorizedCatalog GetTitleCatalog( ID64 ownerApplicationId )
            {
                TypeCategorizedCatalog result;
                return ( null != ownerApplicationId && TryGetValue( ownerApplicationId, out result ) ) ? result : null;
            }

            /// <summary>
            /// 指定タイトルIDのメタタイプでカテゴライズされたカタログを取得します。
            /// 取得したカタログには、指定タイトルIDに依存するコンテンツのみが含まれます。
            /// 本メソッドは、指定IDのカタログが見つからない場合、空のカタログを生成/登録して返します。
            /// </summary>
            /// <param name="ownerApplicationId"></param>
            /// <returns>指定IDのカタログが見つからない場合、空のカタログを生成/登録して返します。</returns>
            /// <exception cref="System.ArgumentNullException"><see cref="ownerApplicationId"/> が null です。</exception>
            public TypeCategorizedCatalog EnsureGetTitleCatalog( ID64 ownerApplicationId )
            {
                if ( null == ownerApplicationId )
                {
                    throw new System.ArgumentNullException();
                }
                var result = GetTitleCatalog( ownerApplicationId );
                if ( null == result )
                {
                    result = new TypeCategorizedCatalog();
                    this[ ownerApplicationId ] = result;
                }
                return result;
            }

            /// <summary>
            /// コンテンツ追加
            /// </summary>
            /// <param name="content"></param>
            public TitleCategorizedCatalog Add( GeneratedContentResult content )
            {
                ID64 key = null;
                switch ( content.Type )
                {
                    case ContentMeta.Type.Application:
                    {
                        key = content.Identifier;
                    }
                    break;
                    case ContentMeta.Type.Patch:
                    case ContentMeta.Type.AddOnContent:
                    {
                        key = content.Properties.GetValueAsID64( PropertyKey.ApplicationId.ToString() );
                    }
                    break;
                    default:
                    break;
                }
                if ( null != key )
                {
                    EnsureGetTitleCatalog( key ).Add( content );
                }
                return this;
            }

            /// <summary>
            /// 各タイトルのタイプ別コンテンツリストに対してバージョンによる昇順ソートを実施します。
            /// </summary>
            /// <returns>自身のインスタンス</returns>
            public TitleCategorizedCatalog SortByVersion()
            {
                foreach ( var title in this )
                {
                    var kvTitles = title.Value;
                    if ( null != kvTitles )
                    {
                        foreach ( var kvTypes in kvTitles )
                        {
                            var catalog = kvTypes.Value;
                            if ( null != catalog )
                            {
                                catalog.SortByVersion();
                            }
                        }
                    }
                }
                return this;
            }

            /// <summary>
            /// デバッグ用インスタンスコンディションダンプ
            /// </summary>
            /// <param name="b"></param>
            /// <returns></returns>
            public string Dump( StringBuilder b = null )
            {
                b = ( null != b ) ? b : new StringBuilder( 256 );
                b.Append( $"TitleCategorizedCatalog.KeyCountt( {Count} ) => {{{System.Environment.NewLine}" );
                foreach ( var title in this )
                {
                    b.Append( $"    Title[ {title.Key} ] {{{System.Environment.NewLine}" );
                    var kvTitles = title.Value;
                    if ( null != kvTitles )
                    {
                        foreach ( var kvTypes in kvTitles )
                        {
                            var catalog = kvTypes.Value;
                            if ( null != catalog )
                            {
                                foreach ( var content in catalog )
                                {
                                    b.Append( $"        Content[ TYPE={content.Type}, ID={content.Identifier}, VER={content.Version} ] {{{System.Environment.NewLine}" );
                                    foreach ( var kvProp in content.Properties )
                                    {
                                        b.Append( $"            {kvProp.Key}={kvProp.Value}{System.Environment.NewLine}" );
                                    }
                                    b.Append( $"        }}{System.Environment.NewLine}" );
                                }
                            }
                        }
                    }
                    b.Append( $"    }}{System.Environment.NewLine}" );
                }
                b.Append( $"}}{System.Environment.NewLine}" );
                return b.ToString();
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// オーサリングツール制御支援
    /// </summary>
    //!----------------------------------------------------------------------------
    public class AuthoringExecutor
    {
        public string ExecutorPath { get; }

        public string OceanKeyConfigPath { get; }

        public AuthoringExecutor()
        {
            ExecutorPath = SigloHelper.ToolPath.FindAuthoringTool();
            OceanKeyConfigPath = SigloHelper.ToolPath.FindAuthoringOceanKeyConfigXml();
            FileHelper.TestExistsFile( ExecutorPath );
            FileHelper.TestExistsFile( OceanKeyConfigPath );
        }

        /// <summary>
        /// 対象nspの cnmt.xml からコンテンツメタ構成を抜き取ります。
        /// </summary>
        /// <param name="intermediateRootPath">作業フォルダルートパス</param>
        /// <param name="targetNspPath">展開対象nspファイルフルパス</param>
        /// <returns></returns>
        public GeneratedContentResult ExtractContentMetaContexture( string intermediateRootPath, string targetNspPath )
        {
            // correct to full path string on the intermediateRootPath.
            intermediateRootPath = System.IO.Path.GetFullPath(
                string.IsNullOrEmpty( intermediateRootPath )
                    ? System.IO.Path.GetTempPath()
                    : intermediateRootPath
            );
            // create random temporary directory for working.
            string workDir = FileHelper.MakeTemporaryDirectory( intermediateRootPath ).FullName;

            try
            {
                // extract
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "extractnsp" );
                b.Append( " -o \"" ).Append( workDir ).Append( '\"' );
                b.Append( " \"" ).Append( targetNspPath ).Append( '\"' );
                b.Append( " --keyconfig \"" ).Append( OceanKeyConfigPath ).Append( '\"' );

                // run the AuthoringTool.
                if ( 0 != CommandLineExecutor.ExecuteOnProcess( ExecutorPath, b.ToString() ) )
                {
                    FileHelper.RemoveDirectory( workDir );
                    throw new UnexpectFailureException( "AuthoringTool extract operation failed." );
                }

                // *.cnmt.xml を捜索
                string[] files = System.IO.Directory.GetFileSystemEntries( workDir, @"*.cnmt.xml" );
                if ( null == files || 0 == files.Length )
                {
                    FileHelper.RemoveDirectory( workDir );
                    throw new UnexpectFailureException( "Could not found the *.cnmt.xml." );
                }

                // xml から抽出
                GeneratedContentResult result = null;
                foreach ( var file in files )
                {
                    Log.WriteLine( "Extract meta from [ {0} ].", file );
                    XmlDocument xmlDoc = new XmlDocument();
                    xmlDoc.Load( file );
                    var ids = xmlDoc.SelectNodes( "//ContentMeta/Id" );
                    var types = xmlDoc.SelectNodes( "//ContentMeta/Type" );
                    var versions = xmlDoc.SelectNodes( "//ContentMeta/Version" );
                    if ( ids.Count > 0 && types.Count > 0 && versions.Count > 0 )
                    {
                        var id = new ID64( ids[ 0 ].InnerText.Trim() );
                        var ver = System.Convert.ToInt32( versions[ 0 ].InnerText.Trim() );
                        var type = CsExtension.ToEnumValue( types[ 0 ].InnerText.Trim(), ContentMeta.Type.Unknown );

                        // advance property collection.
                        var meta = new ContentMeta.Contexture.Advance( id, ver, type );
                        CollectContentProperty( meta.ContentProperties, xmlDoc );
                        CollectExtensionProperty( meta.Properties, xmlDoc );
                        if ( null == result )
                        {
                            // 一つ目は代表メタとしてリストからも同じインスタンスを見るようにする.
                            meta = result = new GeneratedContentResult( meta, targetNspPath );
                        }
                        result.WithinContents.Add( meta );
                    }
                }
                if ( null != result )
                {
                    Log.WriteLine( "Detect valid meta contexture, {0}.", result );
                    return result;
                }
            }
            finally
            {
                FileHelper.RemoveDirectory( workDir );
            }
            throw new UnexpectFailureException( "The `*.cnmt.xml` file was invalid formatted." );
        }

        /// <summary>
        /// 対象nspの cnmt.xml 中のベーシック外メタプロパティを抽出し、コレクションに追加します。
        /// </summary>
        /// <param name="outCollection"></param>
        /// <param name="xmlDoc"></param>
        private static void CollectExtensionProperty( PropertyCollection outCollection, XmlDocument xmlDoc )
        {
            foreach ( var e in System.Enum.GetNames( typeof( ContentMeta.Contexture.Advance.PropertyKey ) ) )
            {
                var pnode = xmlDoc.SelectNodes( $"//ContentMeta/{e}" );
                if ( pnode.Count > 0 )
                {
                    outCollection.PutValue( e, pnode[ 0 ].InnerText.Trim() );
                }
            }
        }

        /// <summary>
        /// 対象nspの cnmt.xml 中の &lt;Content&gt; 要素を抽出し、構成プロパティコレクションをリストに追加します。
        /// </summary>
        /// <param name="outCollection"></param>
        /// <param name="xmlDoc"></param>
        private static void CollectContentProperty( List<PropertyCollection> outCollection, XmlDocument xmlDoc )
        {
            var nodes = xmlDoc.SelectNodes( $"//ContentMeta/Content" );
            for ( int i = 0; i < nodes.Count; ++i )
            {
                var property = new PropertyCollection( 4 );
                var node = nodes[ i ];
                var keys = node.ChildNodes;
                for ( int keyIndex = 0; keyIndex < keys.Count; ++keyIndex )
                {
                    var keyNode = keys[ keyIndex ];
                    property.PutValue( keyNode.Name, keyNode.InnerText.Trim() );
                }
                outCollection.Add( property );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// パッチコマンド支援
        /// </summary>
        //!----------------------------------------------------------------------------
        public class Patch : AuthoringExecutor
        {
            /// <summary>
            /// makepatch コマンド実施時に付与する追加オプションのフラグ指定。
            /// </summary>
            public enum FlagOption : uint
            {
                None = 0,
                WithDelta = 1 << 1,
                MergeDelta = 1 << 2,

                UseDelta = WithDelta | MergeDelta,
            }

            /// <summary>
            /// アプリケーションを作成するときに指定した、desc ファイル。
            /// </summary>
            public string ApplicationDescFilePath { get; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="applicationDescFilePath">アプリケーションを作成するときに指定した、desc ファイルを指定します。</param>
            public Patch( string applicationDescFilePath ) : base()
            {
                FileHelper.TestExistsFile( applicationDescFilePath );
                ApplicationDescFilePath = applicationDescFilePath;
            }

            /// <summary>
            /// パッチ生成
            /// </summary>
            /// <param name="outPatchNspPath">パッチNSPの出力先ファイルフルパス</param>
            /// <param name="original">オリジナルアプリケーションNSPファイルフルパス</param>
            /// <param name="current">パッチ作成対象アプリケーションNSPファイルフルパス</param>
            /// <param name="previous">直前のパッチNSPファイルフルパス</param>
            /// <param name="flags">makepatch コマンド時のオプショナルフラグ指定</param>
            /// <returns>自身参照を返します</returns>
            /// <exception cref="UnexpectFailureException">プロセス実行に失敗した場合スローされます。</exception>
            public Patch Make( string outPatchNspPath, string original, string current, string previous, FlagOption flags = FlagOption.None )
            {
                StringBuilder b = new StringBuilder( 512 );
                b.Append( "makepatch" );
                b.Append( " -o \"" ).Append( outPatchNspPath ).Append( '\"' );
                b.Append( " --desc \"" ).Append( ApplicationDescFilePath ).Append( '\"' );
                b.Append( " --original \"" ).Append( original ).Append( '\"' );
                b.Append( " --current \"" ).Append( current ).Append( '\"' );
                if ( false == string.IsNullOrEmpty( previous ) )
                {
                    b.Append( " --previous \"" ).Append( previous ).Append( '\"' );

                    if ( flags.HasFlag( FlagOption.WithDelta ) )
                    {
                        b.Append( " --with-delta" );
                    }
                    if ( flags.HasFlag( FlagOption.MergeDelta ) )
                    {
                        b.Append( " --merge-delta" );
                    }
                }
                b.Append( " --keyconfig \"" ).Append( OceanKeyConfigPath ).Append( '\"' );

                // run the AuthoringTool.
                if ( 0 != CommandLineExecutor.ExecuteOnProcess( ExecutorPath, b.ToString() ) )
                {
                    throw new UnexpectFailureException( "AuthoringTool makepatch operation failed." );
                }
                FileHelper.TestExistsFile( outPatchNspPath );
                return this;
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// テスト用アプリケーションコンテンツ支援.
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class TestApplication
    {
        public static string GetExecutionToolPath()
        {
            return SigloHelper.ToolPath.FindTools( System.IO.Path.Combine( "CommandLineTools", "MakeTestApplication", "MakeTestApplication.exe" ) );
        }

        /// <summary>
        /// パッチコンテンツ生成要求パラメタ構成基底クラス
        /// </summary>
        /// <typeparam name="TVersionType">パッチバージョンを表現する型</typeparam>
        public class PatchContexture<TVersionType> : System.IComparable<PatchContexture<TVersionType>>
            where TVersionType : System.IComparable<TVersionType>
        {
            public AuthoringExecutor.Patch.FlagOption PatchOption { get; set; }
            public TVersionType Version { get; }

            /// <summary>
            /// <typeparamref name="TVersionType"/> との暗黙変換演算子
            /// </summary>
            /// <param name="contexture"></param>
            public static implicit operator TVersionType( PatchContexture<TVersionType> contexture )
            {
                return contexture.Version;
            }

            /// <summary>
            /// <typeparamref name="TVersionType"/> との暗黙変換演算子
            /// </summary>
            /// <param name="version"></param>
            public static implicit operator PatchContexture<TVersionType>( TVersionType version )
            {
                return new PatchContexture<TVersionType>( version );
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="initialVersion">このコンテクスチャインスンタスが示すパッチバージョン</param>
            /// <param name="option">makepatch コマンド時のオプショナルフラグ指定</param>
            public PatchContexture( TVersionType initialVersion, AuthoringExecutor.Patch.FlagOption option = AuthoringExecutor.Patch.FlagOption.None )
            {
                Version = initialVersion;
                PatchOption = option;
            }

            /// <summary>
            /// 等価比較を行います.
            /// </summary>
            /// <param name="obj">比較対象オブジェクトインスタンス</param>
            /// <returns>
            /// 等価の場合 true が返されます.
            /// </returns>
            /// <see cref="System.NullReferenceException">自身がnullであれば呼び出しに失敗します.</see>
            public override bool Equals( object obj )
            {
                PatchContexture<TVersionType> o;
                return ReferenceEquals( obj, this ) || (
                    null != obj && obj is PatchContexture<TVersionType>
                    && Version.Equals( ( o = ( PatchContexture<TVersionType> )obj ).Version )
                );
            }

            /// <summary>
            /// Hash値を返します.
            /// </summary>
            /// <returns>int 型の Hash 値.</returns>
            public override int GetHashCode()
            {
                return Version.GetHashCode();
            }

            /// <summary>
            ///
            /// </summary>
            /// <param name="other"></param>
            /// <returns></returns>
            int System.IComparable<PatchContexture<TVersionType>>.CompareTo( PatchContexture<TVersionType> other )
            {
                if ( null == other )
                {
                    throw new System.ArgumentNullException( "Could not construct the new instance by the specified argument because the null object." );
                }
                return Version.CompareTo( other.Version );
            }
        }

        /// <summary>
        /// コンテンツ生成要求パラメタ構成基底クラス
        /// </summary>
        /// <typeparam name="TPatchVersionType"></typeparam>
        public abstract class GenerateParameter<TPatchVersionType> : ContentConfiguration<PatchContexture<TPatchVersionType>>
            where TPatchVersionType : System.IComparable<TPatchVersionType>
        {
            /// <summary>
            /// アーキテクチャプラットフォーム ( ビルドアーキテクチャ判定用 )。
            /// 現在意味なし, MakeTestApplicationは arch64.lp64固定。
            /// </summary>
            /// <see cref="SigloHelper.Configuration.DefaultPlatform"/>
            public string ContentPlatform { get; set; }

            /// <summary>
            /// 生成対象コンテンツのの必須バージョン値の指定要求。
            /// MakeTestApplication 仕様に基づいて、対象コンテンツタイプによって以下のような運用変動があります。
            ///   - 設定されるメタ要素
            ///     - Application/Patch:  NintendoSdkMeta/Application/RequiredSystemVersion
            ///     - AddOnContent:       NintendoSdkMeta/AddOnContent/RequiredApplicationReleaseVersion
            ///   - 指定すべきバージョン形式
            ///     - Application/Patch:  32bit Unified Version 形式 ( 上位16bit release, 下位16bit private )
            ///     - AddOnContent:       16bit Release Version 形式
            /// 既定値は 0 です。
            /// </summary>
            public uint RequiredVersion { get; set; }

            /// <summary>
            /// 外部鍵で暗号化された製品化コンテンツの生成要求。
            /// デフォルトは false です。
            /// </summary>
            public bool TicketEncryption { get; set; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="id"></param>
            /// <param name="version"></param>
            /// <param name="type"></param>
            /// <param name="initialPatchCapacity"></param>
            public GenerateParameter( ID64 id, int version, ContentMeta.Type type, int initialPatchCapacity = 8 )
                : base( id, version, type, initialPatchCapacity )
            {
                RequiredVersion = 0;
                TicketEncryption = false;
                ContentPlatform = SigloHelper.Configuration.DefaultPlatform;
            }

            public GenerateParameter<TPatchVersionType> ChangePlatform( string platform )
            {
                ContentPlatform = platform;
                return this;
            }

            public GenerateParameter<TPatchVersionType> ChangeTicketEncryption( bool enable )
            {
                TicketEncryption = enable;
                return this;
            }

            public GenerateParameter<TPatchVersionType> ChangeRequiredVersion( uint requiredVersion )
            {
                RequiredVersion = requiredVersion;
                return this;
            }

            /// <summary>
            /// MakeTestApplication.exe 用コマンドラインオプションパラメタ生成。
            /// </summary>
            /// <remarks>
            /// 新しいパラメタ追加時の注意。
            ///
            /// パッチ生成時にも同パラメタ反映が必要な場合、
            /// 継承先の生成対象コンフィギュレーションの CreatePatchParameter の実装でパラメタ複製を行ってください。
            /// </remarks>
            /// <returns></returns>
            public virtual List<string> CreateToolArguments( int initialListCapacity = 16 )
            {
                List<string> options = new List<string>( initialListCapacity );
                options.Add( "--type " + Type );
                options.Add( "--id 0x" + Identifier );
                options.Add( "--ver " + Version );
                if ( TicketEncryption )
                {
                    options.Add( "--ticket" );
                }
                if ( RequiredVersion > 0 )
                {
                    options.Add( $"--required-version {RequiredVersion}" );
                }
                return options;
            }

            /// <summary>
            /// nspファイルベース名作成。
            /// </summary>
            /// <remarks>
            /// 設定コンフィグ状態に応じたベース名を作成して返します。
            /// ベース名には拡張子( ".nsp" )は含まれません。
            /// </remarks>
            /// <param name="b">文字列生成用バッファ, 意図的な再利用バッファなどがある場合に指定します。最初に Clear() されます。
            /// null の場合、自動で確保します。</param>
            /// <returns>
            /// 設定コンフィグ状態に応じたベース名文字列。
            /// </returns>
            public virtual string CreateNspFileBaseName( StringBuilder b = null )
            {
                b = ( null != b ) ? b.Clear() : new StringBuilder( 128 );
                b.Append( Identifier ).Append( "_v" ).Append( Version );
                string platform = ContentPlatform;
                if ( false == string.IsNullOrEmpty( platform ) )
                {
                    b.Append( '.' ).Append( SigloHelper.Configuration.GetArchitectSignature( platform ) );
                }
                return b.ToString();
            }

            /// <summary>
            /// nspファイルベース名作成。( patch 用 )
            /// </summary>
            /// <param name="patchProperty">対象となる patch contexture property</param>
            /// <param name="b">文字列生成用バッファ, 意図的な再利用バッファなどがある場合に指定します。最初に Clear() されます。
            /// null の場合、自動で確保します。</param>
            /// <returns>設定コンフィグ状態に応じた patch 用ベース名文字列。</returns>
            public string CreateNspFileBaseName( TPatchVersionType patchProperty, StringBuilder b = null )
            {
                b = ( null != b ) ? b.Clear() : new StringBuilder( 128 );
                return b.Append( Identifier ).Append( "_v" ).Append( CreateFileNameVersion( patchProperty ) ).Append( ".patch" ).ToString();
            }

            /// <summary>
            /// パッチ作成用パラメタ複製。
            /// </summary>
            /// <remarks>
            /// 自身をおりじなるとして、必要なパラメタのみを複製します。
            /// </remarks>
            /// <param name="newPatchProperty">新しく割り当てるパッチプロパティ</param>
            /// <returns>複製されたパラメータ</returns>
            public abstract GenerateParameter<TPatchVersionType> CreatePatchParameter( TPatchVersionType newPatchProperty );

            /// <summary>
            /// ファイル名に利用可能な、指定Patchコンテクスチャのバージョンを表現した文字列を生成します。
            /// </summary>
            /// <param name="property">Patchコンテクスチャプロパティ</param>
            /// <returns>ファイル名に利用可能な、指定Patchコンテクスチャのバージョンを表現した文字列</returns>
            protected string CreateFileNameVersion( TPatchVersionType property )
            {
                return property.ToString();
            }
        }

        /// <summary>
        /// アプリケーション用コンフィギュレーション
        /// </summary>
        /// <remarks>
        /// 生成するパッチはバージョンで指定します。
        /// パッチバージョンは、アプリケーションのバージョンよりも大きいバージョン値での構成にしてください。
        /// </remarks>
        public class ApplicationParameter : GenerateParameter<int>
        {
            /// <summary>
            /// 生成するテストアプリケーションのダミーROMデータのサイズ。
            /// 指定可能な最大値は 9,223,372,036,854,775,807 (≒9エクサByte)。(Int64の最大値)
            /// </summary>
            public ulong Size { get; set; }

            /// <summary>
            /// MakeTestApplication の --code オプション指定。
            /// デフォルトは null による未使用です。
            /// MakeTestApplication が提供するデフォルトコードに依存する機能とは共存できません。( SmallCode, RequestNetwork etc )
            /// </summary>
            public string CodePath { get; set; }

            /// <summary>
            /// MakeTestApplication の --small-code 指定。
            /// テストアプリケーションの元となる実行ファイルに小さいサイズのコード(scode ディレクトリ)を使用します。
            /// デフォルトは false です。
            /// </summary>
            public bool UseSmallCode { get; set; }

            /// <summary>
            /// MakeTestApplication の --network-request 指定。
            /// アプリの起動直後にネットワーク接続要求を自動的に実行させるオプションです。既定値は false です。
            /// ※現状では UseSmallCode(--small-code) 指定時には、--network-request は有効になりません。
            /// </summary>
            public bool RequestNetwork { get; set; }

            /// <summary>
            /// MakeTestApplication の --runtime-install-aoc 指定。
            /// アプリのAoc動的コミット機能を有効にするオプションです。既定値は false です。
            /// </summary>
            public bool RuntimeInstallAoc { get; set; }

            /// <summary>
            /// OnCreatePatchParameterから呼び出されるメソッドを指定できます。
            /// 各引数は以下の通りです。
            /// 第一引数: 既定仕様で複製されたパラメータコンテナ。
            /// 第二引数: 複製元パラメータコンテナ。
            /// </summary>
            public System.Action<ApplicationParameter, ApplicationParameter> OnChangePatchParameter { get; set; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="id"></param>
            /// <param name="version"></param>
            /// <param name="size"></param>
            /// <param name="initialPatchCapacity"></param>
            public ApplicationParameter( ID64 id, int version, ulong size = 4, int initialPatchCapacity = 8 )
                : base( id, version, ContentMeta.Type.Application, initialPatchCapacity )
            {
                Size = ( size > 0 ) ? size : 4;
                UseSmallCode = false;
                RequestNetwork = false;
                RuntimeInstallAoc = false;
                CodePath = null;
                OnChangePatchParameter = null;
            }

            /// <summary>
            /// MakeTestApplication.exe 用コマンドラインオプションパラメタ生成。
            /// </summary>
            /// <returns></returns>
            public override List<string> CreateToolArguments(int initialListCapacity = 16)
            {
                var options = base.CreateToolArguments(initialListCapacity);
                options.Add("--size " + Size);
                if ( UseSmallCode )
                {
                    options.Add( "--small-code" );
                }
                if ( RequestNetwork )
                {
                    options.Add( "--network-request" );
                }
                if ( RuntimeInstallAoc )
                {
                    options.Add( "--runtime-install-aoc" );
                }
                if ( false == string.IsNullOrEmpty( CodePath ) && System.IO.Directory.Exists( CodePath ) )
                {
                    options.Add( $"--code {CodePath}" );
                }
                return options;
            }

            /// <summary>
            /// パッチ用アプリケーションパラメタ複製。
            /// </summary>
            /// <param name="newPatchProperty">新しく割り当てるパッチプロパティ</param>
            /// <returns>複製されたパラメータ</returns>
            public override GenerateParameter<int> CreatePatchParameter( int newPatchProperty )
            {
                var result = new ApplicationParameter( Identifier, newPatchProperty, Size );
                result.UseSmallCode = UseSmallCode;
                result.RequestNetwork = RequestNetwork;
                result.RequiredVersion = RequiredVersion;
                result.ChangePlatform( ContentPlatform );
                if ( null != OnChangePatchParameter )
                {
                    OnChangePatchParameter( result, this ); //  ?.invoke は vs2015から
                }
                return result;
            }

            /// <summary>
            /// MakeTestApplication --code オプション指定
            /// </summary>
            /// <param name="codePath"> /code フォルダパス</param>
            /// <returns></returns>
            public ApplicationParameter ChangeCodePath( string codePath )
            {
                CodePath = codePath;
                return this;
            }
        }

        /// <summary>
        /// AddOn 用コンフィギュレーション
        /// </summary>
        public class AddonParameterBase : GenerateParameter<int>
        {
            /// <summary>
            /// 生成するテストアプリケーションのダミーROMデータのサイズ。
            /// 指定可能な最大値は 9,223,372,036,854,775,807 (≒9エクサByte)。(Int64の最大値)
            /// </summary>
            public ulong Size { get; set; }

            /// <summary>
            /// タイトルインデックス ( 1～4095 )
            /// </summary>
            public ushort BeginIndex { get; }
            public ushort EndIndex { get; }

            /// <summary>
            ///
            /// </summary>
            /// <param name="ownerApplicationId"></param>
            /// <param name="version"></param>
            /// <param name="index"></param>
            /// <param name="size"></param>
            /// <param name="initialPatchCapacity"></param>
            public AddonParameterBase(ID64 ownerApplicationId, int version, ushort beginIndex = 1, ushort endIndex = 1, ulong size = 4, int initialPatchCapacity = 8)
                : base(ownerApplicationId, version, ContentMeta.Type.AddOnContent, initialPatchCapacity)
            {
                BeginIndex = beginIndex;
                EndIndex = endIndex;
                Size = (size > 0) ? size : 4;
            }

            /// <summary>
            /// MakeTestApplication.exe 用コマンドラインオプションパラメタ生成。
            /// </summary>
            /// <returns></returns>
            public override List<string> CreateToolArguments( int initialListCapacity = 16 )
            {
                var options = base.CreateToolArguments( initialListCapacity );
                if ( false == TicketEncryption )
                {
                    options.Add( "--no-ticket" );
                }
                options.Add( "--size " + Size );
                options.Add( "--index " + BeginIndex );
                options.Add( "--index " + EndIndex);
                return options;
            }

            /// <summary>
            ///
            /// </summary>
            /// <param name="b"></param>
            /// <returns></returns>
            public override string CreateNspFileBaseName( StringBuilder b = null )
            {
                b = ( null != b ) ? b.Clear() : new StringBuilder( 128 );
                return b.Append( Identifier ).Append( "_v" ).Append( Version ).Append( "_addon" ).Append( BeginIndex.ToString( "x3" ) ).ToString();
            }

            /// <summary>
            /// Addon用アプリケーションパラメタ複製。
            /// </summary>
            /// <param name="newPatchProperty">新しく割り当てるパッチプロパティ</param>
            /// <returns>複製されたパラメータ</returns>
            public override GenerateParameter<int> CreatePatchParameter( int newPatchProperty )
            {
                var result = new AddonParameterBase( Identifier, newPatchProperty, BeginIndex, EndIndex, Size );
                return result.ChangePlatform( ContentPlatform );
            }
        }

        /// <summary>
        /// AddOn 用コンフィギュレーション
        /// </summary>
        public class AddonParameter : AddonParameterBase
        {
            public class IndexRange : System.Tuple<ushort, ushort>
            {
                public ushort Begin { get { return Item1; } }
                public ushort End { get { return Item2; } }
                public IndexRange( ushort begin, ushort end ) : base( begin , end ) {}
            }

            /// <summary>
            ///
            /// </summary>
            /// <param name="ownerApplicationId"></param>
            /// <param name="version"></param>
            /// <param name="index"></param>
            /// <param name="size"></param>
            /// <param name="initialPatchCapacity"></param>
            public AddonParameter(ID64 ownerApplicationId, int version, ushort index = 1, ulong size = 4, int initialPatchCapacity = 8)
                : base(ownerApplicationId, version, index, index, size, initialPatchCapacity)
            {
            }

            public AddonParameter(ID64 ownerApplicationId, int version, IndexRange indexes, ulong size = 4, int initialPatchCapacity = 8)
                : base(ownerApplicationId, version, indexes.Begin, indexes.End, size, initialPatchCapacity)
            {
            }
        }


        /// <summary>
        /// AddOn 用コンフィギュレーション
        /// </summary>
        public class MultipleAddonParameter : AddonParameterBase
        {
            /// <summary>
            ///
            /// </summary>
            /// <param name="ownerApplicationId"></param>
            /// <param name="version"></param>
            /// <param name="index"></param>
            /// <param name="size"></param>
            /// <param name="initialPatchCapacity"></param>
            public MultipleAddonParameter(ID64 ownerApplicationId, int version, ushort beginIndex = 1, ushort endIndex = 1, ulong size = 4, int initialPatchCapacity = 8)
                : base(ownerApplicationId, version, beginIndex, endIndex, size, initialPatchCapacity)
            {
            }
        }


        /// <summary>
        /// MakeTestApplication を用いたデフォルトアプリの作成を行います。
        /// </summary>
        /// <remarks>
        /// 異常発生時は例外が発生します。
        /// 並列で呼び出せるように、作業フォルダはランダムに生成されます。
        /// </remarks>
        /// <param name="nspOutputDirectory">生成nspの出力先フォルダフルパス</param>
        /// <param name="appId">アプリケーションID</param>
        /// <param name="version">アプリケーションバージョン( ReleaseVersion )</param>
        /// <param name="platform">現状未使用, MakeTestApplicationは arch64.lp64固定</param>
        /// <returns>生成した nsp のフルパス</returns>
        public static string MakeOneAsStandard( string nspOutputDirectory, ID64 appId, int version, string platform = SigloHelper.Configuration.DefaultPlatform )
        {
            var parameter = new ApplicationParameter( appId, version );
            return MakeOneAsStandard( nspOutputDirectory, parameter.ChangePlatform( platform ) ).NspPath;
        }

        /// <summary>
        /// MakeTestApplication を用いた標準構成コンテンツの作成を行います。
        /// パッチコレクション指定があってもパッチは作成されません。
        /// </summary>
        /// <remarks>
        /// 異常発生時は例外が発生します。
        /// 並列で呼び出せるように、作業フォルダはランダムに生成されます。
        /// </remarks>
        /// <param name="nspOutputDirectory">Nsp出力先ディレクトリ</param>
        /// <param name="parameter">コンテンツ生成要求パラメタ構成</param>
        /// <returns>生成されたコンテンツ情報</returns>
        public static GeneratedContentResult MakeOneAsStandard( string nspOutputDirectory, GenerateParameter<int> parameter )
        {
            // correct to full path string on the nspOutputDirectory.
            nspOutputDirectory = System.IO.Path.GetFullPath( nspOutputDirectory );

            // create random temporary directory for MakeTestApplication working.
            string workDir = FileHelper.MakeTemporaryDirectory( nspOutputDirectory ).FullName;

            // create the file base name of output nsp file.
            string fileBaseName = parameter.CreateNspFileBaseName();
            string nspPath = System.IO.Path.Combine( nspOutputDirectory, fileBaseName+ ".nsp" );

            // arguments of MakeTestApplication.
            StringBuilder b = new StringBuilder( 512 );
            b.Append( " --output-file-name " ).Append( fileBaseName );
            b.Append( " -o \"" ).Append( nspOutputDirectory ).Append( '\"' );
            b.Append( " --work-directory \"" ).Append( workDir ).Append( '\"' );

            List<string> options = parameter.CreateToolArguments();
            if ( null != options && options.Count > 0 )
            {
                b.Append( ' ' ).Append( string.Join( " ", options ) );
            }

            // run the MakeTestApplication.
            if ( 0 != CommandLineExecutor.ExecuteOnProcess( GetExecutionToolPath(), b.ToString() ) )
            {
                throw new UnexpectFailureException( "MakeTestApplication operation failed." );
            }
            FileHelper.TestExistsFile( nspPath );
            return new AuthoringExecutor().ExtractContentMetaContexture( nspOutputDirectory, nspPath );
        }

        /// <summary>
        /// MakeContents 内部用
        /// </summary>
        /// <param name="outCatalog"></param>
        /// <param name="nspOutputDirectory"></param>
        /// <param name="parameter"></param>
        /// <param name="patchGenerator"></param>
        /// <returns></returns>
        private static GeneratedContentResult.Catalog MakeContents(
            GeneratedContentResult.Catalog outCatalog,
            string nspOutputDirectory,
            GenerateParameter<int> parameter,
            AuthoringExecutor.Patch patchGenerator )
        {
            // correct to full path string on the nspOutputDirectory.
            nspOutputDirectory = System.IO.Path.GetFullPath( nspOutputDirectory );

            // make base original content.
            var originalNsp = MakeOneAsStandard( nspOutputDirectory, parameter );

            // アップロード候補追加
            outCatalog.Add( originalNsp );

            // make patches
            // パッチはバージョン値リストとして渡され、各バージョンごとに順次、
            //      1. 対象バージョンのMakeTestApplication生成.
            //      2. makepatch で patch nsp 生成.
            // バージョン値リストの昇順保証は ContentConfiguration::AddPatch で実施済とする.
            string contentPlatform = parameter.ContentPlatform;
            int originalAppVersion = parameter.Version;
            int previousPatchVersion = originalAppVersion;
            string previousPatchFile = null;
            foreach ( var currentPatch in parameter )
            {
                Log.WriteLine( "makepatch on [ {0} ]", parameter.Identifier );

                if ( currentPatch.Version <= previousPatchVersion )
                {
                    Log.WriteLine( "Warnings: specified patch version are lesser than previous version." );
                    continue;
                }

                // パッチ対象アプリケーション作成 ( アップロード対象外 )
                // v0 が --ticket 付与してるなら --ticket 付与する必要あるか？ → なくても動くはず。(坂井先生談)
                // もし --ticket 必要なら GenerateParameter引数型の MakeOneAsStandard を利用する。
                var currentNsp = MakeOneAsStandard( nspOutputDirectory, parameter.CreatePatchParameter( currentPatch.Version ) ).NspPath;

                // パッチ nsp 生成
                var outPatchNspPath = System.IO.Path.Combine( nspOutputDirectory, parameter.CreateNspFileBaseName( currentPatch.Version ) ) + ".nsp";
                patchGenerator.Make( outPatchNspPath, originalNsp.NspPath, currentNsp, previousPatchFile, currentPatch.PatchOption );

                // 直前パッチ情報更新
                previousPatchVersion = currentPatch.Version;
                previousPatchFile = outPatchNspPath;

                // アップロード候補追加
                outCatalog.Add( patchGenerator.ExtractContentMetaContexture( nspOutputDirectory, outPatchNspPath ) );
            }
            return outCatalog;
        }

        /// <summary>
        /// MakeTestApplication を用いたコンテンツ作成を行います。
        /// パッチコレクション指定があればパッチが作成されます。
        /// </summary>
        /// <param name="nspOutputDirectory">Nsp出力先ディレクトリ</param>
        /// <param name="parameter">生成コンテンツ構成</param>
        /// <returns>
        /// 生成されたnspファイルへのパスリストを返します。
        /// パッチ対象バージョンのアプリケーションnspは含まれません。
        /// </returns>
        public static GeneratedContentResult.Catalog MakeContents( string nspOutputDirectory, GenerateParameter<int> parameter )
        {
            // make result.
            var nspObjects = new GeneratedContentResult.Catalog( parameter.Count() );

            // TestApplication desc file path.
            string descFile = System.IO.Path.Combine( SigloHelper.Path.GetProgramsRoot(), "Iris", "Resources", "SpecFiles", "Application" ) + ".desc";
            return MakeContents( nspObjects, nspOutputDirectory, parameter, new AuthoringExecutor.Patch( descFile ) );
        }

        /// <summary>
        /// MakeTestApplication を用いたコンテンツ作成を行います。
        /// パッチコレクション指定があればパッチが作成されます。
        /// </summary>
        /// <param name="nspOutputDirectory">Nsp出力先ディレクトリ</param>
        /// <param name="parameters">生成コンテンツ構成リスト</param>
        /// <returns>
        /// 生成されたnspファイルへのパスリストを返します。
        /// パッチ対象バージョンのアプリケーションnspは含まれません。
        /// </returns>
        public static GeneratedContentResult.Catalog MakeContents( string nspOutputDirectory, List<GenerateParameter<int>> parameters )
        {
            // Count the nsp files to be creating.
            int expectNspCount = parameters.Count;
            foreach( var parameter in parameters )
            {
                expectNspCount += parameter.Count();
            }
            // make result.
            var nspObjects = new GeneratedContentResult.Catalog( expectNspCount );

            // TestApplication desc file path.
            string descFile = System.IO.Path.Combine( SigloHelper.Path.GetProgramsRoot(), "Iris", "Resources", "SpecFiles", "Application" ) + ".desc";
            var patchGenerator = new AuthoringExecutor.Patch( descFile );

            foreach ( var parameter in parameters )
            {
                MakeContents( nspObjects, nspOutputDirectory, parameter, patchGenerator );
            }
            return nspObjects;
        }

        /// <summary>
        /// 入力されたコンテンツ構成パラメータに基づいて既にnspが存在する場合にコンテンツ生成結果受信コンテナオブジェクトを作成します。
        /// nspが存在しない場合は、例外が発生します。
        /// パッチコレクションは無視されます。
        /// アップロードや再生成などを行わないシーケンスでの情報再構築用のヘルパAPIです。
        /// </summary>
        /// <param name="authoring">nsp展開用オーサリングツールヘルパ</param>
        /// <param name="nspOutputDirectory">Nspファイル存在先ディレクトリ</param>
        /// <param name="parameter">生成済対象コンテンツ構成パラメータ</param>
        /// <param name="usePatchVersion">パッチコンテンツ希望時の希望パッチバージョン
        /// 0 以下の場合、パッチは対象外として扱われ、parameter の代表構成に基づいてオブジェクト生成を行います。</param>
        /// <returns>コンテンツ生成結果受信コンテナオブジェクト</returns>
        public static GeneratedContentResult MakeFromExistsContents( AuthoringExecutor authoring, string nspOutputDirectory, GenerateParameter<int> parameter, int usePatchVersion = 0 )
        {
            var baseFileName = ( usePatchVersion > 0 ) ? parameter.CreateNspFileBaseName( usePatchVersion ) : parameter.CreateNspFileBaseName();
            var nspPath = System.IO.Path.Combine( nspOutputDirectory, baseFileName ) + ".nsp";
            FileHelper.TestExistsFile( nspPath );
            authoring = ( null != authoring ) ? authoring : new AuthoringExecutor();
            return authoring.ExtractContentMetaContexture( nspOutputDirectory, nspPath );
        }

        /// <summary>
        /// 入力されたコンテンツ構成パラメータに基づいて既にnspが存在する場合にコンテンツ生成結果受信コンテナオブジェクトを作成します。
        /// nspが存在しない場合は、例外が発生します。
        /// パッチコレクションの内容も考慮してnspを検索します。
        /// アップロードや再生成などを行わないシーケンスでの情報再構築用のヘルパAPIです。
        /// </summary>
        /// <param name="authoring">nsp展開用オーサリングツールヘルパ</param>
        /// <param name="nspOutputDirectory">Nspファイル存在先ディレクトリ</param>
        /// <param name="parameters">生成済対象コンテンツ構成パラメータ</param>
        /// <returns>
        /// 生成されたnspファイルへのパスリストを返します。
        /// パッチ対象バージョンのアプリケーションnspは含まれません。
        /// </returns>
        public static GeneratedContentResult.Catalog MakeFromExistsContents( AuthoringExecutor authoring, string nspOutputDirectory, List<GenerateParameter<int>> parameters )
        {
            // Count the nsp files to be creating.
            int expectNspCount = parameters.Count;
            foreach ( var parameter in parameters )
            {
                expectNspCount += parameter.Count();
            }
            // make result.
            var nspObjects = new GeneratedContentResult.Catalog( expectNspCount );

            foreach ( var parameter in parameters )
            {
                nspObjects.Add( MakeFromExistsContents( authoring, nspOutputDirectory, parameter ) );
                foreach ( var patchVersion in parameter )
                {
                    nspObjects.Add( MakeFromExistsContents( authoring, nspOutputDirectory, parameter, patchVersion.Version ) );
                }
            }
            return nspObjects;
        }
    }
}
