﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        コンテンツ配信と BGNUP を一緒に開始してスリープ ---> 即半起床後にダウンロードタスクが完了するか
        のテストを行います。

    .DESCRIPTION
        Setup-TargetEnvironment, Setup-TestApplication, Setup-SystemUpdate の事前実行が必要です。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [switch]
    $AgingTest
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath

# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義など
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"
. "$ScriptDirectory\Functions.ps1"


# --------------------------------------------------------------------------
# テスト
# --------------------------------------------------------------------------
$TestSuccess = $false

try
{
    # 不具合調査用にデバイス ID を表示させる
    Write-InfoLog "Device ID = $(Get-TargetDeviceId $Target)"

    # NUP が必要であることを確認
    Test-SystemUpdateIsRequired $Target

    # テストアプリ購入
    # DTL 更新の通知とダウンロードの開始を待つ
    Start-DemoDownload $Target $TestApplicationId

    # BGNUP 開始
    # NUP ダウンロードの開始を待つ
    Start-SystemUpdateDownload $Target

    for ($i = 0; ; ++$i)
    {
        # 寝る
        Invoke-TargetSleep $Target

        # 何度かスリープさせてから起こす
        if ($AgingTest)
        {
            Wait-SleepCycle $Target -SleepCount 1
            Invoke-TargetWakeForAging $Target
        }
        else
        {
            Wait-SleepCycle $Target
            Invoke-TargetWake $Target
        }

        # アプリはダウンロードされているか、バージョンは想定通りか
        $AppDownloadComplete = Test-ApplicationDownloadComplete $Target $TestApplicationId $TestApplicationV0Version

        # 本体更新はダウンロードされているか
        $NupDownloadComplete = Test-SystemUpdateDownloadComplete $Target

        if ($AppDownloadComplete -and $NupDownloadComplete)
        {
            break
        }
        else
        {
            Write-InfoLog "Download hasn't completed yet, but maybe because of some temporary network failures. Will enter sleep again."
        }
    }

    # ダウンロードされた NUP を適用
    Invoke-DevMenuCommandSystem $Target systemupdate,apply-downloaded

    # アプリがちゃんと起動するか確認
    Invoke-RunOnTarget $Target "--failure-timeout",10,$TestApplicationId

    Write-SuccessLog "TEST SUCCESS"
    $TestSuccess = $true
}
catch
{
    Write-FailureLog "TEST FAILURE"
    throw
}
finally
{
    # スリープから復帰しないことがあるので、テストに成功していないならリセットをかける
    if (-not $TestSuccess)
    {
        Invoke-TargetReset $Target
    }

    Teardown-DemoDownload $Target $TestApplicationId
    Teardown-SystemUpdate $Target
}
