﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

function Start-DemoDownload
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string]
        $ApplicationId,

        [int]
        $StartDownloadTimeout = 60
    )

    # アプリ購入
    Invoke-DevMenuCommandSystem -Target $Target -Arguments shop,download-demo,0,--id,$ApplicationId -RetryWithTargetReset

    # DTL 更新の通知とダウンロードの開始を待つ
    $PurchaseDate = Get-Date
    for ($i = 0; ; ++$i)
    {
        $ApplicationView = Get-ApplicationView $Target $ApplicationId
        # TODO: まれに、この時点でダウンロードが完了していることがあるので、その場合はテスト成功としたい
        # たとえば、1 回目の download-demo で開発機との通信が切れて失敗したが購入処理は完了していて、
        # 2 回目の download-demo が成功するまでの間にダウンロードが完了してしまった場合
        if ($ApplicationView -and $ApplicationView.isDownloading)
        {
            break
        }

        $Diff = (Get-Date) - $PurchaseDate
        if ($Diff.TotalSeconds -gt $StartDownloadTimeout)
        {
            $DTLCheckStartDate = Get-Date
            while ($true)
            {
                $DownloadTaskListData = Get-DownloadTaskListData $Target

                if ($DownloadTaskListData -and $DownloadTaskListData.tasks.Length -gt 0)
                {
                    break
                }
                else
                {
                    $Diff = (Get-Date) - $DTLCheckStartDate
                    if ($Diff.TotalSeconds -ge 300)
                    {
                        break
                    }
                }
            }
            throw "App downloading doesn't start in $StartDownloadTimeout seconds after purchasing demo app $ApplicationId."
        }
    }
}

function Teardown-DemoDownload
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string]
        $ApplicationId,

        [int]
        $TrialCount
    )

    Invoke-DevMenuCommandSystem $Target shop,delete-all-rights,0 -TrialCount $TrialCount
    Invoke-DevMenuCommandSystem $Target shop,unlink-device-all -TrialCount $TrialCount

    # FIXME: TrialCount に連動して Timeout を設定するべきではない
    Wait-ServerDownloadTaskListCleared $Target -Timeout ($TrialCount * 60)

    Invoke-DevMenuCommandSystem $Target application,clear-task-status-list -TrialCount $TrialCount
    Invoke-DevMenuCommandSystem $Target application,uninstall,$ApplicationId -NoExitCodeCheck -TrialCount $TrialCount

    Invoke-DevMenuCommandSystem $Target shop,link-device,0 -TrialCount $TrialCount
}

function Process-DevMenuCommandJson
{
    param
    (
        [Parameter(Mandatory = $true, ValueFromPipeline = $true)]
        [AllowEmptyString()]
        [string[]]
        $Logs
    )

    begin
    {
        $JsonProcessStarted = $false
        $JsonProcessFinished = $false
    }
    process
    {
        # FIXME: json の出力途中で他のログを入れられるとうまく解析できない可能性がある
        $Logs | % {
            if ($_ -eq '[' -or $_ -eq '[]')
            {
                $JsonProcessStarted = $true
            }
            if ($_ -eq '[SUCCESS]')
            {
                $JsonProcessFinished = $true
            }

            if ($JsonProcessStarted -and -not $JsonProcessFinished)
            {
                if ($_[0] -eq ' ' -or $_ -eq '[' -or $_ -eq ']' -or $_ -eq '[]')
                {
                    $_
                }
            }
        }
    }
}

function Get-ApplicationView
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [string]
        $ApplicationId
    )

    $ApplicationListViewLogs = Invoke-DevMenuCommandSystem $Target application,list-view
    $ApplicationListViewLogs | Write-Host
    $ApplicationViews = ($ApplicationListViewLogs | Process-DevMenuCommandJson) -join "`n"

    # デバッグ出力
    Write-Log $ApplicationViews

    if ($ApplicationId)
    {
        ConvertFrom-Json20 $ApplicationViews | ? { $_.id -eq $ApplicationId }
    }
    else
    {
        ConvertFrom-Json20 $ApplicationViews
    }
}

function Test-ApplicationDownloadComplete
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string]
        $ApplicationId,

        [Parameter(Mandatory = $true)]
        [int]
        $Version,

        [switch]
        $AllowNoAppExistence,

        [switch]
        $AllowVersionDifference
    )

    $ApplicationView = Get-ApplicationView $Target $ApplicationId
    if (-not $ApplicationView)
    {
        if ($AllowNoAppExistence)
        {
            return $false
        }
        throw "Application $ApplicationId cannot be found via DevMenuCommandSystem application list-view."
    }

    if ($ApplicationView.isDownloading)
    {
        if ($ApplicationView.progress.state -eq "Runnable")
        {
            return $false
        }

        $ResultCode = $ApplicationView.progress.lastResult
        $ResultIdentifier = "unknown"
        $ErrorResult = Find-ResultDefinition $ResultCode
        if ($ErrorResult)
        {
            $ResultIdentifier = "$($ErrorResult.Namespace)::$($ErrorResult.Name)"
        }
        throw "Unexpected download progress state: $($ApplicationView.progress.state), lastResult = $ResultIdentifier ($ResultCode)"
    }

    if ($ApplicationView.version -ne $Version)
    {
        if ($AllowVersionDifference)
        {
            return $false
        }
        throw "Expected app version is $Version, but actually is $($ApplicationView.version)."
    }

    return $true
}

function Test-SystemUpdateIsRequired
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $CheckLatestLogs = Invoke-DevMenuCommandSystem $Target systemupdate,check-latest
    $CheckLatestLogs | Write-Host

    if ($CheckLatestLogs -contains 'NeedsDownload')
    {
        return
    }
    else
    {
        throw "Cannot start system update testing because there are no updates to download."
    }
}

function Start-SystemUpdateDownload
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    Invoke-DevMenuCommandSystem $Target systemupdate,request-bgnup

    for ($i = 0; ; ++$i)
    {
        if ($i -ge 5)
        {
            throw "System update downloading doesn't start after requesting bgnup."
        }

        $BackgroundStateLogs = Invoke-DevMenuCommandSystem $Target systemupdate,get-background-state
        $BackgroundStateLogs | Write-Host

        if ($BackgroundStateLogs -contains "InProgress")
        {
            break
        }
    }
}

function Test-SystemUpdateDownloadComplete
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $GetBackgroundStateLogs = Invoke-DevMenuCommandSystem $Target systemupdate,get-background-state
    $GetBackgroundStateLogs | Write-Host

    if ($GetBackgroundStateLogs -contains "Ready")
    {
        return $true
    }
    # TORIAEZU: InProgress なら常にリトライできると見なす。result を取得する方法はある？
    if ($GetBackgroundStateLogs -contains "InProgress")
    {
        return $false
    }

    throw "Unexpected systemupdate background state"
}

function Teardown-SystemUpdate
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [int]
        $TrialCount
    )

    # これだけで良い？
    Invoke-DevMenuCommandSystem $Target systemprogram,uninstall,$SystemUpdateId -NoExitCodeCheck
    Invoke-DevMenuCommandSystem $Target systemprogram,uninstall,$SystemUpdateContentsId -NoExitCodeCheck
}

function Wait-SleepCycle
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [int]
        $SleepCount = 2
    )

    for ($i = 0; $i -lt $SleepCount; )
    {
        $Sleeping = Wait-NextSleep $Target -Timeout 300
        if ($Sleeping)
        {
            ++$i
        }
        else
        {
            # 電源ボタンを押したにもかかわらずスリープに入らないことがまれにある
            # spsm からは、電源ボタンがスリープ準備中にも押されているように見えてしまい、スリープを
            # スキップして起床してしまうとのこと
            # 原因が分からないため、ひとまずスクリプト側で再度寝かせる
            Write-InfoLog "Unintended awake. Will sleep again."

            Start-Sleep 10
            Invoke-TargetSleep $Target
        }
    }
}

function Invoke-TargetWakeWithBackToSleepPreventionMonitoring
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $TargetIpAddress = Get-TargetIpAddress $Target

    $Process = Start-HalfAwakeLogMonitor $TargetIpAddress --pattern-failure-exit,'(OYASUMI|OHAYO)',--success-timeout,20

    Invoke-TargetWake $Target -NoWait | Write-Host

    return $Process
}

function Ensure-TargetWakeByBackToSleepPreventionMonitoring
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [System.Diagnostics.Process]
        $Process
    )

    for ($i = 0; ; ++$i)
    {
        if ($i -ge 10)
        {
            throw "After trying to prevent sleeping $i times, target went to sleep again. Will give up because something seems broken."
        }

        $ExitCode = Wait-HalfAwakeLogMonitor $Process -AllowFailurePatternExitCode
        if ($ExitCode -eq 0)
        {
            break
        }
        if ($ExitCode -ne 1)
        {
            throw "Assertion failure: Expected HalfAwakeLogMonitor exit code is 1, but actually is $($ExitCode)"
        }

        Write-InfoLog "Unintended sleep. Will wake again."

        $Process = Invoke-TargetWakeWithBackToSleepPreventionMonitoring $Target
    }
}

function Invoke-TargetWakeForAging
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [ValidateScript({$_ -eq $null -or $_ -ge 0})]
        [int]
        $WakeWait,

        [ValidateSet($null, "None", "Enable", "Disable")]
        [string]
        $CradleToggle,

        [ValidateScript({$_ -eq $null -or $_ -ge 0})]
        [int]
        $CradleToggleWait,

        [ValidateScript({$_ -ge 0})]
        [int]
        $MaximumWait = 30,

        [ValidateScript({$_ -ge 0})]
        [int]
        $MinimumWait = 0
    )

    if (-not $WakeWait)
    {
        $WakeWait = Get-Random -Maximum ($MaximumWait + 1) -Minimum $MinimumWait
    }
    if (-not $CradleToggle)
    {
        # 0: enable, 1: disable
        $CradleToggleNum = Get-Random -Maximum 2
        $CradleToggle = if ($CradleToggleNum -eq 0) { "Enable" } else { "Disable" }
    }
    if (-not $CradleToggleWait)
    {
        $CradleToggleWait = Get-Random -Maximum ($MaximumWait + 1) -Minimum $MinimumWait
    }

    # 半起床を待つ
    Wait-NextAwake $Target

    # クレードル操作をしない
    if ($CradleToggle -eq "None")
    {
        Write-InfoLog "Will wake in $WakeWait sec."
        Start-Sleep $WakeWait

        Invoke-TargetWake $Target
    }
    # クレードル操作をしてから電源ボタン押し
    elseif ($WakeWait -gt $CradleToggleWait)
    {
        # クレードルに接続
        if ($CradleToggle -eq "Enable")
        {
            Write-InfoLog "Will enable cradle in $CradleToggleWait sec."
            Start-Sleep $CradleToggleWait

            # スリープ中の場合、クレードル接続状態は操作できない
            # そのため、エラーが起きても無視する
            Invoke-TargetEnableCradle $Target -NoExitCodeCheck

            # クレードルに接続すると、AC 接続の扱いとなって充電中アイコンを表示するために一度起床してすぐスリープに戻る
            # 電源ボタン押しで明示的に起床させる
            Write-InfoLog "Will wake in $($WakeWait - $CradleToggleWait) sec."
            Start-Sleep ($WakeWait - $CradleToggleWait)

            $Process = Invoke-TargetWakeWithBackToSleepPreventionMonitoring $Target
            Ensure-TargetWakeByBackToSleepPreventionMonitoring $Target $Process
        }
        # クレードル接続解除
        elseif ($CradleToggle -eq "Disable")
        {
            Write-InfoLog "Will disable cradle in $CradleToggleWait sec."
            Start-Sleep $CradleToggleWait

            # スリープ中の場合、クレードル接続状態は操作できない
            # そのため、エラーが起きても無視する
            Invoke-TargetDisableCradle $Target -NoExitCodeCheck

            # クレードル接続を解除すると、AC 接続解除の扱いとなって自動的に起床する
            # 電源ボタン押しで起床させようとすると、既に起床していたところ意図せずスリープに移行する可能性がある
            #
            # ただし、HostBridge 的にクレードルが既に無効化されているかどうかは分からない
            # 仮に既に無効化されている場合は、もう一度クレードル接続解除を行っても AC 接続解除の扱いとはならず起床しない
            #
            # 意図せずスリープに移行した場合は HalfAwakeLogMonitor で検出できるので、クレードル接続の
            # 場合と同様ランダムな時間待って電源ボタンを押す
            Write-InfoLog "Will wake in $($WakeWait - $CradleToggleWait) sec."
            Start-Sleep ($WakeWait - $CradleToggleWait)

            $Process = Invoke-TargetWakeWithBackToSleepPreventionMonitoring $Target
            Ensure-TargetWakeByBackToSleepPreventionMonitoring $Target $Process
        }
    }
    # 電源ボタンを押してからクレードル操作
    else
    {
        Write-InfoLog "Will wake in $WakeWait sec."
        Start-Sleep $WakeWait

        $Process = Invoke-TargetWakeWithBackToSleepPreventionMonitoring $Target

        # 先に電源ボタンを押して起床状態への遷移が行われていれば、後からクレードル操作を行っても
        # 意図せずスリープへ移行するなどの問題はないはず

        # クレードルに接続
        if ($CradleToggle -eq "Enable")
        {
            Write-InfoLog "Will enable cradle in $($CradleToggleWait - $WakeWait) sec."
            Start-Sleep ($CradleToggleWait - $WakeWait)

            # スリープ中の場合、クレードル接続状態は操作できない
            # そのため、エラーが起きても無視する
            Invoke-TargetEnableCradle $Target -NoExitCodeCheck
        }
        # クレードル接続解除
        elseif ($CradleToggle -eq "Disable")
        {
            Write-InfoLog "Will disable cradle in $($CradleToggleWait - $WakeWait) sec."
            Start-Sleep ($CradleToggleWait - $WakeWait)

            # スリープ中の場合、クレードル接続状態は操作できない
            # そのため、エラーが起きても無視する
            Invoke-TargetDisableCradle $Target -NoExitCodeCheck
        }

        Ensure-TargetWakeByBackToSleepPreventionMonitoring $Target $Process
    }
}
