﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nifm.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_DocumentApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include "../Common/testContentDelivery_Common.h"

namespace {

    class ReceiveSystemUpdateTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::es::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::InitializeDependenciesForDfc();
#endif
            m_NspDirectoryPath = std::string(nnt::GetHostArgv()[1]);
            NN_LOG("nspDirectoryPath %s\n", m_NspDirectoryPath.c_str());
        }

        virtual void TearDown()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::FinalizeDependenciesForDfc();
#endif
            nn::ncm::Finalize();
            nn::es::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ASSERT(nnt::GetHostArgc() > 1);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetNspDirectoryPath()
        {
            return m_NspDirectoryPath.c_str();
        }

        nn::Result InstallAllTestSystemContent()
        {
            std::vector<std::string> installed;
            return m_SubmissionPackageInstaller.InstallAll(&installed, GetNspDirectoryPath(), nn::ncm::StorageId::BuiltInSystem);
        }

        nn::Result DeleteSystemContentCompletely(const nn::ncm::ContentMetaKey& key) NN_NOEXCEPT
        {
            nn::ncm::ContentMetaDatabase db;
            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuiltInSystem));
            nn::ncm::ContentManagerAccessor accessor(&db, &storage);

            NN_RESULT_DO(accessor.DeleteRedundant(key, nullptr));
            NN_RESULT_DO(db.Commit());

            NN_RESULT_SUCCESS;
        }

        nn::Result DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType metaType) NN_NOEXCEPT
        {
            const int ListCount = 16;
            nn::ncm::ContentMetaKey keyList[ListCount];

            nn::ncm::ContentMetaDatabase db;
            NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
            auto count = db.ListContentMeta(keyList, ListCount, metaType);
            for (int i = 0; i < count.listed; i++)
            {
                NN_RESULT_DO(DeleteSystemContentCompletely(keyList[i]));
            }

            NN_RESULT_SUCCESS;
        }
    private:
        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;
        nnt::ncmutil::SubmissionPackageFileInstaller m_SubmissionPackageInstaller;
        std::string m_NspDirectoryPath;
    };

    bool HasSystemContent(const nn::ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaDatabase db;
        nn::ncm::ContentStorage storage;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuiltInSystem));

        bool has;
        NN_ABORT_UNLESS_RESULT_SUCCESS(db.Has(&has, key));
        if (!has)
        {
            return false;
        }

        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            const int ListCount = 16;
            nn::ncm::ContentInfo infoList[ListCount];
            int count;
            NN_ABORT_UNLESS_RESULT_SUCCESS(db.ListContentInfo(&count, infoList, ListCount, key, offset));
            for (int i = 0; i < count; i++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Has(&has, infoList[i].GetId()));
                if (!has)
                {
                    return false;
                }
            }

            if (count != ListCount)
            {
                break;
            }
            offset += count;
        }

        return true;
    }
}

TEST_F(ReceiveSystemUpdateTest, Basic)
{
    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    nn::ncm::ContentMetaDatabase db;
    nn::ncm::ContentStorage storage;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuiltInSystem));

    // TORIAEZU: SystemUpdate と SystemData を全て消す
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemUpdate));
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemData));

    // 指定されたディレクトリの nsp を全てインストールする
    NNT_EXPECT_RESULT_SUCCESS(InstallAllTestSystemContent());

    // SystemUpdate と SystemData を検索
    // SystemData は SystemUpdate に含まれている前提
    nn::ncm::ContentMetaKey systemUpdateMetaKey;
    nn::ncm::ContentMetaKey systemDataMetaKey;
    {
        auto count = db.ListContentMeta(&systemUpdateMetaKey, 1, nn::ncm::ContentMetaType::SystemUpdate);
        ASSERT_TRUE(count.listed > 0);

        count = db.ListContentMeta(&systemDataMetaKey, 1, nn::ncm::ContentMetaType::SystemData);
        ASSERT_TRUE(count.listed > 0);
    }

    // 再度インストールされている SystemUpdate と SystemData を消す
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemUpdate));
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemData));

    EXPECT_TRUE(!HasSystemContent(systemUpdateMetaKey));
    EXPECT_TRUE(!HasSystemContent(systemDataMetaKey));

    nn::ns::AsyncResult asyncResult;
    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

        NNT_EXPECT_RESULT_SUCCESS(control.SetupToReceiveSystemUpdate());

        // Test 用に SystemDeliveryInfo を作成
        nn::ns::SystemDeliveryInfo info {};
        info.systemUpdateId = systemUpdateMetaKey.id;
        info.systemUpdateVersion = systemUpdateMetaKey.version;

        // TORIAEZU: IP アドレスは適当
        NNT_EXPECT_RESULT_SUCCESS(control.RequestReceiveSystemUpdate(&asyncResult, 0, TestPort, info));

        while (!asyncResult.TryWait())
        {
            NN_LOG("Wating update started...\n");

            if (control.GetReceiveProgress().loaded > 0)
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    while (!asyncResult.TryWait())
    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

        auto progress = control.GetReceiveProgress();
        NN_LOG("Progress %lld / %lld\n", progress.loaded, progress.total);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

    {
        nn::ns::SystemUpdateControl control;
        NNT_EXPECT_RESULT_SUCCESS(control.Occupy());

        EXPECT_TRUE(control.HasReceived());

        // 適用するまではまだ見つからないはず
        EXPECT_TRUE(!HasSystemContent(systemUpdateMetaKey));
        EXPECT_TRUE(!HasSystemContent(systemDataMetaKey));

        control.ApplyReceivedUpdate();

        EXPECT_TRUE(HasSystemContent(systemUpdateMetaKey));
        EXPECT_TRUE(HasSystemContent(systemDataMetaKey));
    }

} // NOLINT(impl/function_size)

