﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/nifm.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ContentDeliveryApi.h>
#include <nn/ns/ns_DocumentApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include "../Common/testContentDelivery_Common.h"

namespace {

    class ReceiveApplicationPatchTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::es::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::InitializeDependenciesForDfc();
#endif
            m_NspDirectoryPath = std::string(nnt::GetHostArgv()[1]);
            NN_LOG("nspDirectoryPath %s\n", m_NspDirectoryPath.c_str());
        }

        virtual void TearDown()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
            nn::ns::FinalizeDependenciesForDfc();
#endif
            nn::ncm::Finalize();
            nn::es::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ASSERT(nnt::GetHostArgc() > 1);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetNspDirectoryPath()
        {
            return m_NspDirectoryPath.c_str();
        }

        nn::Result InstallAllTestApplications(std::vector<nn::ncm::ApplicationContentMetaKey>* outValue, nn::ncm::StorageId storage)
        {
            return m_ApplicationInstaller.InstallAll(outValue, GetNspDirectoryPath(), storage);
        }

        nn::Result InstallAllRedundantApplications()
        {
            std::vector<std::string> installed;
            return m_SubmissionPackageInstaller.InstallAll(&installed, GetNspDirectoryPath());
        }

        nn::Result DeleteContent(nn::Bit64 id, nn::ncm::StorageId storageId)
        {
            nn::ncm::ContentMetaDatabase db;
            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));
            nn::ncm::ContentManagerAccessor accessor(&db, &storage);

            return accessor.DeleteAll(id);
        }

        bool Find(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value)
                {
                    return true;
                }
            }

            return false;
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindMain(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::Application)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindPatch(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::Patch)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        nn::util::optional<nn::ncm::ContentMetaKey> FindAddOnContent(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && key.key.type == nn::ncm::ContentMetaType::AddOnContent)
                {
                    return key.key;
                }
            }

            return nn::util::nullopt;
        }

        uint32_t FindVersion(const std::vector<nn::ncm::ApplicationContentMetaKey>& list, nn::ncm::ApplicationId id)
        {
            uint32_t version = 0;
            for (auto key : list)
            {
                if (key.applicationId.value == id.value && (key.key.type == nn::ncm::ContentMetaType::Patch || key.key.type == nn::ncm::ContentMetaType::Application))
                {
                    if (version < key.key.version)
                    {
                        version = key.key.version;
                    }
                }
            }

            return version;
        }

    private:
        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;
        nnt::ncmutil::SubmissionPackageFileInstaller m_SubmissionPackageInstaller;
        std::string m_NspDirectoryPath;
    };

    nn::Result HasPatchEntity(bool *outValue, nn::ncm::ApplicationId appId) NN_NOEXCEPT
    {
        nn::ns::ApplicationView view;

        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &appId, 1));

        *outValue = view.HasPatchEntity();
        NN_RESULT_SUCCESS;
    }
}

TEST_F(ReceiveApplicationPatchTest, Basic)
{
    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    nn::ns::InvalidateAllApplicationControlCache();

    // 現存する環境をクリアする
    {
        const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
        nn::ns::ApplicationRecord recordList[128];
        auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
        for (int i = 0; i < recordCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletelyForDebug(recordList[i].id, flags));
        }
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteRedundantApplicationEntity());

    // 指定されたディレクトリのアプリケーションを全てインストールする
    std::vector<nn::ncm::ApplicationContentMetaKey> installed;
    NNT_EXPECT_RESULT_SUCCESS(InstallAllTestApplications(&installed, nn::ncm::StorageId::BuiltInUser));

    nn::ncm::ApplicationId idList[128];
    int idCount;
    {
        nn::ns::ApplicationRecord recordList[128];
        auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
        for (int i = 0; i < recordCount; i++)
        {
            const auto& record = recordList[i];
            idList[i] = record.id;
        }

        idCount = recordCount;
    }

    NN_UTIL_SCOPE_EXIT
    {
        for (int i = 0; i < idCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletely(idList[i]));
        }
    };

    // TORIAEZU: インストールされたものの中から Patch を探し出す
    nn::util::optional<nn::ncm::ContentMetaKey> patchKey;
    nn::ncm::ApplicationId appId {};
    for (int i = 0; i < idCount; i++)
    {
        auto key = FindPatch(installed, idList[i]);
        if (key)
        {
            patchKey = key;
            appId = idList[i];
        }
    }
    EXPECT_TRUE(patchKey);

    if (patchKey)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationContentEntities(appId, nn::ns::ApplicationEntityFlag_Patch::Mask, nn::ncm::StorageId::Any));

        bool hasPatch;
        NNT_EXPECT_RESULT_SUCCESS(HasPatchEntity(&hasPatch, appId));
        EXPECT_TRUE(!hasPatch);

        uint32_t ip = 0; // TORIAEZU
        nn::ns::AsyncResult async;
        NNT_EXPECT_RESULT_SUCCESS(nn::ns::RequestReceiveApplication(&async, ip, TestPort, appId, &(*patchKey), 1, nn::ncm::StorageId::Any));
        while(async.TryWait())
        {
            nn::ns::ReceiveApplicationProgress progress;
            auto result = nn::ns::GetReceiveApplicationProgress(&progress, appId);
            if (result.IsSuccess())
            {
                NN_LOG("%lld / %lld\n", progress.downloadedSize, progress.totalSize);
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_SUCCESS(async.Get());
        {
            nn::ns::ReceiveApplicationProgress progress;
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::GetReceiveApplicationProgress(&progress, appId));
            EXPECT_EQ(progress.downloadedSize, progress.totalSize);
            NN_LOG("%lld / %lld\n", progress.downloadedSize, progress.totalSize);
        }

        // Commit 前はまだ実体が登録されていない
        NNT_EXPECT_RESULT_SUCCESS(HasPatchEntity(&hasPatch, appId));
        EXPECT_TRUE(!hasPatch);

        NNT_EXPECT_RESULT_SUCCESS(nn::ns::CommitReceiveApplication(appId));

        // Commit 後は実体が存在している
        NNT_EXPECT_RESULT_SUCCESS(HasPatchEntity(&hasPatch, appId));
        EXPECT_TRUE(hasPatch);
    }
} // NOLINT(impl/function_size)

