﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/socket.h>

#include <nn/nifm.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationViewSystemApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>

namespace {

    class ApplicationNetworkInstallTest : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
            nn::nim::InitializeForNetworkInstallManager();
#else
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
            nn::fs::MountContentStorage(nn::fs::ContentStorageId::SdCard);
#endif
            nn::ns::Initialize();
            nn::ns::ApplicationRecord recordList[16];
            nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
        }

        virtual void TearDown()
        {
            nn::ns::Finalize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::nim::FinalizeForNetworkInstallManager();
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#else
            nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::User));
            nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
#endif
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ALIGNAS(4096) static uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
#endif
        }

        std::vector<nn::ncm::ApplicationId> GetApplicationIdList()
        {
            nn::ncm::ApplicationId array[] = {
                { 0x0100000000000010 },
                { 0x0100000000000011 }
            };

            return std::vector<nn::ncm::ApplicationId>(array, std::end(array));
        }

        std::vector<nn::ns::ApplicationRecord> ListApplicationRecord()
        {
            std::vector<nn::ns::ApplicationRecord> list;
            list.resize(8192);
            auto count = nn::ns::ListApplicationRecord(list.data(), static_cast<int>(list.size()), 0);
            list.resize(count);

            return list;
        }

        void CancelAllApplicationDownload()
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            for (int i = 0; i < count; i++)
            {
                nn::ns::CancelApplicationDownload(list[i].id);
            }
        }

        void DeleteAllApplicationCompletely()
        {
            const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            for (int i = 0; i < count; i++)
            {
                nn::ns::DeleteApplicationCompletelyForDebug(list[i].id, flags);
            }
        }

        void TestApplicationContentHash(nn::ncm::ApplicationId id)
        {
            std::vector<nn::ncm::StorageContentMetaKey> keyList;
            keyList.resize(2048);
            int keyCount;
            NNT_ASSERT_RESULT_SUCCESS(nn::ns::ListApplicationRecordContentMeta(&keyCount, keyList.data(), static_cast<int>(keyList.size()), id, 0));
            for (int i = 0; i < keyCount; i++)
            {
                auto& key = keyList[i];

                nn::ncm::ContentMetaDatabase db;
                NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, key.storageId));
                nn::ncm::ContentStorage storage;
                NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, key.storageId));

                nn::ncm::ContentInfo infoList[16];

                int infoCount;
                NNT_ASSERT_RESULT_SUCCESS(db.ListContentInfo(&infoCount, infoList, 16, key.key, 0));

                nn::ncm::AutoBuffer meta;
                for (int j = 0; j < infoCount; j++)
                {
                    if (infoList[j].GetType() == nn::ncm::ContentType::Meta)
                    {
                        nn::ncm::Path path;
                        storage.GetPath(&path, infoList[j].GetId());

                        NNT_ASSERT_RESULT_SUCCESS(nn::ncm::ReadContentMetaPath(&meta, path.string));
                    }
                }

                nn::ncm::PackagedContentMetaReader dataReader(meta.Get(), meta.GetSize());
                auto infoDataCount = dataReader.CountContent();
                for (int j = 0; j < infoDataCount; j++)
                {
                    auto info = dataReader.GetContentInfo(j);

                    nn::ncm::Path path;
                    storage.GetPath(&path, info->GetId());

                    nn::crypto::Sha256Generator sha;
                    NNT_ASSERT_RESULT_SUCCESS(GetFileHash(&sha, path.string));

                    char hash[nn::crypto::Sha256Generator::HashSize];
                    sha.GetHash(hash, sizeof(hash));

                    EXPECT_EQ(0, std::memcmp(hash, &info->hash, sizeof(hash)));
                }
            }
        }

        void TestAllApplicationContentHash()
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            for (int i = 0; i < count; i++)
            {
                TestApplicationContentHash(list[i].id);
            }
        }

        void TestWaitSomeApplicationPartiallyDownloaded()
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            bool needsContinue = true;
            while (needsContinue)
            {
                for (int i = 0; i < count; i++)
                {
                    nn::ns::ApplicationView view;
                    NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

                    if (view.IsDownloading())
                    {
                        auto downloaded = view.progress.downloaded;
                        auto total = view.progress.total;

                        NN_LOG("id 0x%016llx, downloaded %lld, total %lld, state %u\n",
                            view.id.value, downloaded, total, view.progress.state);

                        if (total > 0 && static_cast<double>(downloaded) / static_cast<double>(total) > 0.1)
                        {
                            needsContinue = false;
                        }
                    }
                }

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }

        void TestWaitAllApplicationDownloaded()
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            while (NN_STATIC_CONDITION(true))
            {
                bool isDownloading = false;
                for (int i = 0; i < count; i++)
                {
                    nn::ns::ApplicationView view;
                    NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

                    if (view.IsDownloading() && view.progress.state == nn::ns::ApplicationDownloadState::Runnable)
                    {
                        NN_LOG("id 0x%016llx, downloaded %lld, total %lld, state %u\n",
                            view.id.value, view.progress.downloaded, view.progress.total, view.progress.state);
                        EXPECT_LE(view.progress.downloaded, view.progress.total);
                        isDownloading = true;
                    }
                }

                if (!isDownloading)
                {
                    break;
                }

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }

        void TestAllApplicationFlag(int flagIndex, bool isOn)
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            for (int i = 0; i < count; i++)
            {
                nn::ns::ApplicationView view;
                NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

                EXPECT_EQ(isOn, view.flag.Test(flagIndex));
                NNT_ASSERT_RESULT_SUCCESS(view.progress.lastResult);
            }
        }

        void TestAllApplicationStable()
        {
            TestAllApplicationFlag(nn::ns::ApplicationViewFlag_Downloading::Index, false);
        }

        void TestAllApplicationLost()
        {
            TestAllApplicationFlag(nn::ns::ApplicationViewFlag_HasAllEntity::Index, false);
        }

        void TestApplicationControlData(nn::ncm::ApplicationId id)
        {
            std::vector<char> buffer;
            buffer.resize(256 * 1024);
            size_t size;
            NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&size, buffer.data(), buffer.size(), nn::ns::ApplicationControlSource::CacheOnly, id));
        }

        void TestAllApplicationControlData()
        {
            auto list = ListApplicationRecord();
            auto count = static_cast<int>(list.size());

            for (int i = 0; i < count; i++)
            {
                std::vector<char> buffer;
                buffer.resize(256 * 1024);
                size_t size;
                NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationControlData(&size, buffer.data(), buffer.size(), nn::ns::ApplicationControlSource::CacheOnly, list[i].id));
            }
        }

        const char* GetDownloadTaskList() const
        {
            return
R"({
  "format_version": 1,
  "last_modified": 1459513720,
  "tasks": [
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b534",
      "owner_application": "0x0100000000000010",
      "titles": [
        {
          "id": "0x0100000000000010",
          "version": 0,
          "type": "Application"
        }
      ]
    },
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b535",
      "owner_application": "0x0100000000000011",
      "titles": [
        {
          "id": "0x0100000000000011",
          "version": 0,
          "type": "Application"
        }
      ]
    }
  ]
})";
        }

        const char* GetAddOnContentDownloadTaskList() const
        {
            return
R"({
  "format_version": 1,
  "last_modified": 1459513720,
  "tasks": [
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b536",
      "owner_application": "0x0100000000000010",
      "titles": [
        {
          "id": "0x0100000000000015",
          "version": 0,
          "type": "AddOnContent"
        }
      ]
    },
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b537",
      "owner_application": "0x0100000000000011",
      "titles": [
        {
          "id": "0x0100000000000016",
          "version": 0,
          "type": "AddOnContent"
        }
      ]
    }
  ]
})";
        }

        const char* GetMultipleContentKeyDownloadTaskList() const
        {
            return
R"({
  "format_version": 1,
  "last_modified": 1459513720,
  "tasks": [
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b534",
      "owner_application": "0x0100000000000010",
      "titles": [
        {
          "id": "0x0100000000000010",
          "version": 0,
          "type": "Application"
        },
        {
          "id": "0x0100000000000810",
          "version": 1,
          "type": "Patch"
        }
      ]
    },
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b535",
      "owner_application": "0x0100000000000011",
      "titles": [
        {
          "id": "0x0100000000000011",
          "version": 0,
          "type": "Application"
        },
        {
          "id": "0x0100000000000811",
          "version": 1,
          "type": "Patch"
        }
      ]
    }
  ]
})";
        }

        const char* GetNotPublishedTaskList() const
        {
            return
R"({
  "format_version": 1,
  "last_modified": 1459513720,
  "tasks": [
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b534",
      "owner_application": "0xFF00000000000010",
      "titles": [
        {
          "id": "0xFF00000000000010",
          "version": 0,
          "type": "Application"
        }
      ]
    },
    {
      "id": "7c3714a1-4e0a-449c-8482-09c353b5b535",
      "owner_application": "0xFF00000000000011",
      "titles": [
        {
          "id": "0xFF00000000000011",
          "version": 0,
          "type": "Application"
        }
      ]
    }
  ]
})";
        }
    };
}

TEST_F(ApplicationNetworkInstallTest, ApplicationControl)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::nifm::SubmitNetworkRequestAndWait();

    auto list = GetApplicationIdList();
    for (auto id : list)
    {
        nn::ns::AsyncResult async;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestDownloadApplicationControlData(&async, id));
        NNT_ASSERT_RESULT_SUCCESS(async.Get());
        TestApplicationControlData(id);
    }
    nn::nifm::CancelNetworkRequest();
}

TEST_F(ApplicationNetworkInstallTest, Basic)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();
    TestAllApplicationControlData();

    DeleteAllApplicationCompletely();

    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));
    auto list = ListApplicationRecord();
    EXPECT_EQ(0, list.size());
}

TEST_F(ApplicationNetworkInstallTest, AddOnContent)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string addOnContentList = GetAddOnContentDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(addOnContentList.c_str(), addOnContentList.size()));

    TestWaitAllApplicationDownloaded();
    TestAllApplicationContentHash();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    TestAllApplicationControlData();

    DeleteAllApplicationCompletely();
}


TEST_F(ApplicationNetworkInstallTest, AddOnContentMiddleDownload)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitSomeApplicationPartiallyDownloaded();

    std::string addOnContentList = GetAddOnContentDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(addOnContentList.c_str(), addOnContentList.size()));

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();

    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, Cancel)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitSomeApplicationPartiallyDownloaded();
    CancelAllApplicationDownload();
    TestAllApplicationLost();
    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, Delete)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitSomeApplicationPartiallyDownloaded();
    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, Load)
{
    nn::ns::InvalidateAllApplicationControlCache();
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitSomeApplicationPartiallyDownloaded();

    nn::ns::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
    nn::nim::FinalizeForNetworkInstallManager();
    nn::nim::InitializeForNetworkInstallManager();
#endif
    nn::ns::Initialize();

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();
    TestAllApplicationControlData();

    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, MultipleContentKey)
{
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetMultipleContentKeyDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    TestWaitAllApplicationDownloaded();
    TestAllApplicationContentHash();
    TestAllApplicationStable();

    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, Update)
{
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    auto list = ListApplicationRecord();
    auto count = static_cast<int>(list.size());

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();

    for (int i = 0; i < count; i++)
    {
        nn::ns::AsyncApplicationUpdateInfo asyncInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestApplicationUpdateInfo(&asyncInfo, list[i].id));
        nn::ns::ApplicationUpdateInfo info;
        NNT_ASSERT_RESULT_SUCCESS(asyncInfo.Get(&info));
        EXPECT_EQ(nn::ns::ApplicationUpdateInfo::Updatable, info);
    }

    for (int i = 0; i < count; i++)
    {
        nn::ns::AsyncResult asyncResult;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestUpdateApplication(&asyncResult, list[i].id));
        NNT_ASSERT_RESULT_SUCCESS(asyncResult.Get());
    }

    TestWaitAllApplicationDownloaded();
    TestAllApplicationContentHash();
    TestAllApplicationStable();

    for (int i = 0; i < count; i++)
    {
        nn::ns::AsyncApplicationUpdateInfo asyncInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestApplicationUpdateInfo(&asyncInfo, list[i].id));
        nn::ns::ApplicationUpdateInfo info;
        NNT_ASSERT_RESULT_SUCCESS(asyncInfo.Get(&info));
        EXPECT_EQ(nn::ns::ApplicationUpdateInfo::UpToDate, info);
    }

    for (int i = 0; i < count; i++)
    {
        nn::ns::AsyncResult async;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestUpdateApplication(&async, list[i].id));
        NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultAlreadyUpToDate, async.Get());
    }

    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, UpdateLost)
{
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetMultipleContentKeyDownloadTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    auto list = ListApplicationRecord();
    auto count = static_cast<int>(list.size());

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();

    for (int i = 0; i < count; i++)
    {
        nn::ns::AsyncApplicationUpdateInfo asyncInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestApplicationUpdateInfo(&asyncInfo, list[i].id));
        nn::ns::ApplicationUpdateInfo info;
        NNT_ASSERT_RESULT_SUCCESS(asyncInfo.Get(&info));
        EXPECT_EQ(nn::ns::ApplicationUpdateInfo::UpToDate, info);
    }

    for (int i = 0; i < count; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::DeleteApplicationEntity(list[i].id));

        {
            nn::ns::AsyncApplicationUpdateInfo asyncInfo;
            NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestApplicationUpdateInfo(&asyncInfo, list[i].id));
            nn::ns::ApplicationUpdateInfo info;
            NNT_ASSERT_RESULT_SUCCESS(asyncInfo.Get(&info));
            EXPECT_EQ(nn::ns::ApplicationUpdateInfo::Updatable, info);
        }

        {
            nn::ns::AsyncResult asyncResult;
            NNT_ASSERT_RESULT_SUCCESS(nn::ns::RequestUpdateApplication(&asyncResult, list[i].id));
            NNT_ASSERT_RESULT_SUCCESS(asyncResult.Get());
        }
    }

    TestWaitAllApplicationDownloaded();
    TestAllApplicationStable();
    TestAllApplicationContentHash();

    DeleteAllApplicationCompletely();
}

TEST_F(ApplicationNetworkInstallTest, NotFound)
{
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetNotPublishedTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    auto list = ListApplicationRecord();
    auto count = static_cast<int>(list.size());

    TestWaitAllApplicationDownloaded();

    for (int i = 0; i < count; i++)
    {
        nn::ns::ApplicationView view;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

        EXPECT_TRUE(view.IsDownloading());
        EXPECT_EQ(nn::ns::ApplicationDownloadState::Fatal, view.progress.state);
        NN_LOG("lastResult 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
    }

    DeleteAllApplicationCompletely();
}

/*
TEST_F(ApplicationNetworkInstallTest, RetryNotFound)
{
    DeleteAllApplicationCompletely();
    nn::ns::ClearTaskStatusList();

    std::string downloadTaskList = GetNotPublishedTaskList();
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::PushDownloadTaskList(downloadTaskList.c_str(), downloadTaskList.size()));

    auto list = ListApplicationRecord();
    auto count = static_cast<int>(list.size());

    TestWaitAllApplicationDownloaded();

    for (int i = 0; i < count; i++)
    {
        nn::ns::ApplicationView view;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

        EXPECT_EQ(nn::ns::ApplicationEntityState::Downloading, view.GetEntityState());
        EXPECT_EQ(nn::ns::ApplicationDownloadState::Suspended, view.progress.state);
        NN_LOG("lastResult 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());

        NNT_ASSERT_RESULT_SUCCESS(nn::ns::ResumeApplicationDownload(list[i].id));

        NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));
        EXPECT_EQ(nn::ns::ApplicationEntityState::Downloading, view.GetEntityState());
        EXPECT_EQ(nn::ns::ApplicationDownloadState::Downloading, view.progress.state);
        NN_LOG("lastResult 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
    }

    TestWaitAllApplicationDownloaded();

    for (int i = 0; i < count; i++)
    {
        nn::ns::ApplicationView view;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &list[i].id, 1));

        EXPECT_EQ(nn::ns::ApplicationEntityState::Downloading, view.GetEntityState());
        EXPECT_EQ(nn::ns::ApplicationDownloadState::Suspended, view.progress.state);
        NN_LOG("lastResult 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
    }

    DeleteAllApplicationCompletely();
}
*/
