﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/socket.h>

#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

namespace {

    class VersionListTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            nn::nim::InitializeForNetworkInstallManager();
        }

        virtual void TearDown()
        {
            nn::nim::FinalizeForNetworkInstallManager();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();
        }

        static void TearDownTestCase()
        {
            nn::nifm::CancelNetworkRequest();
        }
    };

}

TEST_F(VersionListTest, RequestCancel)
{
    auto emptyETag = nn::nim::ETag::MakeEmpty();

    nn::nim::AsyncData asyncVersionList;
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestVersionList(&asyncVersionList, emptyETag));
    asyncVersionList.Cancel();
    while (!asyncVersionList.TryWait())
    {
        NN_LOG("Waiting request version list canceled...\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultHttpConnectionCanceled, asyncVersionList.Get());
    asyncVersionList.Finalize();
}

TEST_F(VersionListTest, Request)
{
    auto emptyETag = nn::nim::ETag::MakeEmpty();

    nn::nim::ETag downloadedETag;
    {
        // ETag なし
        nn::nim::AsyncData asyncVersionList;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestVersionList(&asyncVersionList, emptyETag));
        while (!asyncVersionList.TryWait())
        {
            NN_LOG("Waiting request version list downloaded...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NNT_EXPECT_RESULT_SUCCESS(asyncVersionList.Get());
        NNT_EXPECT_RESULT_SUCCESS(asyncVersionList.GetETag(&downloadedETag));
        asyncVersionList.Finalize();
    }

    {
        // ETag が異なり再ダウンロードされるケース
        nn::nim::ETag dummyValidETag = { "abcdef" };
        nn::nim::AsyncData asyncVersionList;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestVersionList(&asyncVersionList, dummyValidETag));
        while (!asyncVersionList.TryWait())
        {
            NN_LOG("Waiting request version list downloaded...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NNT_EXPECT_RESULT_SUCCESS(asyncVersionList.Get());
        asyncVersionList.Finalize();
    }

    {
        // ETag が同じでスキップされるケース
        nn::nim::AsyncData asyncVersionList;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestVersionList(&asyncVersionList, downloadedETag));
        while (!asyncVersionList.TryWait())
        {
            NN_LOG("Waiting request version list downloaded...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultVersionListIsLatest, asyncVersionList.Get());
        asyncVersionList.Finalize();
    }
}
