﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <cstring>
#include <vector>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_RapidJsonInputStream.h>

namespace nnt { namespace eca {

namespace json {

//!------------------------------------------------------------------------------------------------------
//! JSONレスポンスエントリルックアップ
template<int SizeOfJsonPathMaxDepth, int SizeOfJsonPathMaxLength>
struct EntryLookup
{
    typedef ::nn::http::json::JsonPath<SizeOfJsonPathMaxDepth, SizeOfJsonPathMaxLength> JsonPathType;
    char    pPath[SizeOfJsonPathMaxLength];
    bool    found;

    NN_IMPLICIT EntryLookup() NN_NOEXCEPT : pPath(""), found(false) {}

    NN_IMPLICIT EntryLookup(const char* const pSourcePath) NN_NOEXCEPT : found(false)
    {
        NN_ABORT_UNLESS(nullptr != pSourcePath);
        std::strncpy(pPath, pSourcePath, sizeof(pPath));
    }

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return found;
    }

    bool CanAccept(const JsonPathType& jsonPath) const NN_NOEXCEPT
    {
        return !(*this) && jsonPath.Match(pPath);
    }

    void MarkAccepted() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!found);
        found = true;
    }
};

//!------------------------------------------------------------------------------------------------------
//! JSONレスポンスエントリストア( 文字列 )
template<typename Lookup>
struct EntryStoreBase
{
    Lookup  lookup;

    NN_IMPLICIT EntryStoreBase() NN_NOEXCEPT {}
    NN_IMPLICIT EntryStoreBase(const char* const pSourcePath) NN_NOEXCEPT : lookup(pSourcePath) {}

    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return static_cast<bool>(lookup);
    }

    char* GetLookupPathTop() NN_NOEXCEPT
    {
        return lookup.pPath;
    }

    size_t GetLookupPathCapacity() const NN_NOEXCEPT
    {
        return sizeof(lookup.pPath);
    }

    bool CreateLookupPath(const char* const pPathFormat) NN_NOEXCEPT
    {
        const auto size = GetLookupPathCapacity();
        return (nn::util::SNPrintf(GetLookupPathTop(), size, pPathFormat) < size);
    }

    template<typename TArgs>
    bool CreateLookupPath(const char* const pPathFormat, TArgs index) NN_NOEXCEPT
    {
        const auto size = GetLookupPathCapacity();
        return (nn::util::SNPrintf(GetLookupPathTop(), size, pPathFormat, index) < size);
    }
};

//!------------------------------------------------------------------------------------------------------
//! JSONレスポンスエントリストア( 文字列 )
template<size_t StoreCapacity, typename Lookup>
struct StringStore : public EntryStoreBase<Lookup>
{
    typedef EntryStoreBase<Lookup> BaseType;
    char    store[StoreCapacity];

    NN_IMPLICIT StringStore() NN_NOEXCEPT : store("") {}
    NN_IMPLICIT StringStore(const char* const pSourcePath) NN_NOEXCEPT : BaseType(pSourcePath), store("") {}

    bool Store(const typename Lookup::JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        if (BaseType::lookup.template CanAccept(jsonPath) && valueLength > 0)
        {
            NN_ASSERT(sizeof(store) > static_cast<size_t>(valueLength));
            std::strncpy(store, pValue, sizeof(store));
            store[sizeof(store) - 1] = '\0';
            BaseType::lookup.template MarkAccepted();
            return true;
        }
        return false;
    }
};

//!------------------------------------------------------------------------------------------------------
//! JSONレスポンスエントリストア( 値 )
template<typename TValue, typename Lookup>
struct ValueStore : public EntryStoreBase<Lookup>
{
    typedef EntryStoreBase<Lookup> BaseType;
    typedef TValue ValueType;
    TValue  store;

    NN_IMPLICIT ValueStore() NN_NOEXCEPT : store() {}
    NN_IMPLICIT ValueStore(const char* const pSourcePath) NN_NOEXCEPT : BaseType(pSourcePath), store() {}

    bool Store(const typename Lookup::JsonPathType& jsonPath, const ValueType value) NN_NOEXCEPT
    {
        if (BaseType::lookup.template CanAccept(jsonPath))
        {
            store = static_cast<ValueType>(value);
            BaseType::lookup.template MarkAccepted();
            return true;
        }
        return false;
    }
};

}   // ~nnt::eca::json

//!------------------------------------------------------------------------------------------------------
//! @brief  権利取得レスポンスパーサーアダプタ
class RightReponseAdaptor
{
public:
    static const int JsonPathMaxLength = 80;
    typedef json::EntryLookup<12, JsonPathMaxLength>    EntryLookup;
    typedef json::StringStore<128, EntryLookup>         EntryStore;
    typedef EntryLookup::JsonPathType                   JsonPathType;

    explicit RightReponseAdaptor(int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_RightId.CreateLookupPath("$.rights[%u].right_id", index));
        NN_ABORT_UNLESS(m_ConsumptionId.CreateLookupPath("$.rights[%u].consumption_request_id", index));

        // デフォルトUUIDの生成。
        // ConsumptionRequestId は、複数デバイスを使った権利複製を防止するために追加された識別子。
        ::nn::util::GenerateUuid().ToString(m_ConsumptionId.store, sizeof(m_ConsumptionId.store));
    }

    const char* GetRightId() const NN_NOEXCEPT
    {
        return m_RightId.store;
    }

    const char* GetConsumptionId() const NN_NOEXCEPT
    {
        return m_ConsumptionId.store;
    }

    /**
     * @brief   文字列検出ハンドラ。
     *
     * @details パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
     */
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        m_RightId.Store(jsonPath, pValue, valueLength);
        m_ConsumptionId.Store(jsonPath, pValue, valueLength);
    }

    // 以下検出対象外
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}

private:
    EntryStore  m_RightId;
    EntryStore  m_ConsumptionId;
};

//!------------------------------------------------------------------------------------------------------
//! @brief  Civil カタログレスポンスパーサーアダプタ
//! @note   <{"lang":"ja","consumables":[{"consumable_id":"b7eb25fe683dd99a","name":"eca_name","description":"eca_desc","disclaimer":"eca_disc"}]}>
class ConsumablesReponseAdaptor
{
public:
    static const int JsonPathMaxLength = 80;
    typedef json::EntryLookup<12, JsonPathMaxLength>    EntryLookup;
    typedef json::StringStore<128, EntryLookup>         EntryStore;
    typedef EntryLookup::JsonPathType                   JsonPathType;


    explicit ConsumablesReponseAdaptor(int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_ConsumableId.CreateLookupPath("$.consumables[%u].consumable_id", index));
        NN_ABORT_UNLESS(m_Name.CreateLookupPath("$.consumables[%u].name", index));
    }

    const char* GetConsumableId() const NN_NOEXCEPT
    {
        return m_ConsumableId.store;
    }

    const char* GetName() const NN_NOEXCEPT
    {
        return m_Name.store;
    }

    /**
     * @brief   文字列検出ハンドラ。
     *
     * @details パース中に文字列値をもつJSONパスが見つかった時に呼ばれます。
     */
    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        m_ConsumableId.Store(jsonPath, pValue, valueLength);
        m_Name.Store(jsonPath, pValue, valueLength);
    }

    // 以下検出対象外
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}

private:
    EntryStore m_ConsumableId;
    EntryStore m_Name;
};

//!------------------------------------------------------------------------------------------------------
//! @brief  Civil 消費アイテムレスポンスパーサーアダプタ
//! @note   <{"lang":"ja","length":2,"offset":0,"total":2,"consumable_items":[{"item_id":"eca0000","name":"eca0000","ns_uid":70090000002042,"price":{"sales_status":"onsale","regular_price":{"currency":"JPY","formatted_value":"100円","raw_value":"100"},"discount_price":null}},{"item_id":"eca0001","name":"eca0001","ns_uid":70090000002043,"price":{"sales_status":"onsale","regular_price":{"currency":"JPY","formatted_value":"100円","raw_value":"100"},"discount_price":null}}]}>
class ConsumableItemsReponseAdaptor
{
public:
    static const int JsonPathMaxLength = 96;
    typedef json::EntryLookup<16, JsonPathMaxLength>    EntryLookup;
    typedef json::StringStore<128, EntryLookup>         EntryStore;
    typedef json::ValueStore<uint64_t, EntryLookup>     ValueStore;
    typedef json::ValueStore<int, EntryLookup>          TotalStore;
    typedef EntryLookup::JsonPathType                   JsonPathType;

    struct ItemProperty
    {
        EntryStore  item_id;
        ValueStore  ns_uid;
        EntryStore  sales_status;
        EntryStore  raw_value;

        bool CreateLookupPath(int index) NN_NOEXCEPT
        {
            return (item_id.CreateLookupPath("$.consumable_items[%u].item_id", index)
                && ns_uid.CreateLookupPath("$.consumable_items[%u].ns_uid", index)
                && sales_status.CreateLookupPath("$.consumable_items[%u].price.sales_status", index)
                && raw_value.CreateLookupPath("$.consumable_items[%u].price.regular_price.raw_value", index)
            );
        }
    };
    typedef std::vector<ItemProperty> Properties;

    NN_IMPLICIT ConsumableItemsReponseAdaptor() NN_NOEXCEPT
        : m_TotalStore("$.total"), m_FoundIndex(0)
    {
        m_Properties.reserve(2);
    }

    int GetTotalCount() const NN_NOEXCEPT
    {
        return m_TotalStore.store;
    }

    const Properties& GetProperties() const NN_NOEXCEPT
    {
        return m_Properties;
    }

    void Update(const JsonPathType& jsonPath, const char* pValue, int valueLength) NN_NOEXCEPT
    {
        if (m_TotalStore && m_Properties.size() > m_FoundIndex)
        {
            auto& item = m_Properties.at(m_FoundIndex);
            item.item_id.Store(jsonPath, pValue, valueLength);
            item.sales_status.Store(jsonPath, pValue, valueLength);
            item.raw_value.Store(jsonPath, pValue, valueLength);
        }
    }

    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
    {
        if (m_TotalStore.Store(jsonPath, static_cast<TotalStore::ValueType>(value)))
        {
            const auto n = m_TotalStore.store;
            m_Properties.resize(n);
            m_FoundIndex = 0;

            for (int index = 0; index < n; ++index)
            {
                m_Properties[index].CreateLookupPath(index);
            }
        }
        else if (m_TotalStore && m_Properties.size() > m_FoundIndex)
        {
            auto& item = m_Properties.at(m_FoundIndex);
            item.ns_uid.Store(jsonPath, static_cast<ValueStore::ValueType>(value));
        }
    }

    void Update(const JsonPathType& jsonPath, uint64_t value) NN_NOEXCEPT
    {
        if (m_TotalStore && m_Properties.size() > m_FoundIndex)
        {
            auto& item = m_Properties.at(m_FoundIndex);
            item.ns_uid.Store(jsonPath, static_cast<ValueStore::ValueType>(value));
        }
    }

    void NotifyObjectEnd(const JsonPathType& jsonPath) NN_NOEXCEPT
    {
        char buf[64];
        if (nn::util::SNPrintf(buf, sizeof(buf), "$.consumable_items[%u]", m_FoundIndex) < sizeof(buf) && jsonPath.Match(buf))
        {
            ++m_FoundIndex;
        }
    }

    // 以下検出対象外
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}

private:
    Properties  m_Properties;
    TotalStore  m_TotalStore;
    uint16_t    m_FoundIndex;
};


}}  // ~::nnt::eca
