﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/ncmutil/ncmutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>

#include "../Common/testLocalCommunication_Common.h"

namespace {

    class LocalCommunicationSendSystemUpdateTaskTest : public testing::Test, public nnt::ncmutil::SubmissionPackageFileInstaller
    {
    protected:
        virtual void SetUp()
        {

#if defined( NN_BUILD_CONFIG_OS_WIN )
            SetupBisWorkingDirectory(nnt::GetHostArgv()[1]);
#endif
            nn::ncm::Initialize();
            m_NspDirectory = std::string(nnt::GetHostArgv()[2]);
        }

        virtual void TearDown()
        {
            nn::nifm::CancelNetworkRequest();
            nn::ncm::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(TearDownBisWorkingDirectory());
#endif
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ASSERT(nnt::GetHostArgc() > 2);
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            nn::ovln::InitializeSenderLibraryForOverlay();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
            static NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                                  nn::socket::DefaultSocketMemoryPoolSize,
                                                                  nn::socket::DefaultSocketAllocatorSize,
                                                                  nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
            nn::fs::UnmountHostRoot();
#endif
            nn::nifm::CancelNetworkRequest();
        }

        nn::Result InstallAllNsp(nn::ncm::StorageId storage)
        {
            std::vector<std::string> outList;
            return InstallAll(&outList, GetNspDirectory(), storage);
        }

    private:
        const char* GetNspDirectory() const NN_NOEXCEPT
        {
            return m_NspDirectory.c_str();
        }

        std::string m_NspDirectory;
    };

    nn::Result DeleteContentByKey(const nn::ncm::ContentMetaKey& key, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
        nn::ncm::ContentStorage storage;
        NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));
        nn::ncm::ContentInfo contentInfoList[16];
        int offset = 0;
        while(NN_STATIC_CONDITION(true))
        {
            int infoCount;
            NN_RESULT_DO(db.ListContentInfo(&infoCount, contentInfoList, NN_ARRAY_SIZE(contentInfoList), key, offset));
            for (int j = 0; j < infoCount; j++)
            {
                storage.Delete(contentInfoList[j].GetId());
            }
            if (infoCount < NN_ARRAY_SIZE(contentInfoList))
            {
                break;
            }
            offset += infoCount;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType metaType) NN_NOEXCEPT
    {
        const int ListCount = 16;
        nn::ncm::ContentMetaKey keyList[ListCount];

        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuiltInSystem));
        auto count = db.ListContentMeta(keyList, ListCount, metaType);
        for (int i = 0; i < count.listed; i++)
        {
            NN_RESULT_DO(DeleteContentByKey(keyList[i], nn::ncm::StorageId::BuiltInSystem));
            NN_RESULT_DO(db.Remove(keyList[i]));
        }
        NN_RESULT_DO(db.Commit());

        NN_RESULT_SUCCESS;
    }

}

TEST_F(LocalCommunicationSendSystemUpdateTaskTest, SendPatch)
{
    nn::nim::InitializeForNetworkInstallManager();

    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInSystem;
    nn::ncm::ContentMetaDatabase db;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    nn::ncm::ContentStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));

    // 一旦インストールされている SystemUpdate と SystemData を消す
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemUpdate));
    NNT_ASSERT_RESULT_SUCCESS(DeleteSystemContentByContentMetaType(nn::ncm::ContentMetaType::SystemData));

    // ディレクトリ内の nsp をインストール
    NNT_ASSERT_RESULT_SUCCESS(InstallAllNsp(storageId));

    // SystemUpdate を検索
    nn::ncm::ContentMetaKey systemUpdateMetaKey;
    {
        auto count = db.ListContentMeta(&systemUpdateMetaKey, 1, nn::ncm::ContentMetaType::SystemUpdate);
        ASSERT_TRUE(count.listed > 0);
    }

    // 送信
    {
        NN_LOG("key.id=0x%llx, key.version=%u\n", systemUpdateMetaKey.id, systemUpdateMetaKey.version);
        nn::nim::LocalCommunicationSendSystemUpdateTaskId id;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateLocalCommunicationSendSystemUpdateTask(&id, 0, TestPort, systemUpdateMetaKey));

        // キャンセルできる
        {
            nn::nim::AsyncResult asyncResult;
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestLocalCommunicationSendSystemUpdateTaskRun(&asyncResult, id));
            asyncResult.Cancel();

            NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroyLocalCommunicationSendSystemUpdateTask(id));
        }
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyLocalCommunicationSendSystemUpdateTask(id));

        NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateLocalCommunicationSendSystemUpdateTask(&id, 0, TestPort, systemUpdateMetaKey));

        {
            nn::nim::AsyncResult asyncResult;
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestLocalCommunicationSendSystemUpdateTaskRun(&asyncResult, id));
            asyncResult.Wait();
            NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());

            NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroyLocalCommunicationSendSystemUpdateTask(id));
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyLocalCommunicationSendSystemUpdateTask(id));
    }

    nn::nim::FinalizeForNetworkInstallManager();
}
