﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_NetworkConnection.h>

#include <nn/nifm/nifm_ResultPrivate.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
}

class StealthCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};

TEST_F(StealthCaseTest, Basic)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
    int outCount;
    nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User);
    ASSERT_EQ(1, outCount);

    nn::nifm::NetworkProfileData networkProfileData;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile( &networkProfileData, networkProfileBasicInfo[0].id ));

    // ステルス接続対象ではない接続設定しかない
    ASSERT_FALSE(networkProfileData.wirelessSetting.ssidConfig.nonBroadcast);

    {
        nn::nifm::NetworkConnection networkConnection;
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        ASSERT_FALSE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());
    }

    // ステルス接続対象に変更
    {
        networkProfileData.wirelessSetting.ssidConfig.nonBroadcast = true;

        nn::util::Uuid outId;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &outId, networkProfileData ));
        networkProfileData.id = outId;

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
    }

    // 元に戻す
    {
        networkProfileData.wirelessSetting.ssidConfig.nonBroadcast = false;

        nn::util::Uuid outId;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &outId, networkProfileData ));
        networkProfileData.id = outId;
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
