﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>

#include <nn/util/util_CharacterEncoding.h>

#include <algorithm>
#include <cstring>

#include "../Common/nifm_TestUtility.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    int PrepareTcpSocket(const char* hostname, uint16_t port, bool isExempted)
    {
        auto sockfd = isExempted ?
            nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp) :
            nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        EXPECT_GE(sockfd, 0);

        auto* hostent = nn::socket::GetHostEntByName(hostname);
        EXPECT_NE(nullptr, hostent);

        nn::socket::SockAddrIn serverAddr;
        memset(&serverAddr, 0, sizeof(serverAddr));
        std::memcpy(&serverAddr.sin_addr, hostent->h_addr, hostent->h_length);
        serverAddr.sin_port = nn::socket::InetHtons(port);
        serverAddr.sin_family = nn::socket::Family::Af_Inet;

        auto ret = nn::socket::Connect(sockfd, reinterpret_cast<const nn::socket::SockAddr*>(&serverAddr), sizeof(serverAddr));
        EXPECT_EQ(0, ret);

        if (ret)
        {
            NN_LOG("Errorno:%d\n", nn::socket::GetLastError());
        }

        return sockfd;
    }
}

class SleepCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        nn::nifm::test::ElapsedTime(nn::os::GetSystemTick().GetInt64Value());

        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);
    }
    virtual void TearDown()
    {
        nn::nifm::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::FinalizeForTest();
        };

        // 独占解除
        nn::nifm::test::SetExclusive<nn::nifm::Initialize, nn::nifm::FinalizeForTest>(false);

        NN_LOG("TotalTime [%s]\n", nn::nifm::test::ElapsedTime());
    }
};

TEST_F(SleepCaseTest, NormalEthernet)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // スリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, NormalWireless)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // スリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, KeptInSleepEthernet)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 有線接続はスリープ中に維持する設定でも、スリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, KeptInSleepWithoutSocketDescriptorWireless)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // ソケットディスクリプタを登録していないと、無線接続でもスリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        EXPECT_FALSE(networkConnection.GetResult().IsSuccess());
        EXPECT_FALSE(networkConnectionBG.GetResult().IsSuccess());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDisconnected, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDisconnected, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, KeptInSleepWireless)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection1;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910960");  // imatake-wpa2aes
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection1.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG1.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG1.GetRequestHandle(), true));

        networkConnection1.SubmitRequest();
        networkConnectionBG1.SubmitRequest();

        ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        EXPECT_TRUE(networkConnectionBG1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG1.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        nn::socket::Initialize(g_SocketConfigWithMemory);

        auto exemptSocketFd = PrepareTcpSocket("ctest.cdn.nintendo.net", 80, true);
        auto socketFd = PrepareTcpSocket("ctest.cdn.nintendo.net", 80, false);

        char buffer[10];
        EXPECT_EQ(-1, nn::socket::Recv(exemptSocketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::EAgain, nn::socket::GetLastError());
        EXPECT_EQ(-1, nn::socket::Recv(socketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::EAgain, nn::socket::GetLastError());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RegisterRequestSocketDescriptor(networkConnection1.GetRequestHandle(), exemptSocketFd));

        // 無線接続の場合、スリープに入っても利用要求は維持される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_FALSE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        EXPECT_TRUE(networkConnectionBG1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG1.GetResult());

        // SocketExempt で生成されたソケットは生存している
        EXPECT_EQ(-1, nn::socket::Recv(exemptSocketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::EAgain, nn::socket::GetLastError());

        // Socket で生成されたソケットは shutdown されている
        EXPECT_EQ(-1, nn::socket::Recv(socketFd, buffer, sizeof(buffer), nn::socket::MsgFlag::Msg_DontWait));
        EXPECT_EQ(nn::socket::Errno::ENetDown, nn::socket::GetLastError());

        // 接続が維持されていても、スリープ中に出した要求は起床まで待たされる

        nn::nifm::NetworkConnection networkConnection2;
        nn::nifm::NetworkConnection networkConnectionBG2;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG2.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        networkConnection2.GetSystemEvent().Clear();
        networkConnectionBG2.GetSystemEvent().Clear();
        networkConnection2.SubmitRequest();
        networkConnectionBG2.SubmitRequest();

        EXPECT_FALSE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection2.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG2.IsRequestOnHold());
        EXPECT_FALSE(networkConnection2.IsAvailable());
        EXPECT_FALSE(networkConnectionBG2.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection2.IsRequestOnHold());
        EXPECT_FALSE(networkConnection2.IsAvailable());

        EXPECT_TRUE(networkConnectionBG2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG2.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG2.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection2.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection2.GetResult());

        networkConnection1.CancelRequest();
        networkConnection2.CancelRequest();
        networkConnectionBG1.CancelRequest();
        networkConnectionBG2.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection1.GetSystemEvent().Clear();
        networkConnectionBG1.GetSystemEvent().Clear();
        networkConnection1.SubmitRequest();
        networkConnectionBG1.SubmitRequest();

        ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        EXPECT_TRUE(networkConnectionBG1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG1.GetResult());

        EXPECT_EQ(0, nn::socket::Close(exemptSocketFd));
        EXPECT_EQ(0, nn::socket::Close(socketFd));

        NNT_EXPECT_RESULT_SUCCESS(nn::socket::Finalize());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, KeptInSleepWirelessSsidList)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("1a765924-cfb0-4833-a9da-aa81d83aea4a");  // mmt
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestKeptInSleep(networkConnectionBG.GetRequestHandle(), true));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // SSID リストの設定利用中はスリープ中に維持する設定でも、スリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, KeptInSleepNeighborDetection)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_NeighborDetectionForSystemProcess));

        networkConnection1.SubmitRequest();

        ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // NeighborDetection 接続の場合、スリープに入っても利用要求は維持される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_FALSE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection1.GetResult());

        // 接続が維持されていても、スリープ中に出した要求は起床まで待たされる

        nn::nifm::NetworkConnection networkConnection2;
        nn::nifm::NetworkConnection networkConnectionBG1;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG1.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        networkConnection2.GetSystemEvent().Clear();
        networkConnectionBG1.GetSystemEvent().Clear();
        networkConnection2.SubmitRequest();
        networkConnectionBG1.SubmitRequest();

        EXPECT_FALSE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_FALSE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection2.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG1.IsRequestOnHold());
        EXPECT_FALSE(networkConnection2.IsAvailable());
        EXPECT_FALSE(networkConnectionBG1.IsAvailable());

        NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
        nn::os::ThreadType CleanUpThread;

        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&CleanUpThread,
            [](void* p) {
            auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
            nn::os::SystemEvent& systemEvent = pNetworkConnection->GetSystemEvent();
            while (NN_STATIC_CONDITION(true))
            {
                systemEvent.Wait();
                nn::nifm::RequestState requestState = pNetworkConnection->GetRequestState();
                switch (requestState)
                {
                case nn::nifm::RequestState::RequestState_Blocking:
                    pNetworkConnection->CancelRequest();
                    return;
                default:
                    break;
                }
            }
        },
            &networkConnection1, &stackOfThread, sizeof(stackOfThread), 24));
        nn::os::StartThread(&CleanUpThread);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // スリープ中に提出された FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection2.IsRequestOnHold());
        EXPECT_FALSE(networkConnection2.IsAvailable());

        // スリープ中に提出された BG インターネット利用要求は受理され，すれちがい要求が却下される

        EXPECT_TRUE(networkConnectionBG1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG1.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG1.GetResult());

        nn::os::DestroyThread(&CleanUpThread);

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection2.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection2.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection2.GetResult());

        networkConnection1.CancelRequest();
        networkConnection2.CancelRequest();
        networkConnectionBG1.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection1.GetSystemEvent().Clear();
        networkConnection1.SubmitRequest();

        ASSERT_TRUE(networkConnection1.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection1.IsAvailable());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

TEST_F(SleepCaseTest, PersistentEthernet)
{
    nn::nifm::test::ScopedInitializer<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest> initializer;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    // 一般的な使い方（ブロッキング）
    {
        nn::nifm::NetworkConnection networkConnection;
        nn::util::Uuid uuid;
        uuid.FromString("93188440-b6d5-14b7-056b-08cd57910964");  // ethernet
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestPersistent(networkConnection.GetRequestHandle(), true));

        nn::nifm::NetworkConnection networkConnectionBG;
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBG.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcessSharable));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnectionBG.GetRequestHandle(), uuid));
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnectionBG.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        // DispatchLoop が1周まわるのを待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // スリープに入るときに受理されていた要求は却下される

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnection.GetResult());
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultDevicePutToSleep, networkConnectionBG.GetResult());

        // スリープ中に出した要求は起床まで待たされる

        networkConnectionBG.GetSystemEvent().Clear();
        networkConnectionBG.SubmitRequest();

        EXPECT_FALSE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));

        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_TRUE(networkConnectionBG.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());
        EXPECT_FALSE(networkConnectionBG.IsAvailable());

        networkConnection.GetSystemEvent().Clear();

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake

        // Persistent な FG 利用要求は MinimumAwake では評価されない

        EXPECT_FALSE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(5)));
        EXPECT_TRUE(networkConnection.IsRequestOnHold());
        EXPECT_FALSE(networkConnection.IsAvailable());

        EXPECT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake

        // FullAwake 後は FG 利用要求も評価される

        EXPECT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        EXPECT_TRUE(networkConnection.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());

        networkConnection.CancelRequest();
        networkConnectionBG.CancelRequest();

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // 起床後は再び要求が通るようになる

        networkConnection.GetSystemEvent().Clear();
        networkConnectionBG.GetSystemEvent().Clear();
        networkConnection.SubmitRequest();
        networkConnectionBG.SubmitRequest();

        ASSERT_TRUE(networkConnection.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));
        ASSERT_TRUE(networkConnectionBG.GetSystemEvent().TimedWait(nn::TimeSpan::FromSeconds(60)));

        EXPECT_TRUE(networkConnection.IsAvailable());
        EXPECT_TRUE(networkConnectionBG.IsAvailable());
        NNT_EXPECT_RESULT_SUCCESS(networkConnection.GetResult());
        NNT_EXPECT_RESULT_SUCCESS(networkConnectionBG.GetResult());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::FinalizeAdminForTest());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
