﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Log.h>

#include <nn/nifm.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    bool g_ConnectionTimeoutShort = false;
    bool g_ApplyConfigTimeoutShort = false;
}

class FwdbgSettingsCaseTest : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
    }
    virtual void TearDown()
    {
    }

    static void TearDownTestCase()
    {
        nn::nifm::Initialize();
        NN_UTIL_SCOPE_EXIT
        {
            nn::nifm::FinalizeForTest();
        };

        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};


TEST_F(FwdbgSettingsCaseTest, ShortTimeout)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());

    // imatake-opennone-stealth
    const nn::nifm::WirelessSettingData wirelessSetting0 = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting0 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile0 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true, // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting0
        },
        ipSetting0
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
    ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    if (g_ApplyConfigTimeoutShort)
    {
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultApplyIpConfigTimeout, networkConnection.GetResult());
    }
    else if (g_ConnectionTimeoutShort)
    {
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpOperationTimedout, networkConnection.GetResult());
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());
    }

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 0; i < argc; ++i)
    {
        if (std::strcmp(argv[i], "--apply_config_timeout_short") == 0)
        {
            g_ApplyConfigTimeoutShort = true;
        }
        else if (std::strcmp(argv[i], "--connection_timeout_short") == 0)
        {
            g_ConnectionTimeoutShort = true;
        }
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
