﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/util/util_StringUtil.h>

#include <nn/nifm/detail/util/nifm_EventHandler.h>


// 渡された文字列の指定したインデックスの位置に、何番目に呼ばれたかを書き込むコールバック
class MyCallback : public nn::nifm::detail::CallbackObject
{
private:
    static int s_Counter;
    static char* s_pChar;

    int m_Index;

public:
    // pChar の指すバッファは、すべて '\0' で初期化しておくこと
    explicit MyCallback(int index) NN_NOEXCEPT
        : m_Index(index)
    {
    }

    virtual void ExecuteImpl() NN_NOEXCEPT NN_OVERRIDE
    {
        for( int i = 0; i < m_Index; ++i )
        {
            if(s_pChar[i] == '\0' )
            {
                s_pChar[i] = ' ';
            }
        }

        s_pChar[m_Index] = static_cast<char>('0' + s_Counter);
        ++s_Counter;
    }

    static void Reset(char* pChar, size_t count) NN_NOEXCEPT
    {
        s_Counter = 0;
        s_pChar = pChar;
        std::memset(pChar, '\0', count);
    }
};

int MyCallback::s_Counter = 0;
char* MyCallback::s_pChar = nullptr;

TEST(SingleEventHandlerCase, EventHandler)
{
    char resultString[64];

    MyCallback::Reset(resultString, sizeof(resultString));

    MyCallback myCallback0(0);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    singleEventHandler0.Register(&myCallback0);

    singleEventHandler0.LinkTo(&multiWait);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0", resultString);

    EXPECT_FALSE(event0.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase1, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 上位のコールバック2登録 → 下位のハンドラ登録 → 多重待ち登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 下位のハンドラ0解除 → 上位のコールバック3解除 → 上位のコールバック2解除 → 下位のハンドラ0再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("012", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("032", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("432", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase2, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 上位のコールバック2登録 → 多重待ち登録 → 下位のハンドラ登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 下位のハンドラ0解除 → 上位のコールバック3解除 → 下位のハンドラ0再登録 → 上位のコールバック2解除

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("012", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("314", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("354", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase3, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 下位のハンドラ登録 → 上位のコールバック2登録 → 多重待ち登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 下位のハンドラ0解除 → 下位のハンドラ0再登録 → 上位のコールバック3解除 → 上位のコールバック2解除

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("1 23", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("1453", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("1653", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase4, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 下位のハンドラ登録 → 多重待ち登録 → 上位のコールバック2登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 上位のコールバック3解除 → 下位のハンドラ0解除 → 上位のコールバック2解除 → 下位のハンドラ0再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ(" 01", resultString);

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("201", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("231", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("431", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase5, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 多重待ち登録 → 下位のハンドラ登録 → 上位のコールバック2登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 上位のコールバック3解除 → 下位のハンドラ0解除 → 下位のハンドラ0再登録 → 上位のコールバック2解除

    MyCallback::Reset(resultString, sizeof(resultString));

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ(" 01", resultString);

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("201", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("304", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("354", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiEventHandlerCase6, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 多重待ち登録 → 上位のコールバック2登録 → 下位のハンドラ登録 → 上位のコールバック3登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler2.LinkTo(&multiWait);

    multiEventHandler2.Register(&myCallback2);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler2.Add(singleEventHandler1);

    multiEventHandler2.Register(&myCallback3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler2.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 12", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0345", resultString);

    // 上位のコールバック3解除 → 上位のコールバック2解除 → 下位のハンドラ0解除 → 下位のハンドラ0再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback3));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ(" 01", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ(" 21", resultString);

    multiEventHandler2.Remove(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(singleEventHandler0.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("321", resultString);

    multiEventHandler2.Add(singleEventHandler0);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("421", resultString);

    multiEventHandler2.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

// 以下のパターンは、操作不能なので検証しない

// 下位のハンドラ0解除 → 上位のコールバック2解除 → 上位のコールバック3解除 → 下位のハンドラ0再登録
// 下位のハンドラ0解除 → 上位のコールバック2解除 → 下位のハンドラ0再登録 → 上位のコールバック3解除
// 下位のハンドラ0解除 → 下位のハンドラ0再登録 → 上位のコールバック2解除 → 上位のコールバック3解除
// 上位のコールバック2解除 → 下位のハンドラ0解除 → 上位のコールバック3解除 → 下位のハンドラ0再登録
// 上位のコールバック2解除 → 下位のハンドラ0解除 → 下位のハンドラ0再登録 → 上位のコールバック3解除
// 下位のハンドラ0再登録 → 下位のハンドラ0解除 → 上位のコールバック3解除 → 上位のコールバック2解除
// 下位のハンドラ0再登録 → 下位のハンドラ0解除 → 上位のコールバック2解除 → 上位のコールバック3解除
// 上位のコールバック3解除 → 下位のハンドラ0再登録 → 下位のハンドラ0解除 → 上位のコールバック2解除
// 上位のコールバック2解除 → 上位のコールバック3解除 → 下位のハンドラ0解除 → 下位のハンドラ0再登録
// 上位のコールバック2解除 → 下位のハンドラ0再登録 → 下位のハンドラ0解除 → 上位のコールバック3解除
// 下位のハンドラ0再登録 → 上位のコールバック3解除 → 下位のハンドラ0解除 → 上位のコールバック2解除
// 下位のハンドラ0再登録 → 上位のコールバック2解除 → 下位のハンドラ0解除 → 上位のコールバック3解除
// 上位のコールバック3解除 → 上位のコールバック2解除 → 下位のハンドラ0再登録 → 下位のハンドラ0解除
// 上位のコールバック3解除 → 下位のハンドラ0再登録 → 上位のコールバック2解除 → 下位のハンドラ0解除
// 上位のコールバック2解除 → 上位のコールバック3解除 → 下位のハンドラ0再登録 → 下位のハンドラ0解除
// 上位のコールバック2解除 → 下位のハンドラ0再登録 → 上位のコールバック3解除 → 下位のハンドラ0解除
// 下位のハンドラ0再登録 → 上位のコールバック3解除 → 上位のコールバック2解除 → 下位のハンドラ0解除
// 下位のハンドラ0再登録 → 上位のコールバック2解除 → 上位のコールバック3解除 → 下位のハンドラ0解除

TEST(MultiMultiEventHandlerCase1, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 上位のコールバック登録 → 下位のハンドラ登録 → 多重待ち登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Register(&myCallback4);

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    multiEventHandler4.LinkTo(&multiWait);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 下位のハンドラ2解除 → 上位のコールバック4解除 → 下位のハンドラ2再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0 1", resultString);

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0213", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("4253", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiMultiEventHandlerCase2, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 上位のコールバック登録 → 多重待ち登録 → 下位のハンドラ登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Register(&myCallback4);

    multiEventHandler4.LinkTo(&multiWait);

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 下位のハンドラ2解除 → 下位のハンドラ2再登録 → 上位のコールバック4解除

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0 1", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("2 3 4", resultString);

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("25364", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiMultiEventHandlerCase3, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 下位のハンドラ登録 → 上位のコールバック登録 → 多重待ち登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    multiEventHandler4.Register(&myCallback4);

    multiEventHandler4.LinkTo(&multiWait);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 上位のコールバック4解除 → 下位のハンドラ2解除 → 下位のコールバック2解除 → 下位のハンドラ2再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ(" 0 1", resultString);

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("2031", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("4031", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("5031", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiMultiEventHandlerCase4, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 下位のハンドラ登録 → 多重待ち登録 → 上位のコールバック登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    multiEventHandler4.LinkTo(&multiWait);

    multiEventHandler4.Register(&myCallback4);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 下位のハンドラ2解除 → 上位のコールバック4解除 → 下位のコールバック2解除 → 下位のハンドラ2再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0 1", resultString);

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("0213", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("4213", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("5213", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiMultiEventHandlerCase5, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 多重待ち登録 → 下位のハンドラ登録 → 上位のコールバック登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.LinkTo(&multiWait);

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    multiEventHandler4.Register(&myCallback4);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 下位のハンドラ2解除 → 下位のコールバック2解除 → 上位のコールバック4解除 → 下位のハンドラ2再登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0 1", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("2 1", resultString);

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("2314", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("5314", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

TEST(MultiMultiEventHandlerCase6, EventHandler)
{
    char resultString[64];

    MyCallback myCallback0(0);
    MyCallback myCallback1(1);
    MyCallback myCallback2(2);
    MyCallback myCallback3(3);
    MyCallback myCallback4(4);

    nn::os::Event event0(nn::os::EventClearMode_AutoClear);
    nn::os::Event event1(nn::os::EventClearMode_AutoClear);

    nn::nifm::detail::SingleEventHandler singleEventHandler0(event0);
    nn::nifm::detail::SingleEventHandler singleEventHandler1(event1);

    singleEventHandler0.Register(&myCallback0);
    singleEventHandler1.Register(&myCallback1);

    nn::nifm::detail::MultiEventHandler multiEventHandler2;
    nn::nifm::detail::MultiEventHandler multiEventHandler3;

    multiEventHandler2.Register(&myCallback2);
    multiEventHandler3.Register(&myCallback3);

    multiEventHandler2.Add(singleEventHandler0);
    multiEventHandler3.Add(singleEventHandler1);

    nn::nifm::detail::MultiEventHandler multiEventHandler4;

    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    // 多重待ち登録 → 上位のコールバック登録 → 下位のハンドラ登録

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.LinkTo(&multiWait);

    multiEventHandler4.Register(&myCallback4);

    multiEventHandler4.Add(multiEventHandler2);
    multiEventHandler4.Add(multiEventHandler3);

    nn::os::MultiWaitHolderType* pHolder;

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    multiEventHandler4.ExecuteCallbackIfSignaled(pHolder);
    EXPECT_STRCASEEQ("0 1 2", resultString);

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("03145", resultString);

    // 下位のハンドラ2解除 → 下位のコールバック2解除 → 下位のハンドラ2再登録 → 上位のコールバック4解除

    MyCallback::Reset(resultString, sizeof(resultString));

    multiEventHandler4.Remove(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("0 1", resultString);

    EXPECT_TRUE(multiEventHandler2.Unregister(&myCallback2));

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_FALSE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_TRUE(multiEventHandler2.ExecuteCallbackIfSignaled(singleEventHandler0.GetMultiWaitHolderPointer()));
    EXPECT_STRCASEEQ("2 1", resultString);

    multiEventHandler4.Add(multiEventHandler2);

    event0.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("3 1 4", resultString);

    EXPECT_TRUE(multiEventHandler4.Unregister(&myCallback4));

    event1.Signal();

    pHolder = nn::os::TryWaitAny(&multiWait);

    EXPECT_TRUE(multiEventHandler4.ExecuteCallbackIfSignaled(pHolder));
    EXPECT_STRCASEEQ("35164", resultString);

    multiEventHandler4.RemoveAll();

    EXPECT_FALSE(event0.TryWait());
    EXPECT_FALSE(event1.TryWait());
} // NOLINT(readability/fn_size)

// 以下のパターンは、操作不能なので検証しない

// 下位のハンドラ2解除 → 上位のコールバック4解除 → 下位のコールバック2解除 → 下位のハンドラ2再登録
// 下位のハンドラ2解除 → 下位のハンドラ2再登録 → 下位のコールバック2解除 → 上位のコールバック4解除
// 下位のハンドラ2解除 → 下位のハンドラ2再登録 → 上位のコールバック4解除 → 下位のコールバック2解除
// 上位のコールバック4解除 → 下位のハンドラ2解除 → 下位のハンドラ2再登録 → 下位のコールバック2解除
// 下位のコールバック2解除 → 下位のハンドラ2解除 → 下位のハンドラ2再登録 → 上位のコールバック4解除
// 下位のハンドラ2再登録 → 下位のハンドラ2解除 → 下位のコールバック2解除 → 上位のコールバック4解除
// 下位のコールバック2解除 → 下位のハンドラ2解除 → 上位のコールバック4解除 → 下位のハンドラ2再登録
// 下位のハンドラ2再登録 → 下位のハンドラ2解除 → 上位のコールバック4解除 → 下位のコールバック2解除
// 上位のコールバック4解除 → 下位のコールバック2解除 → 下位のハンドラ2解除 → 下位のハンドラ2再登録
// 上位のコールバック4解除 → 下位のハンドラ2再登録 → 下位のハンドラ2解除 → 下位のコールバック2解除
// 下位のコールバック2解除 → 下位のハンドラ2再登録 → 下位のハンドラ2解除 → 上位のコールバック4解除
// 下位のハンドラ2再登録 → 下位のコールバック2解除 → 下位のハンドラ2解除 → 上位のコールバック4解除
// 下位のコールバック2解除 → 上位のコールバック4解除 → 下位のハンドラ2解除 → 下位のハンドラ2再登録
// 下位のハンドラ2再登録 → 上位のコールバック4解除 → 下位のハンドラ2解除 → 下位のコールバック2解除
// 上位のコールバック4解除 → 下位のコールバック2解除 → 下位のハンドラ2再登録 → 下位のハンドラ2解除
// 上位のコールバック4解除 → 下位のハンドラ2再登録 → 下位のコールバック2解除 → 下位のハンドラ2解除
// 下位のコールバック2解除 → 下位のハンドラ2再登録 → 上位のコールバック4解除 → 下位のハンドラ2解除
// 下位のハンドラ2再登録 → 下位のコールバック2解除 → 上位のコールバック4解除 → 下位のハンドラ2解除
// 下位のコールバック2解除 → 上位のコールバック4解除 → 下位のハンドラ2再登録 → 下位のハンドラ2解除
// 下位のハンドラ2再登録 → 上位のコールバック4解除 → 下位のコールバック2解除 → 下位のハンドラ2解除
