﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/ae.h>

#include <nn/applet/applet.h>

#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiHandleNetworkRequestResult.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>

#include <nn/util/util_StringUtil.h>

#include <nn/la/la_CommonArgumentsWriter.h>

#include <nn/la/la_Api.h>

#include "../Common/nifm_TestUtility.h"

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    struct TestContext
    {
        char nextTestName[32];
    };

    static TestContext g_TestContext = { "AppletNetConnect1" };
    static nn::ae::LibraryAppletSelfInfo g_LibraryAppletSelfInfo;
}

class DummyWebTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
    }

    static void TearDownTestCase()
    {
    }
};

// ResultNetworkNotFound でネット接続アプレットが起動
TEST_F(DummyWebTest, AppletNetConnect1)
{
    char testName[] = "AppletNetConnect1";
    if(nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    if(!g_LibraryAppletSelfInfo.isUnwound)
    {
        // not-exist-ap
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x6e,0x6f,0x74,0x2d,0x65,0x78,0x69,0x73,0x74,0x2d,0x61,0x70 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    8,  // length
                    "password"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());

        nn::applet::LibraryAppletHandle libraryAppletHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PrepareHandlingNetworkRequestResult(networkConnection.GetRequestHandle(), &libraryAppletHandle));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::StartHandlingNetworkRequestResult(libraryAppletHandle, reinterpret_cast<void*>(testName)));
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, nn::nifm::FinishHandlingNetworkRequestResult(g_LibraryAppletSelfInfo.previousLibraryAppletHandle));

        nn::util::Strlcpy(g_TestContext.nextTestName, "AppletNetConnect2", sizeof(g_TestContext.nextTestName));
        g_LibraryAppletSelfInfo.isUnwound = false;
    }
}

// ResultDhcpFailedOrPassphraseWrong でネット接続アプレットが起動
TEST_F(DummyWebTest, AppletNetConnect2)
{
    char testName[] = "AppletNetConnect2";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        // imatake-openwep40
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    17,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    5,  // length
                    "Wrong"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        char uuidString[nn::util::Uuid::StringSize];
        NN_LOG("ID:%s\n", temporaryNetworkProfile.GetId().ToString(uuidString, nn::util::Uuid::StringSize));

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultDhcpFailedOrPassphraseWrong, networkConnection.GetResult());

        nn::applet::LibraryAppletHandle libraryAppletHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PrepareHandlingNetworkRequestResult(networkConnection.GetRequestHandle(), &libraryAppletHandle));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::StartHandlingNetworkRequestResult(libraryAppletHandle, reinterpret_cast<void*>(testName)));
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, nn::nifm::FinishHandlingNetworkRequestResult(g_LibraryAppletSelfInfo.previousLibraryAppletHandle));

        nn::util::Strlcpy(g_TestContext.nextTestName, "AppletError", sizeof(g_TestContext.nextTestName));
        g_LibraryAppletSelfInfo.isUnwound = false;
    }
}

TEST_F(DummyWebTest, AppletError)
{
    char testName[] = "AppletError";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        nn::nifm::NetworkConnection networkConnection1;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        nn::nifm::NetworkConnection networkConnection2;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 50));

        networkConnection2.SubmitRequest();

        // [TORIEAZU]
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultProcessing, networkConnection2.GetResult());

        nn::applet::LibraryAppletHandle libraryAppletHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PrepareHandlingNetworkRequestResult(networkConnection2.GetRequestHandle(), &libraryAppletHandle));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::StartHandlingNetworkRequestResult(libraryAppletHandle, reinterpret_cast<void*>(testName)));
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingFailure, nn::nifm::FinishHandlingNetworkRequestResult(g_LibraryAppletSelfInfo.previousLibraryAppletHandle));

        nn::util::Strlcpy(g_TestContext.nextTestName, "AppletWifiWebAuthWithoutWinding", sizeof(g_TestContext.nextTestName));
        g_LibraryAppletSelfInfo.isUnwound = false;
    }
}

// Winding なしの WifiWebAuth アプレット呼び出し
// Winding ありは DummyNetConnect2 経由で呼び出す
TEST_F(DummyWebTest, AppletWifiWebAuthWithoutWinding)
{
    char testName[] = "AppletWifiWebAuthWithoutWinding";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting
        },
        ipSetting
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
    ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 10));

    // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、4回までは再試行
    for (int i = 0; i < 4; ++i)
    {
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        if (networkConnection.GetResult().IsSuccess())
        {
            NN_LOG("WISPr auth is success: %d. Waiting to retry...", i);
            networkConnection.CancelRequest();
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
            continue;
        }

        break;
    }

    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection.GetResult());
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, networkConnection.HandleNetworkRequestResult());

    nn::util::Strlcpy(g_TestContext.nextTestName, "NoApplet", sizeof(g_TestContext.nextTestName));
    g_LibraryAppletSelfInfo.isUnwound = false;
}

TEST_F(DummyWebTest, NoApplet)
{
    char testName[] = "NoApplet";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    nn::nifm::NetworkConnection networkConnection;
    networkConnection.SubmitRequest();
    networkConnection.CancelRequest();

    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());
    // もし ResultCanceled でエラービューアが起動すれば DummyError の Assertion にかかる
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingFailure, networkConnection.HandleNetworkRequestResult());

    // 終了
}

void SaveContext() NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_TestContext)));
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, &g_TestContext, sizeof(g_TestContext)));
    nn::ae::PushToContextStack(storageHandle);
}

void RestoreContext(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    ASSERT_TRUE(nn::ae::TryPopFromContextStack(&storageHandle));
    ASSERT_EQ(sizeof(g_TestContext), nn::applet::GetStorageSize(storageHandle));
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, &g_TestContext, sizeof(g_TestContext)));
    nn::applet::ReleaseStorage(storageHandle);
}


nn::ae::LibraryAppletStartHookResult MyLibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void *pUserArgument) NN_NOEXCEPT
{
    SaveContext();

    NN_LOG("pUserArgument : %s\n", reinterpret_cast<char*>(pUserArgument));

    if (nullptr == pUserArgument)
    {
        return nn::ae::LibraryAppletStartHookResult_Normal;
    }
    else
    {
        return nn::ae::LibraryAppletStartHookResult_WindProgram;
    }
}

void LibraryAppletMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    if (info.isUnwound)
    {
        RestoreContext(info);
    }

    g_LibraryAppletSelfInfo = info;

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}

// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // コールバック登録
    nn::ae::SetLibraryAppletStartHook(&MyLibraryAppletStartHook);

    // LibraryApplet としてアプリを起動する
    nn::ae::InvokeLibraryAppletMain(LibraryAppletMain);
}
