﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>

#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_ApiHandleNetworkRequestResult.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/la/la_Api.h>
#include <nn/applet/applet_Storage.h>
#include <nn/util/util_StringUtil.h>

#include "../Common/nifm_TestUtility.h"

/**
 * DummyNetConnect2（本テスト）から，HandleNetworkRequestResultForOcean() を呼び出すことで
 * 1. DummyNetConnect が起動する場合（実際には起動しない）
 * 2. DummyError が起動する場合（Winding されない）
 * 3. DummyWifiWebAuth が起動する場合（Winding される）
 * の状況をテストする．
 *
 * Ocean では，HandleNetworkRequestResultForOcean() 的な実装を呼び出してエラーハンドリングを行い，
 * Winding からの復帰後は GetResultHandlingNetworkRequestResult() を呼び出して結果を取得すればよいはず．
 */

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    struct TestContext
    {
        char nextTestName[32];
    };

    static TestContext g_TestContext = { "AppletNetConnect" };

    static nn::ae::LibraryAppletSelfInfo g_LibraryAppletSelfInfo;

    nn::Result HandleNetworkRequestResultForOcean(const nn::nifm::RequestHandle& requestHandle)
    {
        nn::applet::LibraryAppletHandle libraryAppletHandle;
        nn::Result result = nn::nifm::PrepareHandlingNetworkRequestResult(requestHandle, &libraryAppletHandle);
        if (result.IsFailure())
        {
            return result;
        }

        nn::applet::AppletId appletId = nn::applet::GetLibraryAppletId(libraryAppletHandle);

        // Ocean では NetConnect から NetConnect は起動されないはず
        if (nn::applet::AppletId_LibraryAppletNetConnect == appletId)
        {
            return nn::nifm::ResultErrorHandlingFailure();
        }

        // Ocean LA は nn::la::LibraryAppletStartHookUserArg を見て Winding の有無などを判定する
        nn::la::LibraryAppletStartHookUserArg libraryAppletStartHookUserArg;
        libraryAppletStartHookUserArg.isExtremity = (appletId == nn::applet::AppletId_LibraryAppletError);

        // この関数内の nn::applet::StartLibraryApplet 実行時に，呼び出し元が Winding される可能性がある
        result = nn::nifm::StartHandlingNetworkRequestResult(libraryAppletHandle, reinterpret_cast<void*>(&libraryAppletStartHookUserArg));
        if (result.IsFailure())
        {
            return result;
        }

        // Winding されない場合、すべての終了処理を Ocean Framework 以外の場所で受け持つ
        return nn::nifm::FinishHandlingNetworkRequestResult(libraryAppletHandle);
    }
}

class DummyNetConnect2Test : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
    }

    static void TearDownTestCase()
    {
    }
};

TEST_F(DummyNetConnect2Test, AppletNetConnect)
{
    char testName[] = "AppletNetConnect";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        // not-exist-ap
        const nn::nifm::WirelessSettingData wirelessSetting1 = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x6e,0x6f,0x74,0x2d,0x65,0x78,0x69,0x73,0x74,0x2d,0x61,0x70 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    8,  // length
                    "password"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting1 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile1 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting1
            },
            ipSetting1
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, networkConnection.GetResult());

        // エラーハンドリング
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingFailure, HandleNetworkRequestResultForOcean(networkConnection.GetRequestHandle()));
    }
    else
    {
        // NetConnect から NetConnect は起動しないので，ここにくることもない
        ASSERT_TRUE(false);
    }

    nn::util::Strlcpy(g_TestContext.nextTestName, "AppletError", sizeof(g_TestContext.nextTestName));
    g_LibraryAppletSelfInfo.isUnwound = false;
}

TEST_F(DummyNetConnect2Test, AppletError)
{
    char testName[] = "AppletError";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        nn::nifm::NetworkConnection networkConnection1;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection1.GetRequestHandle(), nn::nifm::RequirementPreset_LocalForApplet));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

        nn::nifm::NetworkConnection networkConnection2;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 50));

        networkConnection2.SubmitRequest();

        // [TORIEAZU] 将来的には ResultProcessing では ErrorViewer を起動しないかもしれない
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultProcessing, networkConnection2.GetResult());

        // エラーハンドリング
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingFailure, HandleNetworkRequestResultForOcean(networkConnection2.GetRequestHandle()));
    }
    else
    {
        // ErorViewer の起動では Winding されないので，ここにくることもない
        ASSERT_TRUE(false);
    }

    nn::util::Strlcpy(g_TestContext.nextTestName, "AppletWifiWebAuth", sizeof(g_TestContext.nextTestName));
    g_LibraryAppletSelfInfo.isUnwound = false;
}

TEST_F(DummyNetConnect2Test, AppletWifiWebAuth)
{
    char testName[] = "AppletWifiWebAuth";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            false,  // isAuto
            { { 52, 68, 203, 240 } },  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting
        },
        ipSetting
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
    ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 10));

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、4回までは再試行
        // CI では 直前の Connect テストの WISPr 認証成功から 60 秒経過しておらず 1回は再試行するはず
        for (int i = 0; i < 4; ++i)
        {
            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            if (networkConnection.GetResult().IsSuccess())
            {
                NN_LOG("WISPr auth is success: %d. Waiting to retry...", i);
                networkConnection.CancelRequest();
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
                continue;
            }

            break;
        }

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));

        // エラーハンドリング
        // 実際には Winding されるので，この ASSERTION が働くことはない
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, HandleNetworkRequestResultForOcean(networkConnection.GetRequestHandle()));
    }
    else
    {
        nn::applet::LibraryAppletHandle libraryAppletHandle = g_LibraryAppletSelfInfo.previousLibraryAppletHandle;

        nn::applet::JoinLibraryApplet(libraryAppletHandle);
        nn::applet::LibraryAppletExitReason exitReason = nn::applet::GetLibraryAppletExitReason(libraryAppletHandle);

        ASSERT_TRUE(nn::applet::LibraryAppletExitReason_Normal == exitReason);

        nn::applet::StorageHandle storageHandle;

        // 結果の取得
        ASSERT_TRUE(nn::applet::TryPopFromOutChannel(&storageHandle, libraryAppletHandle));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, nn::nifm::GetResultHandlingNetworkRequestResult(storageHandle));

        // DummyWifiWebAuth を本当のアプレットに差し替えれば確認可能
        // 再度疎通確認
        //EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        //NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        // 実際には Winding から復帰した後の終了処理は Ocean Framework が処理するはず
        nn::applet::ReleaseStorage(storageHandle);
        nn::applet::CloseLibraryApplet(libraryAppletHandle);

        nn::util::Strlcpy(g_TestContext.nextTestName, "NoApplet", sizeof(g_TestContext.nextTestName));
        g_LibraryAppletSelfInfo.isUnwound = false;
    }
}

TEST_F(DummyNetConnect2Test, NoApplet)
{
    char testName[] = "NoApplet";
    if (nn::util::Strncmp(testName, g_TestContext.nextTestName, sizeof(testName)) != 0)
    {
        return;
    }

    if (!g_LibraryAppletSelfInfo.isUnwound)
    {
        nn::nifm::NetworkConnection networkConnection;
        networkConnection.SubmitRequest();
        networkConnection.CancelRequest();

        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultCanceled, networkConnection.GetResult());

        // エラーハンドリング
        // もし ResultCanceled でエラービューアが起動すれば DummyError の Assertion にかかる
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingFailure, HandleNetworkRequestResultForOcean(networkConnection.GetRequestHandle()));
    }
    else
    {
        // なにもアプレットは起動されず Winding もされないので，ここにくることもない
        ASSERT_TRUE(false);
    }

    // 終了
}

void SaveContext() NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_TestContext)));
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, &g_TestContext, sizeof(g_TestContext)));
    nn::ae::PushToContextStack(storageHandle);
}

void RestoreContext(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    ASSERT_TRUE(nn::ae::TryPopFromContextStack(&storageHandle));
    ASSERT_EQ(sizeof(g_TestContext), nn::applet::GetStorageSize(storageHandle));
    NNT_ASSERT_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, &g_TestContext, sizeof(g_TestContext)));
    nn::applet::ReleaseStorage(storageHandle);
}

nn::ae::LibraryAppletStartHookResult MyLibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void *pUserArgument) NN_NOEXCEPT
{
    SaveContext();

    nn::la::LibraryAppletStartHookUserArg* pLibraryAppletStartHookUserArg = reinterpret_cast<nn::la::LibraryAppletStartHookUserArg*>(pUserArgument);

    return pLibraryAppletStartHookUserArg->isExtremity ? nn::ae::LibraryAppletStartHookResult_Normal : nn::ae::LibraryAppletStartHookResult_WindProgram;
}

void LibraryAppletMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    if (info.isUnwound)
    {
        RestoreContext(info);
    }

    g_LibraryAppletSelfInfo = info;

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}

// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // コールバック登録
    nn::ae::SetLibraryAppletStartHook(&MyLibraryAppletStartHook);

    // LibraryApplet としてアプリを起動する
    nn::ae::InvokeLibraryAppletMain(LibraryAppletMain);
}
