﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nnc/result/result_CConverter.h>

#include <nn/os.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nnc/nifm/nifm_ApiProxy.h>

#include "../Common/nifm_TestUtility.h"


extern "C" nnResult call_nnnifmGetCurrentProxySetting(nnnifmProxySetting* pOutProxySetting);

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    void SetExclusive(bool isExclusive)
    {
        if (isExclusive)
        {
            nn::nifm::InitializeAdmin();

            nn::nifm::ClientId clientId = nn::nifm::GetClientId();
            nn::nifm::SetExclusiveClient(clientId);

            nn::nifm::InternetConnectionStatus internetConnectionStatus;

            // プロセス独占後，全インターネット接続が切断されるまで最長で30秒待つ
            nn::Result connectionResult = nn::ResultSuccess();
            for (int i = 0; i < 30; ++i)
            {
                connectionResult = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);
                if (connectionResult.IsFailure())
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
        }
        else
        {
            nn::nifm::InitializeAdmin();

            nn::nifm::ClientId clientId = { nn::nifm::InvalidClientIdValue };
            nn::nifm::SetExclusiveClient(clientId);
        }
    }
}

class CLinkageCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        SetExclusive(true);
    }

    static void TearDownTestCase()
    {
        // 本テストプロセスで独占解除
        SetExclusive(false);
    }
};


TEST_F(CLinkageCaseTest, GetCurrentProxySetting)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nnnifmProxySetting proxySetting;
    nnResult result = call_nnnifmGetCurrentProxySetting( &proxySetting );
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::result::ConvertFromC(result));

    // imatake-opennone-stealth
    const nn::nifm::WirelessSettingData wirelessSetting0 = {
        {  // ssidConfig
            {  // ssid
                24,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x6e,0x6f,0x6e,0x65,0x2d,0x73,0x74,0x65,0x61,0x6c,0x74,0x68 }  // hex
            },
        true  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Open,  // authentication
                nn::nifm::Encryption_None  // encryption
            },
            {  // sharedKey
                7,  // length
                "NotUsed"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting0 = { {true, {}, {}, {}}, {true, {}, {}}, {true, 8080, "dummy.proxy.jp", {true, "dummy_user", "dummy_password"}}, 1400 };
    const nn::nifm::NetworkProfileData networkProfile0 = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, true, true, { wirelessSetting0 }, ipSetting0 };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
    ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

    // テンポラリ接続要求0
    nn::nifm::NetworkConnection networkConnection0;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection0.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection0, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

    result = call_nnnifmGetCurrentProxySetting( &proxySetting );
    NNT_ASSERT_RESULT_SUCCESS(nn::result::ConvertFromC(result));

    EXPECT_TRUE( ipSetting0.proxy.isEnabled == proxySetting.isEnabled );
    EXPECT_TRUE( ipSetting0.proxy.port == proxySetting.port );
    EXPECT_TRUE( strncmp( ipSetting0.proxy.proxy, proxySetting.proxy, sizeof(proxySetting.proxy) ) == 0 );
    EXPECT_TRUE( ipSetting0.proxy.authentication.isEnabled == proxySetting.authentication.isEnabled );
    EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.username, proxySetting.authentication.username, sizeof(proxySetting.authentication.username) ) == 0 );
    EXPECT_TRUE( strncmp( ipSetting0.proxy.authentication.password, proxySetting.authentication.password, sizeof(proxySetting.authentication.password) ) == 0 );
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
