﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNpt_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    const uint32_t   AgingRepeat = 1000000;
    const uint32_t   RandomSleepTimeMax = 300;
    const char* NptDeviceState[] =
    {
        "Init",          //!< タグを探していない状態です。
        "Search",        //!< タグを探している状態です。
        "Active",        //!< DeviceState_Search でタグを検知するとこの状態になります。
        "Deactive",      //!< タグを検知した後、タグが離れるとこの状態になります。
        "Unexpected"     //!< 想定していない状態です。
    };
    const char* NptLibraryState[] =
    {
        "None",          //!< NPT ライブラリが初期化されていない状態です。起動時はこの状態です。
        "Init",          //!< NPT ライブラリが初期化されている状態です。
    };

    // 送信データ(READ コマンド)
    nn::Bit8 g_sendReadCommandData[2];

    // READ コマンドに対するレスポンスデータ
    size_t g_sendReadCommandResponseDataSize;
    nn::Bit8 g_sendReadCommandResponseData[16];

    // 送信データ(WRITE コマンド)
    nn::Bit8 g_sendWriteCommandData[6];

    // WRITE コマンドに対するレスポンスデータ
    size_t g_sendWriteCommandResponseDataSize;
    nn::Bit8 g_sendWriteCommandResponseData[1];
} // end of anonymous namespace

//------------------------------------------------------------------
// Macro Defnition
//------------------------------------------------------------------

#define NPT_LOG( ... ) \
    do { \
        NN_LOG("[NPT_AGING] LIB_STATE = %s DEV_STATE = %s LINE = %04d ",\
               NptLibraryState[nnt::npt::wrapper::GetState()],          \
               NptDeviceState[nnt::npt::wrapper::GetDeviceState()],     \
               __LINE__);                                               \
        NN_LOG(__VA_ARGS__);                                            \
    } while(NN_STATIC_CONDITION(false))

#define NNT_NPT_SWITCH_NFC_ENABLE // NfcのON／OFF切り替え機能スイッチ

//================================================================================
// テストで使用する共通関数です。
//================================================================================
namespace {
    int RandomSleep(int max)
    {
        nn::Bit16 randNum;
        nnt::npt::CreateRandomNumberSequence(&randNum, 1);
        int ms = randNum % max;
        nnt::npt::Sleep(ms);
        return ms;
    }

    bool IsAllowResultSendCommandByPassThrough(nn::Result targetResult)
    {
        if(targetResult <= nn::nfc::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultAccessError())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultKeepPassThroughSession(nn::Result targetResult)
    {
        if(targetResult <= nn::nfc::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultReleasePassThroughSession(nn::Result targetResult)
    {
        if(targetResult <= nn::nfc::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultStopDetect(nn::Result targetResult)
    {
        if(targetResult <= nn::nfc::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfc::ResultNfcDisabled())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool CheckForActivateEvent(bool isActivate)
    {
        EXPECT_TRUE(isActivate);
        return isActivate;
    }

    bool ReadWriteTag() NN_NOEXCEPT
    {
        nn::Result result;

        nnt::npt::PrepareSendWriteCommandData(g_sendWriteCommandData, sizeof(g_sendWriteCommandData));
        nnt::npt::PrintSendWriteCommandData(g_sendWriteCommandData, sizeof(g_sendWriteCommandData));

        nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(nnt::npt::SendCommandTimeout);
        if(false == IsAllowResultSendCommandByPassThrough(
               nnt::npt::wrapper::SendCommandByPassThrough(g_sendWriteCommandResponseData, &g_sendWriteCommandResponseDataSize,
                                              g_sendWriteCommandData, sizeof(g_sendWriteCommandData),
                                              sizeof(g_sendWriteCommandResponseData), timeout)))
        {
            return false;
        }
        EXPECT_TRUE(g_sendWriteCommandResponseDataSize == 1 && g_sendWriteCommandResponseData[0] == 0xa);

        nnt::npt::PrepareSendReadCommandData(g_sendReadCommandData, sizeof(g_sendReadCommandData));
        if(false == IsAllowResultSendCommandByPassThrough(
               nnt::npt::wrapper::SendCommandByPassThrough(g_sendReadCommandResponseData, &g_sendReadCommandResponseDataSize,
                                              g_sendReadCommandData, sizeof(g_sendReadCommandData),
                                              sizeof(g_sendReadCommandResponseData), timeout)))
        {
            return false;
        }
        nnt::npt::PrintSendReadCommandResponseData(g_sendReadCommandResponseData, g_sendReadCommandResponseDataSize);
        // 書き込んだデータを読み込めたか確認
        nnt::npt::CheckReadData(g_sendReadCommandResponseData, g_sendReadCommandResponseDataSize,
                                g_sendWriteCommandData, sizeof(g_sendWriteCommandData));

        return true;
    }

#if defined(NNT_NPT_SWITCH_NFC_ENABLE)
    // スレッドスタックサイズ
    const size_t ThreadStackSize = 4 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_Thread;
    // スレッド継続停止のスイッチ
    bool g_ThreadSwitch = true;
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    // ランダムでSetNfcEnabled()を切り替えるスレッド
    void SwitchNfcEnabledThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        NPT_LOG("%s start \n", NN_CURRENT_FUNCTION_NAME);
        // Nfc On 時間の最大値と最小値(ms)
        const int NfcOnTimeMax = 30000;
        const int NfcOnTimeMin = 10000;
        // Nfc Off 時間の最大値と最小値(ms)
        const int NfcOffTimeMax = 5000;
        const int NfcOffTimeMin = 1000;
        // 試験が終了するまで繰り返す
        g_ThreadSwitch = true;
        while(g_ThreadSwitch)
        {
            std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
            // スリープ終了でNfc有効無効を切り替える
            if(nnt::npt::IsNfcEnable())
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOnTimeMax - NfcOnTimeMin)) + NfcOnTimeMin;
                NPT_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::npt::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::npt::NfcOff();
                NPT_LOG("%s <<Nfc Off>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::npt::IsNfcEnable());
            }
            else
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOffTimeMax - NfcOffTimeMin)) + NfcOffTimeMin;
                NPT_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::npt::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::npt::NfcOn();
                NPT_LOG("%s <<Nfc On>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::npt::IsNfcEnable());
            }
        }
        NPT_LOG("%s end \n", NN_CURRENT_FUNCTION_NAME);
    }
#endif // defined(NNT_NPT_SWITCH_NFC_ENABLE)
}

//================================================================================
// Nptライブラリの状態に応じて書込みとフォーマットを繰り返すエイジングです。
//================================================================================

class NptAgingRemoveTag : public nnt::npt::TestFramework
{
protected:

    NptAgingRemoveTag() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::npt::InitializeHidController();
        nnt::npt::wrapper::SetApiCallLoggingMode(nnt::npt::wrapper::LogMode_Aging);
    }
};

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(NptAgingRemoveTag,TestCaseAgingRemoveTag)
{

    bool doNextStep = true;

    for(int agingCount = 0; agingCount < AgingRepeat; agingCount++)
    {
        if(true == doNextStep)
        {
            NPT_LOG("RepeatCount : %0d \n", agingCount);
        }
        doNextStep = true;

        int ms = RandomSleep(RandomSleepTimeMax);

        nn::Bit16 command;
        nnt::npt::CreateRandomNumberSequence(&command, 1);

        nn::nfc::DeviceState deviceState = nnt::npt::wrapper::GetDeviceState();
        nn::nfc::State libState = nnt::npt::wrapper::GetState();
        NPT_LOG("SleepTime : %04d \n", ms);
        NPT_LOG("DeviceState  : %02d \n", deviceState);
        NPT_LOG("LibraryState : %02d \n", libState);

        if(libState != nn::nfc::State_Init)
        {
            NNT_NPT_ASSERT_EQUAL(nn::nfc::DeviceState_Unexpected,deviceState);
            NPT_LOG("nnt::npt::Initialize\n");
            NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::InitializeSystemWithRetry());
#if defined(NNT_NPT_SWITCH_NFC_ENABLE)
            // NfcのOn／Offを切り替えるスレッドを起動する
            NPT_LOG("\nCreate SwitchNfcEnabledThread\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                    &g_Thread, SwitchNfcEnabledThread, nullptr, g_ThreadStack,
                    sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
            nn::os::StartThread(&g_Thread);
#endif // defined(NNT_NPT_SWITCH_NFC_ENABLE)
            continue;
        }

        switch(deviceState)
        {
        case nn::nfc::DeviceState_Init:
            {
                NPT_LOG("nnt::npt::DoSearch\n");
                nnt::npt::DoSearch();
            }
            break;
        case nn::nfc::DeviceState_Search:
            {
                NPT_LOG("======= Waiting Tag Attached =======\n");

                switch (command % 2)
                {
                case 0:
                    {
                        doNextStep = CheckForActivateEvent(nnt::npt::WaitForActivate());
                        if(false == doNextStep)
                        {
                            break;
                        }
                        NPT_LOG("Tag is detected\n");
                    }
                    break;
                case 1:
                    {
                        NPT_LOG("nn::nfc::pt::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::npt::wrapper::StopDetection());
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case nn::nfc::DeviceState_Keep:
            {
                switch(command % 4)
                {
                case 0:
                    {
#if defined(NNT_NPT_SWITCH_NFC_ENABLE)
                        g_ThreadSwitch = false;
                        nn::os::WaitThread(&g_Thread);
                        nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NPT_SWITCH_NFC_ENABLE)
                        NPT_LOG("nnt::npt::FinalizeSystem\n");
                        NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::FinalizeSystem());
                    }
                    break;
                case 1:
                    {
                        NPT_LOG("nn::nfc::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::npt::wrapper::StopDetection());
                    }
                    break;
                case 2:
                    {
                        doNextStep = ReadWriteTag();
                    }
                    break;
                case 3:
                    {
                        NPT_LOG("nn::nfc::ReleasePassThroughSession\n");
                        doNextStep = IsAllowResultReleasePassThroughSession(nnt::npt::wrapper::ReleasePassThroughSession());
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case nn::nfc::DeviceState_Active:
            {
                switch(command % 3)
                {
                case 0:
                    {
                        NPT_LOG("nn::nfc::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::npt::wrapper::StopDetection());
                    }
                    break;
                case 1:
                    {
                        doNextStep = ReadWriteTag();
                    }
                    break;
                case 2:
                    {
                        NPT_LOG("nn::nfc::KeepPassThroughSession\n");
                        doNextStep = IsAllowResultKeepPassThroughSession(nnt::npt::wrapper::KeepPassThroughSession());
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case nn::nfc::DeviceState_Deactive:
            {
                switch(command % 3)
                {
                case 0:
                    {
#if defined(NNT_NPT_SWITCH_NFC_ENABLE)
                        g_ThreadSwitch = false;
                        nn::os::WaitThread(&g_Thread);
                        nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NPT_SWITCH_NFC_ENABLE)
                        NPT_LOG("nnt::npt::FinalizeSystem\n");
                        NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::FinalizeSystem());
                    }
                    break;
                case 1:
                    {
                        NPT_LOG("nn::nfc::pt::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::npt::wrapper::StopDetection());
                    }
                    break;
                case 2:
                    {
                        NPT_LOG("nnt::npt::DoSearch\n");
                        nnt::npt::DoSearch();
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case nn::nfc::DeviceState_Unexpected:
            {
                NPT_LOG("nnt::npt::DoSearch\n");
                nnt::npt::DoSearch();
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
            break;
        }
        if(false == doNextStep)
        {
            //処理失敗時のリトライなのでループカウントがインクリメントされないようにする
            agingCount--;
        }
    }
#if defined(NNT_NPT_SWITCH_NFC_ENABLE)
    //NfcのOn／Offを切り替えるスレッドを停止する
    g_ThreadSwitch = false;
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NPT_SWITCH_NFC_ENABLE)

    NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::wrapper::StopDetection());
    NNT_NPT_ASSERT_RESULT_SUCCESS_FATAL(nnt::npt::FinalizeSystem());
} // NOLINT(impl/function_size)
