﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nfp.h>
#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// 定数定義です。
//================================================================================

//! アプリケーション専用領域にアクセスするために必要な ID です。
const uint32_t AccessId = nnt::nfp::NfpManagerUsedTagId;

//================================================================================
//! @brief      検出されたタグの状態を表す定数です。
//================================================================================
enum TagState
{
    TagState_None,                     //!< タグは見つかっていません。
    TagState_Good,                     //!< 正常なタグです。
    TagState_NotSupported,             //!< NFP 以外のタグが検出されました。
    TagState_InvalidFormatVersion,     //!< タグのバージョンが不正です。
    TagState_NeedRestore,              //!< バックアップからの復旧が必要なタグです。
    TagState_NeedFormat                //!< フォーマットが必要なタグです。
};

//================================================================================
//! @brief      検出したタグの情報です。
//================================================================================
struct TagData
{
    nn::Bit32                 state;
    nn::nfp::TagInfo      tagInfo;
    nn::nfp::CommonInfo   commonInfo;
    nn::nfp::RegisterInfoPrivate registerInfo;
    nn::nfp::AdminInfo    adminInfo;
    nn::Bit8              applicationArea[nn::nfp::ApplicationAreaSizeV2];
    uint16_t              systemWriteCounter;
    bool                  hasTagInfo;
    bool                  hasCommonInfo;
    bool                  hasRegisterInfo;
    bool                  hasAdminInfo;
    bool                  hasApplicationArea;
    NN_PADDING1;
};

//================================================================================
//! @brief      タグ検出スレッドを開始します。
//================================================================================
void StartTagDetectionThread() NN_NOEXCEPT;

//================================================================================
//! @brief      タグ検出スレッドを終了します。
//================================================================================
void EndTagDetectionThread() NN_NOEXCEPT;

//================================================================================
//! @brief      NFP ライブラリを初期化します。
//================================================================================
nn::Result NfpInitialize() NN_NOEXCEPT;

//================================================================================
//! @brief      NFP ライブラリを解放します。
//================================================================================
nn::Result NfpFinalize() NN_NOEXCEPT;

//================================================================================
//! @brief      NFCデバイスハンドル取得します。
//================================================================================
nn::Result NfpListDevices() NN_NOEXCEPT;

//================================================================================
//! @brief      タグをマウントします。
//================================================================================
nn::Result NfpMountDirect() NN_NOEXCEPT;

//================================================================================
//! @brief      タグのマウントを解除します。
//================================================================================
nn::Result NfpUnmount() NN_NOEXCEPT;

#if defined(NNT_NFP_LIB_VERSION_BETA)
//================================================================================
//! @brief      WiFi状態を切り替えます
//================================================================================
nn::Result NfpWifiSwitch() NN_NOEXCEPT;
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

//================================================================================
//! @brief      タグのバックアップデータを削除します
//================================================================================
nn::Result NfpDeleteSystemSaveData() NN_NOEXCEPT;

//================================================================================
//! @brief      バックアップデータの作成日を出力します。
//================================================================================
nn::Result CheckBackup( void ) NN_NOEXCEPT;

//================================================================================
//! @brief      タグの検知を開始します。
//================================================================================
nn::Result NfpStartDetection() NN_NOEXCEPT;

//================================================================================
//! @brief      タグの検知を終了します。
//================================================================================
nn::Result NfpStopDetection() NN_NOEXCEPT;

#if defined(NNT_NFP_LIB_VERSION_ALPHA)
//================================================================================
//! @brief      タグを復元します。
//================================================================================
nn::Result NfpRestore() NN_NOEXCEPT;
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)

//================================================================================
//! @brief      タグをフォーマットします。
//================================================================================
nn::Result NfpFormat() NN_NOEXCEPT;

//================================================================================
//! @brief      アプリケーション領域を作成します。
//================================================================================
nn::Result NfpCreateApplicationArea() NN_NOEXCEPT;

//================================================================================
//! @brief      アプリケーション領域を更新します。
//================================================================================
nn::Result NfpSetApplicationArea() NN_NOEXCEPT;

#if defined(NNT_NFP_PLATFORM_NX)
//================================================================================
//! @brief      アプリケーション領域を再作成します。
//================================================================================
nn::Result NfpRecreateApplicationArea() NN_NOEXCEPT;
#endif

//================================================================================
//! @brief      書き込んだ結果をタグに反映します。
//================================================================================
nn::Result NfpFlush() NN_NOEXCEPT;

//================================================================================
//! @brief      アプリケーション領域を削除します。
//================================================================================
nn::Result NfpDeleteApplicationArea() NN_NOEXCEPT;

//================================================================================
//! @brief      初期登録情報を登録します。
//================================================================================
nn::Result NfpSetRegisterInfo() NN_NOEXCEPT;

//================================================================================
//! @brief      初期登録情報を登録します。
//================================================================================
nn::Result NfpSetAnotherRegisterInfo() NN_NOEXCEPT;

//================================================================================
//! @brief      初期登録情報を削除します。
//================================================================================
nn::Result NfpDeleteRegisterInfo() NN_NOEXCEPT;

//================================================================================
//! @brief      マウントされているタグの情報を取得します。
//! @param[out] タグ情報の出力先です。
//================================================================================
void NfpGetTagData(TagData* pOutTagData) NN_NOEXCEPT;

#if defined(NNT_NFP_LIB_VERSION_BETA)
//================================================================================
//! @brief      Amiibo 設定ニックネームとオーナーの設定画面を表示します。
//================================================================================
nn::Result NfpStartNicknameAndOwnerSettings() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定ゲームデータの消去設定画面を表示します。
//================================================================================
nn::Result NfpStartGameDataEraser() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定データの復旧設定画面を表示します。
//================================================================================
nn::Result NfpStartRestorer() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定ニックネームとオーナーの設定画面を表示します。(PrivateApi)
//================================================================================
nn::Result NfpStartNicknameAndOwnerSettingsPrivate() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定ゲームデータの消去設定画面を表示します。(PrivateApi)
//================================================================================
nn::Result NfpStartGameDataEraserPrivate() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定データの復旧設定画面を表示します。(PrivateApi)
//================================================================================
nn::Result NfpStartRestorerPrivate() NN_NOEXCEPT;

//================================================================================
//! @brief      Amiibo 設定フォーマット画面を表示します。(PrivateApi)
//================================================================================
nn::Result NfpStartFormatterPrivate() NN_NOEXCEPT;
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

//================================================================================
//! @brief      マウントされているタグを破壊します。
//================================================================================
nn::Result NfpBreakTag() NN_NOEXCEPT;
